package com.instabug.library.sessionreplay

import com.instabug.library.SessionSyncListener
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.sessionreplay.model.SessionMetadata
import com.instabug.library.util.extenstions.logError
import com.instabug.library.util.extenstions.logVerbose
import java.util.concurrent.ExecutorService
import java.util.concurrent.TimeUnit
import java.util.concurrent.TimeoutException

private const val CALLBACK_DISABLED_ERROR_MESSAGE = "The callback evaluation isn't enabled" +
        " as it seems to be disabled for your Instabug company account." +
        " Please, contact support to switch it on for you."

class SessionSyncListenerWrapper(
    private val executor: ExecutorService,
    private val configurations: SRConfigurationsProvider,
    private val timeoutInSeconds: Long = 10,
    private val userSyncListenerGetter: () -> SessionSyncListener?
) :
    SessionSyncListener {
    override fun onSessionReadyToSync(metadata: SessionMetadata): Boolean = userSyncListenerGetter()
        ?.takeIf { configurations.isCallbackEnabled }
        .also { logIfFeatureIsDisabled() }
        ?.runCatching {
            val future = executor
                .submit<Boolean> {
                    onSessionReadyToSync(metadata)
                }
            future[timeoutInSeconds, TimeUnit.SECONDS]
        }?.onFailure(::handleException)
        ?.getOrDefault(true) ?: true

    private fun logIfFeatureIsDisabled() {
        if (!configurations.isCallbackEnabled)
            (CALLBACK_DISABLED_ERROR_MESSAGE)
                .logVerbose(SR_LOG_TAG)

    }

    private fun handleException(exception: Throwable) {
        if (exception is TimeoutException) {
            "the session Replay evaluation callback Timeout".logVerbose(SR_LOG_TAG)
        } else {
            "The session replay evaluation callback crashed during the evaluation. Returning default state true."
                .logError(exception, tag = SR_LOG_TAG)
        }
    }
}
