package com.instabug.library.sessionreplay

import com.instabug.library.InstabugNetworkJob
import com.instabug.library.core.InstabugCore
import com.instabug.library.core.eventbus.V3SessionSyncResultEventBus
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent
import com.instabug.library.core.eventbus.coreeventbus.IBGSdkCoreEvent.V3Session
import com.instabug.library.core.eventbus.eventpublisher.Subscriber
import com.instabug.library.internal.filestore.FileOperation
import com.instabug.library.model.v3Session.IBGSessionMapper.asForegroundStartEvent
import com.instabug.library.sessionV3.manager.IBGSessionManager
import com.instabug.library.sessionV3.model.V3SessionSyncResult
import com.instabug.library.sessionreplay.configurations.SRConfigurationsHandler
import com.instabug.library.sessionreplay.configurations.SRConfigurationsProvider
import com.instabug.library.sessionreplay.monitoring.ErrorFinalizingSROldSessions
import com.instabug.library.sessionreplay.monitoring.SRConfigurations
import com.instabug.library.sessionreplay.monitoring.SRMonitoringDelegate
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.extenstions.defensiveExecute
import com.instabug.library.util.extenstions.defensiveSubmit
import com.instabug.library.util.extenstions.logVerbose
import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.threading.OrderedExecutorService
import com.instabug.library.visualusersteps.ReproCapturingProxy
import io.reactivexport.disposables.Disposable
import java.util.concurrent.Future


interface SRDelegateDependencies {
    val srDir: SRFilesDirectory
    val syncJob: InstabugNetworkJob
    val loggingController: SRLoggingController
    val executor: OrderedExecutorService
    val metadataDBHandler: SRMetadataDBHandler
    val compressOperation: FileOperation<List<SRSessionDirectory>, List<String>>
    val configurationsProvider: SRConfigurationsHandler
    val reproProxy: ReproCapturingProxy
    val monitorDelegate: SRMonitoringDelegate
    val stateListener: SRStateChangeListener
    val v3SessionSyncResultEventBus: V3SessionSyncResultEventBus
    val srUserEvaluator: SessionReplayUserEvaluator
}

class SRDelegate(
    dependencies: SRDelegateDependencies
) : Subscriber<IBGSdkCoreEvent> {
    private val srFilesDirectory: SRFilesDirectory = dependencies.srDir
    private val syncJob: InstabugNetworkJob = dependencies.syncJob
    private val loggingController: SRLoggingController = dependencies.loggingController
    private val executor: OrderedExecutorService = dependencies.executor
    private val cacheHandler: SRMetadataDBHandler = dependencies.metadataDBHandler
    private val compressSRDirsOperation = dependencies.compressOperation
    private val configurationHandler = dependencies.configurationsProvider
    private val reproProxy = dependencies.reproProxy
    private var monitorDelegate = dependencies.monitorDelegate
    private val stateListener = dependencies.stateListener
    private var pendingLog: Future<Unit>? = null
    private var runningSessionCompositeId: String? = null

    private var isSREnabledFromSessionStart: Boolean = false
    private var v3SessionSyncResultEventBus: V3SessionSyncResultEventBus =
        dependencies.v3SessionSyncResultEventBus
    private var v3SessionSyncDisposable: Disposable? = null
    private val sessionReplayUserEvaluator = dependencies.srUserEvaluator

    init {
        if (configurationHandler.srEnabled && InstabugCore.isV3SessionEnabled())
            subscribeV3SyncEventBus()
    }

    private fun subscribeV3SyncEventBus() {
        executor.defensiveExecute(
            key = SRExecutionQueues.Main,
            errorMessage = "Something went wrong while subscribing to V3SessionSyncEventBus",
            tag = SR_LOG_TAG
        ) {
            takeIf { v3SessionSyncDisposable == null }?.let {
                v3SessionSyncDisposable =
                    v3SessionSyncResultEventBus.subscribe {
                        executor.defensiveExecute(
                            key = SRExecutionQueues.Main,
                            errorMessage = "Something went wrong while handling V3SessionSyncResult",
                            tag = SR_LOG_TAG
                        ) {
                            handleV3SyncResult(it)
                        }
                    }
            }

        }
    }

    private fun handleV3SyncResult(result: V3SessionSyncResult) {
        when (result) {
            is V3SessionSyncResult.Success -> {
                finalizeRunningSessions()
                cacheHandler.updateSyncStatus(
                    result.sessionIds,
                    SyncStatus.OFFLINE,
                    SyncStatus.READY_FOR_SYNC
                )
                syncJob.start()
            }

            is V3SessionSyncResult.RateLimited -> {
                result.sessionIds.forEach {
                    deleteSessionById(it)
                }
            }
        }
    }

    private fun unSubscribeV3SyncEventBus() {
        executor.defensiveExecute(
            key = SRExecutionQueues.Main,
            errorMessage = "Something went wrong while unsubscribing from V3SessionSyncEventBus",
            tag = SR_LOG_TAG
        ) {
            v3SessionSyncDisposable?.dispose()
            v3SessionSyncDisposable = null
        }
    }

    override fun onNewEvent(event: IBGSdkCoreEvent) = executor.defensiveExecute(
        key = SRExecutionQueues.Main,
        errorMessage = "Failure while handling new event",
        tag = SR_LOG_TAG
    ) {
        when (event) {
            is V3Session -> handleSessionsEvents(event)
            is IBGSdkCoreEvent.FeaturesFetched -> onFeaturesFetched(event)
            is IBGSdkCoreEvent.ReproState -> onReproStateChanged(event.modesMap)
            else -> { /*No-OP*/
            }
        }
    }

    fun setPendingLog(future: Future<Unit>) {
        executor.defensiveExecute(
            key = SRExecutionQueues.Main,
            errorMessage = "Failure while setting pending log",
            tag = SR_LOG_TAG
        ) { pendingLog = future }
    }

    fun handleRuntimeConfigurationsChange(change: (SRConfigurationsProvider) -> Unit) =
        executor.defensiveExecute(
            key = SRExecutionQueues.Main,
            errorMessage = "Failure while handling runtime configurations",
            tag = SR_LOG_TAG
        ) {
            "Feature runtime configurations changed, processing new configurations".logVerbose(tag = SR_LOG_TAG)
            change(configurationHandler)
            onConfigurationsChanged()
        }

    private fun handleSessionsEvents(event: V3Session): Unit = when (event) {
        is V3Session.V3StartedInForeground -> {
            isSREnabledFromSessionStart = configurationHandler.srEnabled
            onSessionStarted(event)
        }

        V3Session.V3SessionFinished -> onSessionFinished()
        else -> Unit
    }

    private fun onFeaturesFetched(featuresFetched: IBGSdkCoreEvent.FeaturesFetched) {
        "Features configurations fetched, processing new configurations".logVerbose(tag = SR_LOG_TAG)
        configurationHandler.handleConfigurationsChange(featuresFetched.response)
        monitorDelegate.onConfigurationsChanged(configurationHandler.isMonitoringAvailable)
        onConfigurationsChanged()
    }

    private fun onConfigurationsChanged() {
        "== Handling feature configuration changes".logVerbose(tag = SR_LOG_TAG)
        loadCacheAndEvaluateReproConfig()
        SRConfigurations(configurationHandler, isSREnabledFromSessionStart)
            .also(monitorDelegate::analyzeAndTrackConfigurations)
        val isCurrentStateDisabled = !configurationHandler.srEnabled
        stateListener.onSRStateChanged(!isCurrentStateDisabled)
        val isV3SessionsDisabled = !InstabugCore.isV3SessionEnabled()
        if (isCurrentStateDisabled || isV3SessionsDisabled) {
            "== Feature is disabled, cleansing old files".logVerbose(tag = SR_LOG_TAG)
            runningSessionCompositeId = null
            srFilesDirectory.setCurrentSpanId(null)
            loggingController.onSessionEnded()
            monitorDelegate.onSRDisabled()
            deleteAllSessions()
            unSubscribeV3SyncEventBus()
        } else {
            subscribeV3SyncEventBus()
            startSessionIfPossible()
        }
    }

    private fun startSessionIfPossible() {
        val isASessionRunning = srFilesDirectory.currentSpanDirectory != null
        if (isASessionRunning) {
            "== A session already running, aborting session starting attempt".logVerbose(tag = SR_LOG_TAG)
            return
        }
        IBGSessionManager.currentSession
            ?.takeUnless { it.startTime.isBackground }
            ?.asForegroundStartEvent
            ?.let(::onSessionStarted)
            ?: "== Starting new session is not possible, v3 is not started".logVerbose(tag = SR_LOG_TAG)
    }

    private fun onReproStateChanged(modeMap: Map<Int, Int>) {
        "Repro configurations changed, processing new configurations".logVerbose(tag = SR_LOG_TAG)
        configurationHandler.handle(modeMap)
        SRConfigurations(configurationHandler, isSREnabledFromSessionStart)
            .also(monitorDelegate::analyzeAndTrackConfigurations)
        reproProxy.evaluate(configurationHandler)
    }

    private fun deleteAllSessions() {
        cacheHandler.queryAll()
            .map { it.uuid }
            .forEach(::deleteSessionById)
    }

    private fun deleteSessionById(sessionId: String) {
        srFilesDirectory
            .operate(DeleteSessionDirectoryOperation(sessionId))
            .get()
        cacheHandler.delete(sessionId)
    }

    private fun onSessionFinished() {
        "Running session ended, waiting on pending logs".logVerbose(tag = SR_LOG_TAG)
        pendingLog?.get()
        pendingLog = null
        executor.defensiveExecute(
            key = SRExecutionQueues.Main,
            errorMessage = "Failure while ending running session",
            tag = SR_LOG_TAG
        ) {
            "Ending running session".logVerbose(tag = SR_LOG_TAG)
            runningSessionCompositeId = null
            loggingController.onSessionEnded()
            srFilesDirectory.setCurrentSpanId(null)
            monitorDelegate.onSessionEnded()
            finalizeRunningSessions()
        }
    }

    private fun onSessionStarted(sessionStartEvent: V3Session.V3StartedInForeground) =
        with(sessionStartEvent) {
            "New session is starting".logVerbose(tag = SR_LOG_TAG)
            loadCacheAndEvaluateReproConfig()
            if (!configurationHandler.srEnabled) {
                "== Feature disabled, aborting starting process".logVerbose(tag = SR_LOG_TAG)
                return
            }
            with(sessionStartEvent) { runningSessionCompositeId = "$startTime-$partialId" }
            with(monitorDelegate) {
                onSessionStarted(uuid)
                SRConfigurations(configurationHandler, isSREnabledFromSessionStart)
                    .also(this::analyzeAndTrackConfigurations)
            }
            srFilesDirectory.setCurrentSpanId(uuid)
            CalculateSRAggregateSizeOperation()
                .let(srFilesDirectory::operateOnOld)
                .also(loggingController::onSessionStarted)
            finalizeRunningSessions()
            sessionReplayUserEvaluator.evaluate()
            cacheHandler.insert(toSRMetaData())
            syncJob.start()
        }

    private fun finalizeRunningSessions() {
        runCatching {
            "== Finalizing old sessions".logVerbose(tag = SR_LOG_TAG)
            var oldRunningSessions =
                cacheHandler.queryByStatus(SyncStatus.RUNNING).map(SRSessionMetadata::uuid)

            val currentSessionUuid = IBGSessionManager.currentSession?.id
            oldRunningSessions = oldRunningSessions.filter { it != currentSessionUuid }
            val operation =
                MapSessionsAndExecuteOperation(oldRunningSessions, compressSRDirsOperation)
            srFilesDirectory.operate(operation).get()
                ?.forEach { sessionID ->
                    cacheHandler.updateStatus(
                        sessionID,
                        SyncStatus.OFFLINE
                    )
                }
        }.onFailure { t -> monitorDelegate.analyzeAndTrackException(ErrorFinalizingSROldSessions(t)) }
            .runOrLogAndReport("Error finalizing old SR sessions")
    }

    private fun V3Session.V3StartedInForeground.toSRMetaData() = SRSessionMetadata(
        uuid = uuid,
        startTime = startTime,
        partialId = partialId,
        status = SyncStatus.RUNNING
    )

    private fun loadCacheAndEvaluateReproConfig() {
        SettingsManager.getInstance()
            .reproConfigurations?.modesMap?.let(configurationHandler::handle)
        reproProxy.evaluate(configurationHandler)
    }


    fun getRunningSessionLink(): Future<String?> =
        executor.defensiveSubmit(
            key = SRExecutionQueues.Main,
            errorMessage = "Error while getting running session composite id",
            tag = SR_LOG_TAG
        ) { composeRunningSessionLink() }

    private fun composeRunningSessionLink(): String? =
        configurationHandler.sessionLinkPrefix?.let { linkPrefix ->
            runningSessionCompositeId?.let { linkId ->
                SESSION_SHAREABLE_LINK_FORMAT.format(linkPrefix, linkId)
            }
        }
}
