package com.instabug.library.networkv2.utils

import androidx.annotation.VisibleForTesting
import com.instabug.library.networkv2.request.Constants.BASE_URL
import com.instabug.library.networkv2.request.Endpoints
import com.instabug.library.networkv2.request.Request
import com.instabug.library.util.StringUtility
import kotlin.math.pow

object NetworkRequestRetryingHelper {

    private const val MAX_RETRIAL_COUNT: Int = 6

    /**
     * Hold a regex format for the urls they should be whitelisted and eligible to be retried.
     *
     * Allow retrying any sdk network request by adding it's url as a key along with it's initial
     * attempted count as a value (By default 0).
     *
     * (Url is added as a regex format with ability to modify any dynamic part of it at runtime
     * to be able to match with the actual request url).
     */
    internal val retryingWhitelist: MutableMap<String, Int> = mutableMapOf(
        StringUtility.getRegexFormatForUrl(
            "$BASE_URL${Endpoints.REPORT_BUG}", null
        ) to 0,
        StringUtility.getRegexFormatForUrl(
            "$BASE_URL${Endpoints.BUG_LOGS}", ":bug_token"
        ) to 0,
        StringUtility.getRegexFormatForUrl(
            "$BASE_URL${Endpoints.ADD_BUG_ATTACHMENT}", ":bug_token"
        ) to 0
    )

    /**
     * Check if the given request should be retried or not.
     *
     * @param request that required to be checked if it should be retried.
     *
     * @return true if the given request exist in the {@link #retryingWhitelist} and didn't
     * exceeded the {@link #MAX_RETRIAL_COUNT} or false otherwise.
     */
    @JvmStatic
    fun shouldRetryRequest(request: Request): Boolean {
        return isRequestEligibleForRetrying(request)
                && getCurrentAttemptNumber(request) < MAX_RETRIAL_COUNT
    }

    /**
     * Increment attempt counter with each new retry.
     *
     * @param request that will be retried to increment it's linked attempt counter.
     */
    @JvmStatic
    fun incrementRequestAttemptCounter(request: Request) {
        getRequestKey(request)?.let { key ->
            retryingWhitelist[key] = retryingWhitelist.getValue(key) + 1
        }
    }

    /**
     * Reset the request attempt counter, this is usually invoke with each new request.
     */
    @JvmStatic
    fun resetRequestAttemptCounter(request: Request) {
        getRequestKey(request)?.let { key ->
            retryingWhitelist[key] = 0
        }
    }

    /**
     * Check if the given request is eligible for retrying or not.
     *
     * @param request to check if it's eligible for retrying.
     *
     * @return true if the request url matches any of the regex in the retryingWhitelist or false otherwise.
     */
    @VisibleForTesting
    fun isRequestEligibleForRetrying(request: Request): Boolean {
        return getRequestKey(request) != null
    }

    /**
     * Find the key that's linked to the given request.
     *
     * @param request that contains a url to find it's linked key in the {@link #retryingWhitelist}.
     *
     * @return the regex key in {@link #retryingWhitelist} for the given request.
     */
    @VisibleForTesting
    fun getRequestKey(request: Request): String? {
        val requestUrl = request.requestUrl
        retryingWhitelist.keys.forEach { key ->
            if (requestUrl.matches(key.toRegex()))
                return key
        }
        return null
    }

    /**
     * Find the attempt retrying number for the given request.
     *
     * @param request to find the linked attempt retrying number.
     *
     * @return The current attempt number for the request if it's eligible for retrying or 0 otherwise.
     */
    @VisibleForTesting
    fun getCurrentAttemptNumber(request: Request): Int {
        getRequestKey(request)?.let { key ->
            return retryingWhitelist.getValue(key)
        }
        return 0
    }

    /**
     * Use the exponential approach to calculate the time required before retrying the request again.
     *
     * @param request that need to get timeout for.
     *
     * @return The time unit the next request retry is triggered (in seconds).
     */
    @JvmStatic
    @JvmName("getNextRetryTimeout")
    fun getNextRetryTimeout(request: Request): Long {
        val nextAttemptedNumber = getCurrentAttemptNumber(request) + 1
        return StrictMath.E.pow(nextAttemptedNumber).toLong()
    }

    fun getRetryingWhitelist(): Map<String, Int> {
        return retryingWhitelist
    }

}