package com.instabug.library.networkv2.connection;

import com.instabug.library.core.InstabugCore;
import com.instabug.library.networkv2.NetworkUtils;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.RequestException;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.authorization.NetworkOfficer;
import com.instabug.library.networkv2.request.Constants;
import com.instabug.library.networkv2.request.Header;
import com.instabug.library.networkv2.request.Request;
import com.instabug.library.networkv2.request.RequestParameter;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * Implementation of the abstraction layer for different connections provided by the {@link com.instabug.library.networkv2.NetworkManager}
 *
 * @see NormalConnectionManager
 * @see MultipartConnectionManager
 * @see FileDownloadConnectionManager
 * @see com.instabug.library.networkv2.request.RequestType
 */
public abstract class InstabugBaseConnectionManagerImpl implements InstabugBaseConnectionManager {
    static final int DEFAULT_READ_TIME_OUT = 10000;
    static final int DEFAULT_CONNECTION_TIME_OUT = 15000;

    @Override
    public HttpURLConnection create(Request request) throws Exception {
        InstabugSDKLogger.d(com.instabug.library.Constants.LOG_TAG, "Starting a request to url: " + request.getRequestUrlForLogging());
        HttpURLConnection httpURLConnection = this.buildBaseConnection(request);
        httpURLConnection = this.setupConnection(httpURLConnection, request);
        return httpURLConnection;
    }

    @Override
    public HttpURLConnection buildBaseConnection(Request request) throws Exception {
        URL url = new URL(request.getRequestUrl());
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestProperty(Header.CONTENT_TYPE, getContentType());
        connection.setRequestProperty(Header.ACCEPT_CHARSET, Constants.UTF_8);
        String requestMethod = request.getRequestMethod();
        if (requestMethod != null) {
            connection.setRequestMethod(requestMethod);
        }
        if (NetworkUtils.isAuthorizationEnabled()) {
            if (NetworkUtils.isAuthorizationRequired(connection.getURL().toString())) {
                String authHeader = NetworkOfficer.getAuthHeader(request);
                if (!authHeader.isEmpty()) {
                    connection.setRequestProperty(Header.AUTHORIZATION, authHeader);
                }
            }
        }
        for (RequestParameter header : request.getHeaders()) {
            connection.setRequestProperty(header.getKey(), (String) header.getValue());
        }
        connection.setDoInput(true);
        return connection;
    }

    @Override
    public String convertStreamToString(InputStream is) {
        BufferedReader reader;
        InputStreamReader in = null;
        try {
            in = new InputStreamReader(is, Constants.UTF_8);
            reader = new BufferedReader(in);

            StringBuilder sb = new StringBuilder();

            String line;
            while ((line = reader.readLine()) != null) {
                sb.append(line).append('\n');
            }
            in.close();
            return sb.toString();
        } catch (OutOfMemoryError e) {
            InstabugCore.reportError(e, "Failed to convert stream of a request: " + e.getMessage());
            InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, "Failed to convert stream of a request due an OOM ");
        } catch (Exception e) {
            InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Failed to convert stream of a request", e);
        } finally {
            try {
                is.close();
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                try {
                    if (in != null) {
                        in.close();
                    }
                } catch (IOException ex) {
                    InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Failed to close stream of a request", e);
                }
                InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, e.getMessage() != null ? e.getMessage() : "Failed to close stream of a request", e);
            }
        }
        return "";
    }

    @Override
    public Map<String, String> getHeaderFields(HttpURLConnection connection) {
        Map<String, String> headers = new HashMap<>();
        for (String key : connection.getHeaderFields().keySet()) {
            if (key != null)
                headers.put(key, connection.getHeaderField(key));
        }
        return headers;
    }

    @Override
    public Throwable handleServerError(HttpURLConnection connection) throws Exception {
        Throwable error ;
        try {
            InputStream errorInputStream = connection.getErrorStream();
            String body = "";
            if (connection.getURL() != null && NetworkUtils.isInstabugRequest(connection.getURL().toString())) {
                InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, "Network request got error");
                body = convertStreamToString(errorInputStream);
                InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG, "Error getting Network request response: " + body);
            }
            error = (!body.isEmpty() &&
                    connection.getResponseCode() == RequestResponse.HttpStatusCode._4xx.RATE_LIMIT_REACHED)
                    ? RateLimitedException.fromResponse(body)
                    : new RequestException(connection.getResponseCode() ,body);
        } catch (OutOfMemoryError outOfMemoryError) {
            InstabugCore.reportError(outOfMemoryError, "OOM while getting network request response: " + outOfMemoryError.getMessage());
            InstabugSDKLogger.e(com.instabug.library.Constants.LOG_TAG,"OOM while getting network request response: " +  outOfMemoryError.getMessage(), outOfMemoryError);
            error = outOfMemoryError;
        }
        return error;
    }
}
