package com.instabug.library.model;


import static com.instabug.library.model.StepType.APPLICATION_CREATED;
import static com.instabug.library.model.StepType.DOUBLE_TAP;
import static com.instabug.library.model.StepType.LONG_PRESS;
import static com.instabug.library.model.StepType.PINCH;
import static com.instabug.library.model.StepType.SCROLL;
import static com.instabug.library.model.StepType.SHAKE;
import static com.instabug.library.model.StepType.SWIPE;
import static com.instabug.library.model.StepType.TAP;
import static com.instabug.library.sessionreplay.SRConstantsKt.LOG_TYPE_KEY;

import android.annotation.SuppressLint;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.Cacheable;
import com.instabug.library.sessionreplay.model.SRLog;
import com.instabug.library.sessionreplay.model.SRLogType;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.StringUtility;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

/**
 * The type User step.
 *
 * @author Hossam
 */
public class UserStep implements Cacheable, Serializable, SRLog {
    private static final String TAG = "UserStep";

    private static final String KEY_TIMESTAMP = "timestamp";
    private static final String KEY_MESSAGE = "message";
    private static final String KEY_TYPE = "type";
    private static final String KEY_ARGS = "args";
    private static final String SOMETHING_WENT_WRONG_WHILE_MAPPING_USER_STEP_TO_JSON_FOR_SR = "Something Went Wrong While mapping User Step to Json for SR";

    private long timeStamp;
    @Nullable
    private String message;
    @Nullable
    private UserStep.Type type;
    @Nullable
    private Args args;

    /**
     * From json array list.
     *
     * @param userStepsAsJsonArray the user steps as json array
     * @return the UserSteps array list
     * @throws JSONException the json exception
     */
    public static ArrayList<UserStep> fromJson(JSONArray userStepsAsJsonArray) throws JSONException {
        ArrayList<UserStep> userSteps = new ArrayList<>();
        if (userStepsAsJsonArray != null && userStepsAsJsonArray.length() > 0) {
            for (int i = 0; i < userStepsAsJsonArray.length(); i++) {
                UserStep userStep = new UserStep();
                userStep.fromJson(userStepsAsJsonArray.getJSONObject(i).toString());
                userSteps.add(userStep);
            }
        }
        return userSteps;
    }

    public static JSONArray toJson(@Nullable List<UserStep> userStepArrayList) {
        JSONArray jsonArray = new JSONArray();
        if (userStepArrayList != null && userStepArrayList.size() > 0) {
            for (UserStep userStep : userStepArrayList) {
                try {
                    jsonArray.put(new JSONObject(userStep.toJson()));
                } catch (JSONException e) {
                    InstabugSDKLogger.v(TAG, e.toString());
                }
            }
        }
        return jsonArray;
    }

    /**
     * Gets time stamp.
     *
     * @return the time stamp
     */
    public long getTimeStamp() {
        return timeStamp;
    }

    /**
     * Sets time stamp.
     *
     * @param timeStamp the time stamp
     */
    public void setTimeStamp(long timeStamp) {
        this.timeStamp = timeStamp;
    }

    /**
     * Gets message.
     *
     * @return the message
     */
    @Nullable
    public String getMessage() {
        return message;
    }

    /**
     * Sets message.
     *
     * @param message the message
     */
    public void setMessage(@Nullable String message) {
        this.message = message;
    }

    /**
     * Gets type.
     *
     * @return the type
     */
    @Nullable
    public UserStep.Type getType() {
        return type;
    }

    /**
     * Sets type.
     *
     * @param type the type
     */
    public void setType(@Nullable UserStep.Type type) {
        this.type = type;
    }

    @Nullable
    public Args getArgs() {
        return args;
    }

    public void setArgs(@Nullable Args args) {
        this.args = args;
    }

    public void setType(@Nullable @StepType String stepType) {
        if (stepType == null) setType(Type.NOT_AVAILABLE);
        else switch (stepType) {
            case TAP:
                setType(Type.TAP);
                break;
            case DOUBLE_TAP:
                setType(Type.DOUBLE_TAP);
                break;
            case LONG_PRESS:
                setType(Type.LONG_PRESS);
                break;
            case SCROLL:
                setType(Type.SCROLL);
                break;
            case SWIPE:
                setType(Type.SWIPE);
                break;
            case PINCH:
                setType(Type.PINCH);
                break;
            case SHAKE:
                setType(Type.MOTION);
                break;
            case APPLICATION_CREATED:
                setType(Type.APPLICATION);
                break;
            default:
                setType(Type.VIEW);
        }

    }

    /**
     * toJson converts data model into Json object
     *
     * @return UserStep as JSONObject
     * @throws JSONException the json exception
     */
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        JSONObject userStepJsonObject = getJsonObject();
        return userStepJsonObject.toString();
    }
    @Nullable
    @Override
    public JSONObject getSrJsonRep() {
        try {
            JSONObject userJsonObject = getJsonObject();
            userJsonObject.put(LOG_TYPE_KEY, getLogType());
            return userJsonObject;
        } catch (JSONException e) {
            IBGDiagnostics.reportNonFatalAndLog(e,
                    SOMETHING_WENT_WRONG_WHILE_MAPPING_USER_STEP_TO_JSON_FOR_SR,
                    Constants.LOG_TAG);
            return null;
        }
    }
    @NonNull
    private JSONObject getJsonObject() throws JSONException {
        JSONObject userStepJsonObject = new JSONObject();
        userStepJsonObject.put(KEY_TIMESTAMP, getTimeStamp());
        userStepJsonObject.put(KEY_MESSAGE, getMessage());
        userStepJsonObject.put(KEY_TYPE, getType() == null ? null : getType().toString());
        if (getArgs() != null)
            userStepJsonObject.put(KEY_ARGS, getArgs().toJson());
        return userStepJsonObject;
    }

    @Override
    public void fromJson(String modelAsJson) throws JSONException {
        JSONObject userStepAsJsonObject = new JSONObject(modelAsJson);
        if (userStepAsJsonObject.has(KEY_TIMESTAMP))
            if (StringUtility.isNumeric(userStepAsJsonObject.getString(KEY_TIMESTAMP))) {
                setTimeStamp(userStepAsJsonObject.getLong(KEY_TIMESTAMP));
            } else {
                try {
                    SimpleDateFormat oldDateStyleFormatter =
                            new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.US);
                    Date date = oldDateStyleFormatter.parse(
                            userStepAsJsonObject.getString(KEY_TIMESTAMP));
                    if (date != null) {
                        setTimeStamp(date.getTime());
                    }
                } catch (ParseException e) {
                    InstabugSDKLogger.e(TAG, e.toString());
                }
            }

        if (userStepAsJsonObject.has(KEY_MESSAGE))
            setMessage(userStepAsJsonObject.getString(KEY_MESSAGE));
        if (userStepAsJsonObject.has(KEY_TYPE))
            switch (userStepAsJsonObject.getString(KEY_TYPE)) {
                case "application":
                    setType(Type.APPLICATION);
                    break;
                case "view":
                    setType(Type.VIEW);
                    break;
                case "motion":
                    setType(Type.MOTION);
                    break;
                case "tap":
                    setType(Type.TAP);
                    break;
                case "pinch":
                    setType(Type.PINCH);
                    break;
                case "long_press":
                    setType(Type.LONG_PRESS);
                    break;
                case "scroll":
                    setType(Type.SCROLL);
                    break;
                case "swipe":
                    setType(Type.SWIPE);
                    break;
                case "double_tap":
                    setType(Type.DOUBLE_TAP);
                    break;
                default:
                    setType(Type.NOT_AVAILABLE);
                    break;
            }
        if (userStepAsJsonObject.has(KEY_ARGS)) {
            Args args = new Args();
            args.fromJson(userStepAsJsonObject.getString(KEY_ARGS));
            setArgs(args);
        }

    }

    @NonNull
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toString() {
        return "UserStep{" + "timeStamp='" + timeStamp + '\'' + ", message='" + message + '\'' + ", type=" + type + '}';
    }

    @Override
    public long getTimestamp() {
        return timeStamp;
    }


    @NonNull
    @Override
    @SRLogType
    public String getLogType() {
        return SRLogType.USER_STEP;
    }

    /**
     * The enum Type.
     */
    public enum Type {
        /**
         * Application type.
         */
        APPLICATION("application"),
        /**
         * View type.
         */
        VIEW("view"),
        /**
         * Motion type.
         */
        MOTION("motion"),

        TAP("tap"),
        PINCH("pinch"), LONG_PRESS("long_press"), SCROLL("scroll"), SWIPE("swipe"), DOUBLE_TAP("double_tap"),
        /**
         * Not available type.
         */
        NOT_AVAILABLE("not_available");
        private final String name;

        Type(String name) {
            this.name = name;
        }

        @NonNull
        @Override
        public String toString() {
            return name;
        }
    }

    public static class Args implements Cacheable, Serializable {
        private static final String KEY_EVENT = "event";
        private static final String KEY_LABEL = "label";
        private static final String KEY_CLASS = "class";
        private static final String KEY_VIEW = "view";

        @Nullable
        private Type event;
        @Nullable
        private String label;
        @Nullable
        private String uiClass;
        @Nullable
        private String view;

        public Args() {
        }

        public Args(@Nullable Type type, @Nullable String label, @Nullable String uiClass, @Nullable String view) {
            setEvent(type);
            setLabel(label);
            setUiClass(uiClass);
            setView(view);
        }

        @Override
        @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
        public String toJson() throws JSONException {
            JSONObject userStepJsonObject = new JSONObject();
            userStepJsonObject.put(KEY_EVENT, getEvent());
            userStepJsonObject.put(KEY_LABEL, getLabel());
            userStepJsonObject.put(KEY_CLASS, getUiClass());
            userStepJsonObject.put(KEY_VIEW, getView());
            return userStepJsonObject.toString();
        }

        @Override
        public void fromJson(String modelAsJson) throws JSONException {
            JSONObject userStepAsJsonObject = new JSONObject(modelAsJson);
            if (userStepAsJsonObject.has(KEY_EVENT))
                switch (userStepAsJsonObject.getString(KEY_EVENT)) {
                    case "application":
                        setEvent(Type.APPLICATION);
                        break;
                    case "view":
                        setEvent(Type.VIEW);
                        break;
                    case "motion":
                        setEvent(Type.MOTION);
                        break;
                    case "tap":
                        setEvent(Type.TAP);
                        break;
                    case "pinch":
                        setEvent(Type.PINCH);
                        break;
                    case "pan":
                        setEvent(Type.LONG_PRESS);
                        break;
                    case "scroll":
                        setEvent(Type.SCROLL);
                        break;
                    case "swipe":
                        setEvent(Type.SWIPE);
                        break;
                    case "double_tap":
                        setEvent(Type.DOUBLE_TAP);
                        break;
                    default:
                        setEvent(Type.NOT_AVAILABLE);
                        break;
                }

            if (userStepAsJsonObject.has(KEY_CLASS))
                setUiClass(userStepAsJsonObject.getString(KEY_CLASS));
            if (userStepAsJsonObject.has(KEY_LABEL))
                setLabel(userStepAsJsonObject.getString(KEY_LABEL));
            if (userStepAsJsonObject.has(KEY_VIEW))
                setView(userStepAsJsonObject.getString(KEY_VIEW));

        }

        @Nullable
        public Type getEvent() {
            return event;
        }

        public void setEvent(@Nullable Type event) {
            this.event = event;
        }

        @Nullable
        public String getLabel() {
            return label;
        }

        public void setLabel(@Nullable String label) {
            this.label = label;
        }

        @Nullable
        public String getUiClass() {
            return uiClass;
        }

        public void setUiClass(@Nullable String uiClass) {
            this.uiClass = uiClass;
        }

        @Nullable
        public String getView() {
            return view;
        }

        public void setView(@Nullable String view) {
            this.view = view;
        }
    }
}
