package com.instabug.library.model;

import android.annotation.SuppressLint;
import android.webkit.MimeTypeMap;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.internal.storage.cache.Cacheable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author mesbah
 */
public class Attachment implements Cacheable, Serializable {

    // keys used by toJson() & fromJson() methods
    static final String KEY_NAME = "name";
    static final String KEY_LOCALE_PATH = "local_path";
    static final String KEY_URL = "url";
    static final String KEY_TYPE = "type";
    static final String KEY_ATTACHMENT_STATE = "attachment_state";
    static final String KEY_VIDEO_ENCODED = "video_encoded";
    static final String KEY_DURATION = "duration";
    private static final String KEY_ENCRYPTED = "isEncrypted";

    private long id = -1;
    @Nullable
    private String name;
    @Nullable
    private String localPath;
    @Nullable
    private String url;
    @Nullable
    private Type type;
    @Nullable
    private AttachmentState attachmentState;
    private boolean isVideoEncoded = false;
    @Nullable
    private String duration;
    private boolean isEncrypted = false;

    // field to animate the icons in image attachments
    private boolean shouldAnimate = true;

    private boolean isRemoved = false;

    public Attachment() {
        setType(Type.NOT_AVAILABLE);
        setAttachmentState(AttachmentState.NOT_AVAILABLE);
    }

    public static JSONArray toJson(List<Attachment> attachments) throws JSONException {
        JSONArray attachmentsJsonArray = new JSONArray();
        if(attachments != null) {
            for (int i = 0; i < attachments.size(); i++) {
                attachmentsJsonArray.put(new JSONObject(attachments.get(i).toJson()));
            }
        }
        return attachmentsJsonArray;
    }

    public static List<Attachment> fromJson(JSONArray attachmentsJsonArray) throws
            JSONException {
        ArrayList<Attachment> attachments = new ArrayList<>();
        for (int i = 0; i < attachmentsJsonArray.length(); i++) {
            Attachment attachment = new Attachment();
            attachment.fromJson(attachmentsJsonArray.getJSONObject(i).toString());
            attachments.add(attachment);
        }
        return attachments;
    }

    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    @Nullable
    public String getName() {
        return name;
    }

    public Attachment setName(@Nullable String name) {
        this.name = name;
        return this;
    }

    @Nullable
    public String getLocalPath() {
        return localPath;
    }

    public Attachment setLocalPath(@Nullable String localPath) {
        this.localPath = localPath;
        return this;
    }

    @Nullable
    public String getUrl() {
        return url;
    }

    public Attachment setUrl(String url) {
        this.url = url;
        return this;
    }

    @Nullable
    public Attachment.Type getType() {
        return type;
    }

    public Attachment setType(Attachment.Type type) {
        this.type = type;
        return this;
    }

    public boolean shouldAnimate() {
        return shouldAnimate;
    }

    public void setShouldAnimate(boolean shouldAnimate) {
        this.shouldAnimate = shouldAnimate;
    }

    public boolean isVideoEncoded() {
        return isVideoEncoded;
    }

    public Attachment setVideoEncoded(boolean videoEncoded) {
        isVideoEncoded = videoEncoded;
        return this;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String getFileType() {
        String fileType;
        String extension = MimeTypeMap.getFileExtensionFromUrl(getName());
        String type = getType() == null ? "" : getType().toString();
        if (extension != null && !extension.equals("")) {
            fileType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
            return (fileType != null && !fileType.equals("")) ? fileType : type;
        } else {
            return type;
        }
    }

    @Nullable
    public AttachmentState getAttachmentState() {
        return attachmentState;
    }

    public Attachment setAttachmentState(AttachmentState attachmentState) {
        this.attachmentState = attachmentState;
        return this;
    }

    public boolean isEncrypted() {
        return isEncrypted;
    }

    public void setEncrypted(boolean encrypted) {
        isEncrypted = encrypted;
    }

    @Nullable
    public String getDuration() {
        return duration;
    }

    public void setDuration(@Nullable String duration) {
        this.duration = duration;
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toJson() throws JSONException {
        JSONObject attachment = new JSONObject();
        attachment.put(KEY_NAME, getName())
                .put(KEY_LOCALE_PATH, getLocalPath())
                .put(KEY_URL, getUrl())
                .put(KEY_VIDEO_ENCODED, isVideoEncoded())
                .put(KEY_ENCRYPTED, isEncrypted())
                .put(KEY_DURATION, getDuration());
        if (getType() != null)
            attachment.put(KEY_TYPE, getType().toString());
        if (getAttachmentState() != null)
            attachment.put(KEY_ATTACHMENT_STATE, getAttachmentState().toString());
        return attachment.toString();
    }

    @Override
    public void fromJson(String attachmentAsJson) throws JSONException {
        JSONObject attachmentJsonObject = new JSONObject(attachmentAsJson);
        if (attachmentJsonObject.has(KEY_NAME))
            setName(attachmentJsonObject.getString(KEY_NAME));
        if (attachmentJsonObject.has(KEY_LOCALE_PATH))
            setLocalPath(attachmentJsonObject.getString(KEY_LOCALE_PATH));
        if (attachmentJsonObject.has(KEY_URL))
            setUrl(attachmentJsonObject.getString(KEY_URL));
        if (attachmentJsonObject.has(KEY_TYPE)) {
            setType(Type.get(attachmentJsonObject.getString(KEY_TYPE)));
        }
        if (attachmentJsonObject.has(KEY_ATTACHMENT_STATE))
            setAttachmentState(AttachmentState.valueOf(attachmentJsonObject.getString
                    (KEY_ATTACHMENT_STATE)));
        if (attachmentJsonObject.has(KEY_VIDEO_ENCODED))
            setVideoEncoded(attachmentJsonObject.getBoolean(KEY_VIDEO_ENCODED));
        if (attachmentJsonObject.has(KEY_DURATION))
            setDuration(attachmentJsonObject.getString(KEY_DURATION));
        if (attachmentJsonObject.has(KEY_ENCRYPTED)) {
            setEncrypted(attachmentJsonObject.getBoolean(KEY_ENCRYPTED));
        }

    }

    public boolean isRemoved() {
        return isRemoved;
    }

    public void setRemoved(boolean removed) {
        isRemoved = removed;
    }

    @NonNull
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public String toString() {
        return "Name: " + getName() + ", Local Path: " + getLocalPath() + ", Type: " + getType()
                + ", Duration: " + getDuration() + ", Url: " + getUrl() + ", Attachment State: "
                + getAttachmentState();
    }

    @Override
    @SuppressLint("NP_METHOD_PARAMETER_TIGHTENS_ANNOTATION")
    public boolean equals(@Nullable Object attachment) {
        if(attachment == null) {
            return false;
        } else {
            if (attachment instanceof Attachment) {
                Attachment comparedAttachment = (Attachment) attachment;
                return String.valueOf(comparedAttachment.getName()).equals(String.valueOf(getName()))
                        && String.valueOf(comparedAttachment.getLocalPath()).equals(String.valueOf
                        (getLocalPath()))
                        && String.valueOf(comparedAttachment.getUrl()).equals(String.valueOf(getUrl()))
                        && comparedAttachment.getType() == getType()
                        && comparedAttachment.getAttachmentState() == getAttachmentState()
                        && comparedAttachment.isVideoEncoded() == isVideoEncoded()
                        && String.valueOf(comparedAttachment.getDuration()).equals(String.valueOf
                        (getDuration()));
            } else {
                return false;
            }
        }
    }

    @Override
    public int hashCode() {
        if (getName() != null)
            return getName().hashCode();
        else
            return -1;
    }

    public enum Type {
        MAIN_SCREENSHOT("main-screenshot"), AUDIO("audio"),
        EXTRA_IMAGE("extra_image"), EXTRA_VIDEO("extra_video"),
        GALLERY_IMAGE("image_gallery"), GALLERY_VIDEO("video_gallery"),
        ATTACHMENT_FILE("attachment-file"), VIEW_HIERARCHY("view-hierarchy-v2"),
        NOT_AVAILABLE("not-available"), VISUAL_USER_STEPS("user-repro-steps-v2"),
        AUTO_SCREEN_RECORDING_VIDEO("auto-screen-recording-v2");
        private final String name;

        // Reverse-lookup map for getting a day from an abbreviation
        private static final Map<String, Type> lookup = new HashMap<>();

        static {
            for (Type type : Type.values()) {
                lookup.put(type.name, type);
            }
        }

        Type(String name) {
            this.name = name;
        }

        @NonNull
        @Override
        public String toString() {
            return name;
        }

        public static Type get(String name) {
            Type type = lookup.get(name);
            return type == null ? NOT_AVAILABLE : type;
        }
    }

    public enum AttachmentState {
        OFFLINE, SYNCED, NOT_AVAILABLE
    }
}
