package com.instabug.library.logscollection

import com.instabug.library.util.extenstions.runOrLogAndReport
import com.instabug.library.util.threading.OrderedExecutorService

interface DataWatcher {
    /**
     * Adds a watcher by ID to block cleansing if not consented.
     * @param watcherId the watcher id to be added.
     */
    fun addWatcher(watcherId: Int)

    /**
     * Consent on cleansing by a given watcher ID.
     * @param watcherId the watcher id to consider the consent from.
     */
    fun consentOnCleansing(watcherId: Int)

    /**
     * Removes a watcher from the watchers list, hence, stop blocking for its consent.
     * @param watcherId the watcher ID to be removed.
     */
    fun removeWatcher(watcherId: Int)

    /**
     * Query the state of a given watcher by its id.
     * @return a nullable [Boolean] value. If null, means the watcher is not added
     */
    fun queryWatcherConsent(watcherId: Int): Boolean?
}

interface GarbageCollector {
    fun invoke()
}

class SimpleDataWatcher(
    private val executor: OrderedExecutorService,
    private val collector: GarbageCollector,
    private val executionQueue: String
) : DataWatcher {
    private val watchersMap: MutableMap<Int, Boolean> = mutableMapOf()

    override fun addWatcher(watcherId: Int) = executor.execute(executionQueue) {
        if (!watchersMap.containsKey(watcherId)) {
            watchersMap[watcherId] = false
        }
    }

    override fun consentOnCleansing(watcherId: Int) = executor.execute(executionQueue) {
        if (watchersMap.containsKey(watcherId)) {
            watchersMap[watcherId] = true
            cleanseIfNeeded()
        }
    }

    override fun removeWatcher(watcherId: Int) = executor.execute(executionQueue) {
        if (watchersMap.containsKey(watcherId)) {
            watchersMap.remove(watcherId)
            cleanseIfNeeded()
        }
    }

    override fun queryWatcherConsent(watcherId: Int): Boolean? =
        executor.submit(executionQueue) { watchersMap[watcherId] }.get()

    private fun cleanseIfNeeded() = runCatching {
        if (watchersMap.all { (_, hasConsented) -> hasConsented }) {
            collector.invoke()
            watchersMap.keys.forEach { watcher -> watchersMap[watcher] = false }
        }
    }.runOrLogAndReport("Couldn't cleanse")
}