package com.instabug.library.internal.sharedpreferences;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.library.Constants;
import com.instabug.library.encryption.EncryptionManager;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.Map;

import static com.instabug.library.settings.SettingsManager.MIGRATION_STATE_SHARED_PREF;

public class SharedPreferencesMigrationEngine {

    private static final String[] OLD_SHARED_PREFERENCES_FILES = {"APM_SHARED_PREFERENCES", "SHARED_PREFERENCES_NDK_CRASHES", "com.instabug.chat"};
    private static final String[] NEW_SHARED_PREFERENCES_FILES = {"instabug_apm", "instabug_ndk_crashes", "instabug_chat"};

    private static final String[] SHARED_PREFERENCES_FILES = {
            "instabug", "instabug_apm", "instabug_chat",
            "instabug_announcements", "instabug_bug_reporting", "instabug_crash",
            "instabug_survey", "instabug_ndk_crashes"};

    public static void startMigration(final boolean isEncryptionEnabled, @Nullable final Context context) {
        if (context != null) {
            CoreServiceLocator.getSharedPreferencesExecutor().execute(new Runnable() {
                @SuppressLint("ApplySharedPref")
                @Override
                public void run() {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "SharedPreferences started migration. Encryption enabled: " + isEncryptionEnabled);
                    SharedPreferences sharedPreferences = context.getSharedPreferences(MIGRATION_STATE_SHARED_PREF, Context.MODE_PRIVATE);
                    SharedPreferences.Editor editor = sharedPreferences.edit();
                    migrateOldFiles(context);
                    for (String sharedPreferencesName : SHARED_PREFERENCES_FILES) {
                        editor.putBoolean(sharedPreferencesName, false).commit();
                        migrateFile(sharedPreferencesName, isEncryptionEnabled, context);
                        editor.putBoolean(sharedPreferencesName, true).commit();
                    }
                    InstabugSDKLogger.d(Constants.LOG_TAG, "SharedPreferences finished migration");

                }
            });
        }
    }

    public static void runMigrationIfNeeded(final boolean isEncryptionEnabled, @Nullable final Context context) {
        if (context != null) {
            CoreServiceLocator.getSharedPreferencesExecutor().execute(new Runnable() {
                @SuppressLint("ApplySharedPref")
                @Override
                public void run() {
                    SharedPreferences sharedPreferences = context.getSharedPreferences(MIGRATION_STATE_SHARED_PREF, Context.MODE_PRIVATE);
                    InstabugSDKLogger.d(Constants.LOG_TAG, "SharedPreferences continue migration. Encryption enabled: " + isEncryptionEnabled);
                    migrateOldFiles(context);
                    SharedPreferences.Editor editor = sharedPreferences.edit();
                    for (String sharedPreferencesName : SHARED_PREFERENCES_FILES) {
                        if (!sharedPreferences.getBoolean(sharedPreferencesName, false)) {
                            migrateFile(sharedPreferencesName, isEncryptionEnabled, context);
                            editor.putBoolean(sharedPreferencesName, true).commit();
                        }
                    }
                    InstabugSDKLogger.d(Constants.LOG_TAG, "SharedPreferences finished migration");
                }
            });
        }
    }

    @SuppressLint("ApplySharedPref")
    private static void migrateOldFiles(Context context) {
        for (int i = 0; i < OLD_SHARED_PREFERENCES_FILES.length; i++) {
            SharedPreferences oldPref = context.getSharedPreferences(OLD_SHARED_PREFERENCES_FILES[i], Context.MODE_PRIVATE);
            SharedPreferences newPref = context.getSharedPreferences(NEW_SHARED_PREFERENCES_FILES[i], Context.MODE_PRIVATE);
            SharedPreferences.Editor newEditor = newPref.edit();
            Map<String, ?> all = oldPref.getAll();
            for (Map.Entry<String, ?> x : all.entrySet()) {
                if (x.getValue().getClass().equals(Boolean.class))
                    newEditor.putBoolean(x.getKey(), (Boolean) x.getValue());
                else if (x.getValue().getClass().equals(Float.class))
                    newEditor.putFloat(x.getKey(), (Float) x.getValue());
                else if (x.getValue().getClass().equals(Integer.class))
                    newEditor.putInt(x.getKey(), (Integer) x.getValue());
                else if (x.getValue().getClass().equals(Long.class))
                    newEditor.putLong(x.getKey(), (Long) x.getValue());
                else if (x.getValue().getClass().equals(String.class))
                    newEditor.putString(x.getKey(), (String) x.getValue());
            }
            newEditor.commit();
            SharedPreferences.Editor oldEditor = oldPref.edit();
            oldEditor.clear();
            oldEditor.commit();
        }
    }

    @SuppressLint("ApplySharedPref")
    @WorkerThread
    private static void migrateFile(String sharedPreferencesName, boolean isEncryptionEnabled, Context context) {
        InstabugSDKLogger.v(Constants.LOG_TAG, "Migrating shared preferences file " + sharedPreferencesName);
        SharedPreferences sharedPreferences = context.getSharedPreferences(sharedPreferencesName, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        Map<String, ?> map = sharedPreferences.getAll();
        for (Map.Entry<String, ?> entry : map.entrySet()) {
            if (entry.getValue() instanceof String) {
                String value = (String) entry.getValue();
                if (isEncryptionEnabled) {
                    // if can't be decrypted then value is plain text that needs to be encrypted
                    if (!canBeDecrypted(value)) {
                        value = encrypt(value);
                    }
                } else {
                    value = decrypt(value);
                }
                editor.putString(entry.getKey(), value);
            }
        }
        editor.commit();
    }

    private static String encrypt(String value) {

        String encryptedValue = EncryptionManager.encrypt(value);
        if (encryptedValue == null) {
            return value;
        }
        return encryptedValue;
    }

    private static String decrypt(String value) {
        String decryptedValue = EncryptionManager.decrypt(value);
        if (decryptedValue == null) {
            return value;
        }
        return decryptedValue;
    }

    private static boolean canBeDecrypted(String value) {
        String decryptedValue = EncryptionManager.decrypt(value);
        return decryptedValue != null && !value.equals(decryptedValue);
    }
}
