package com.instabug.library.internal.servicelocator;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;

/**
 * ServiceLocator abstracts object creation and dependencies providing; by providing a simple
 * interface to clients. This reduces the client’s complexity.
 * In addition, the same client or other clients can reuse the same Service Locator.
 */
@RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
public abstract class ServiceLocator {

    /**
     * Constructs and returns a ServiceLocator instance from the provided {@link InitialContext}
     * & {@link ServiceRegistry}
     *
     * @param initialContext the ServiceLocator initial context
     * @param registry       the ServiceLocator service registry
     * @return ServiceLocator instance from the provided context & registry
     */
    @NonNull
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static ServiceLocator getInstance(@NonNull InitialContext initialContext, @NonNull ServiceRegistry registry) {
        return new ServiceLocatorImpl(initialContext, registry);
    }

    /**
     * Get a service described by its class type. If no such service exists, this method returns null.
     *
     * @param serviceType the class type of the requested service
     * @return the service instance if present or {@code null} otherwise
     */
    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public abstract <T> T getService(@NonNull Class<T> serviceType);

    /**
     * Releases all held resources. All subsequent calls to {@link #getService(Class)} will lead to
     * re-initializing the requested service if present in the {@link InitialContext}
     */
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public abstract void releaseHeldResources();
}
