package com.instabug.library.internal.resolver;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.nonfatals.settings.NonFatalSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Non fatal settings resolver which will hold the non-fatals setting to get any settings
 */

public class NonFatalSettingResolver {

    private static NonFatalSettingResolver nonFatalSettingResolver;
    @Nullable
    private volatile NonFatalSettings nonFatalsSettings;

    public static synchronized NonFatalSettingResolver getInstance() {
        if (nonFatalSettingResolver == null) {
            nonFatalSettingResolver = new NonFatalSettingResolver();
        }
        return nonFatalSettingResolver;
    }

    /**
     * Resolve non-fatals settings from json object to return a non-fatals setting
     * it will return a new object with the new settings.
     *
     * @param jsonObject response of diagnostics settings.
     * @return non-fatals settings.
     * @throws JSONException
     */
    @Nullable
    private NonFatalSettings resolveNonFatalsSettings(@NonNull JSONObject jsonObject) throws JSONException {
        NonFatalSettings nonFatalSettings = new NonFatalSettings();
        nonFatalSettings.fromJson(jsonObject);
        return nonFatalsSettings;
    }

    public void setNonFatalSettings(@Nullable JSONObject nonFatalsJson) throws JSONException {
        if (nonFatalsJson == null) {
            nonFatalsSettings = null;
            return;
        }
        this.nonFatalsSettings = resolveNonFatalsSettings(nonFatalsJson);
        SettingsManager.getInstance().setNonFatalsFeatureSettings(nonFatalsJson.toString());
    }

    /**
     * A method to get non-fatals settings
     *
     * @return non-fatals setting or null if non-fatals setting not found
     */
    @Nullable
    public NonFatalSettings getNonFatalsSettings() {
        try {
            String json = SettingsManager.getInstance().getNonFatalsFeatureSettings();
            if (json != null) {
                NonFatalSettings nonFatalSettings = new NonFatalSettings();
                nonFatalSettings.fromJson(json);
                nonFatalsSettings = nonFatalSettings;
            }
        } catch (Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong while getting NonFatals settings", e);
        }
        return nonFatalsSettings;
    }
}
