package com.instabug.library.internal.media;

import android.media.MediaPlayer;

import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class AudioPlayer {
    @NonNull
    private final Map<String, OnStopListener> onStopListeners;
    @Nullable
    private String filePath;
    @Nullable
    private MediaPlayer mediaPlayer;
    @Nullable
    private OnGetDurationListener onGetDurationListener;
    @Nullable
    private MediaPlayer.OnCompletionListener onCompletionListener;

    public AudioPlayer() {
        onStopListeners = new HashMap<>();
    }

    enum Action {
        START, PAUSE, GET_DURATION

    }

    private void prepare(final Action actionOnPrepared) {
        try {
            mediaPlayer = new MediaPlayer();
            if (filePath != null) mediaPlayer.setDataSource(filePath);
            mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
                @Override
                public void onPrepared(MediaPlayer mediaPlayer) {
                    doAction(actionOnPrepared);
                }
            });
            mediaPlayer.prepareAsync();

            if (onCompletionListener != null) {
                mediaPlayer.setOnCompletionListener(this.onCompletionListener);
            }
        } catch (IOException e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Playing audio file failed", e);
        }
    }

    public void addOnStopListener(OnStopListener onStopListener) {
        this.onStopListeners.put(onStopListener.getFilePath(), onStopListener);
        if (this.onCompletionListener == null) {
            this.onCompletionListener = new MediaPlayer.OnCompletionListener() {
                @Override
                public void onCompletion(MediaPlayer mediaPlayer) {
                    notifyOnStopListeners();
                }
            };

            if (mediaPlayer != null) {
                mediaPlayer.setOnCompletionListener(this.onCompletionListener);
            }
        }
    }

    private void notifyOnStopListeners() {

        for (OnStopListener onStopListener : onStopListeners.values()) {
            onStopListener.onStop();
        }
    }

    private void stream(String filePath, Action actionOnPrepared) {
        if (filePath == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Audio file path can not be null");
            return;
        }
        if (filePath.equals(this.filePath)) {
            doAction(actionOnPrepared);
        } else {
            this.filePath = filePath;
            prepare(actionOnPrepared);
        }
    }

    public void getDurationAsync(String filePath, OnGetDurationListener onGetDurationListener) {
        stream(filePath, Action.GET_DURATION);
        this.onGetDurationListener = onGetDurationListener;
    }


    public void start(String filePath) {
        pause();
        stream(filePath, Action.START);
    }

    public void pause() {
        notifyOnStopListeners();
        doAction(Action.PAUSE);
    }

    public void release() {
        filePath = null;
        if (mediaPlayer != null) {
            if (mediaPlayer.isPlaying()) {
                mediaPlayer.stop();
            }
            mediaPlayer.release();
            mediaPlayer = null;
        }

    }


    private void doAction(Action action) {

        switch (action) {

            case START:
                if (mediaPlayer != null && !mediaPlayer.isPlaying()) {
                    mediaPlayer.start();
                }
                break;
            case PAUSE:

                if (mediaPlayer != null && mediaPlayer.isPlaying()) {
                    mediaPlayer.pause();
                }
                break;

            case GET_DURATION:
                if (onGetDurationListener != null && mediaPlayer != null) {
                    onGetDurationListener.onGetDuration(mediaPlayer.getDuration());
                }
                break;


        }
    }

    public interface OnGetDurationListener {
        void onGetDuration(int duration);
    }

    public static abstract class OnStopListener {
        private String filePath;

        public String getFilePath() {
            return filePath;
        }

        public OnStopListener(String filePath) {
            this.filePath = filePath;
        }

        public abstract void onStop();
    }

    public static String getFormattedDurationText(long durationInMillis) {

        final int HOUR = 60 * 60 * 1000;
        final int MINUTE = 60 * 1000;
        final int SECOND = 1000;

        int durationHour = (int) (durationInMillis / HOUR);
        int durationMint = (int) ((durationInMillis % HOUR) / MINUTE);
        int durationSec = (int) ((durationInMillis % MINUTE) / SECOND);

        if (durationHour > 0) {
            return String.format(Locale.getDefault(), "%02d:%02d:%02d", durationHour, durationMint, durationSec);
        } else {
            return String.format(Locale.getDefault(),"%02d:%02d", durationMint, durationSec);
        }
    }

}
