package com.instabug.library.diagnostics.nonfatals.cache;

import static com.instabug.library.diagnostics.nonfatals.cache.NonFatalsDBHelper.NOT_FOUND;

import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.diagnostics.nonfatals.di.ServiceLocator;
import com.instabug.library.diagnostics.nonfatals.model.NonFatal;
import com.instabug.library.diagnostics.nonfatals.model.Occurrence;
import com.instabug.library.diagnostics.nonfatals.settings.NonFatalSettings;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation;
import com.instabug.library.internal.storage.operation.WriteStateToFileDiskOperation;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Created by Shannan on 02/09/2021.
 */
public final class NonFatalCacheManagerImpl implements NonFatalCacheManager {


    @NonNull
    private final NonFatalsDBHelper nonFatalsDBHelper;
    @NonNull
    private final OccurrencesDBHelper occurrencesDBHelper;
    @NonNull
    private final NonFatalSettings nonFatalSettings;

    public NonFatalCacheManagerImpl(@NonNull NonFatalsDBHelper nonFatalsDBHelper,
                                    @NonNull OccurrencesDBHelper occurrencesDBHelper,
                                    @NonNull NonFatalSettings nonFatalSettings) {
        this.nonFatalsDBHelper = nonFatalsDBHelper;
        this.occurrencesDBHelper = occurrencesDBHelper;
        this.nonFatalSettings = nonFatalSettings;
    }

    @Override
    public void saveNonFatal(@NonNull NonFatal nonFatal) {

        long nonFatalId = nonFatalsDBHelper.getNonFatalId(nonFatal);
        boolean newNonFatal = nonFatalId == NOT_FOUND;
        if (newNonFatal) {
            nonFatalId = nonFatalsDBHelper.insertNonFatal(nonFatal);
            if(nonFatalId == NOT_FOUND) return;
            List<Long> ids = nonFatalsDBHelper.getEligibleToBeDeleted(nonFatalSettings.getNonFatalsMaxCount());
            deleteStateFilesForNonFatals(ids);
            nonFatalsDBHelper.trimToLimit(ids);
        }

        boolean hasValidNonFatalId = nonFatalId != NOT_FOUND;
        if (hasValidNonFatalId) {
            int occurrencesCount = occurrencesDBHelper.getNonFatalOccurrencesCount(nonFatalId);
            if (occurrencesCount < nonFatalSettings.getOccurrencesMaxCount()) {
                File stateFile = prepareState();
                if (stateFile != null) {
                    Occurrence occurrence = new Occurrence(nonFatalId, System.currentTimeMillis(), stateFile.toURI().toString());
                    boolean inserted = occurrencesDBHelper.insertOccurrence(occurrence);
                    if(!inserted) stateFile.delete();
                }
            }
            InstabugSDKLogger.d(Constants.LOG_TAG, nonFatal.getExceptionType() + " has been reported");
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Something went wrong! NonFatal not reported!!");
        }
    }

    private void deleteStateFilesForNonFatals(@Nullable List<Long> ids) {
        if (ids != null) {
            for (long id : ids) {
                String[] stateFiles = occurrencesDBHelper.getStateFilesForNonFatal(id);
                if (stateFiles != null) {
                    for (String stateFile : stateFiles)
                        new DeleteUriDiskOperation(Uri.parse(stateFile)).execute(null);
                }
            }
        }
    }

    @Override
    public List<Long> saveNonFatals(@NonNull List<NonFatal> nonFatals) {
        List<Long> ids = new ArrayList<>();
        for (NonFatal nonFatal : nonFatals) {
            ids.add(nonFatalsDBHelper.insertNonFatal(nonFatal));
        }

        return ids;
    }

    @NonNull
    @Override
    public List<NonFatal> getAllNonFatals() {
        return nonFatalsDBHelper.getAllNonFatals();
    }

    @NonNull
    @Override
    public List<Occurrence> getNonFatalOccurrences(long nonFatalId) {
        return occurrencesDBHelper.getNonFatalOccurrences(nonFatalId);
    }

    @Nullable
    private File prepareState() {

        Context context = ServiceLocator.getContext();
        if (context == null) return null;
        State state = new State.Builder(context)
                .buildSimplifiedState();
        File file = DiskUtils.createStateTextFile(context, NON_FATAL_STATE);
        Uri uri;
        try {
            uri = DiskUtils.with(context)
                    .writeOperation(new WriteStateToFileDiskOperation(file, state.toJson()))
                    .execute();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
        state.setUri(uri);
        return file;
    }

    @Override
    public void saveOccurrence(Occurrence occurrence) {
        occurrencesDBHelper.insertOccurrence(occurrence);
    }

    @Override
    public void clearCache() {
        occurrencesDBHelper.clearAll();
        nonFatalsDBHelper.clearAll();
    }

    @NonNull
    @Override
    public List<Occurrence> getAllOccurrences() {
        return occurrencesDBHelper.getAllOccurrences();
    }

    @Override
    public List<String> getStateFilesForAllOccurrences() {
        return occurrencesDBHelper.getStateFiles();
    }

    @Override
    public void deleteNonFatal(long id) {
        nonFatalsDBHelper.deleteNonFatal(id);
    }

    @Override
    public void deleteOccurrence(@Nullable String stateFileUri) {
        if (stateFileUri != null) {
            occurrencesDBHelper.deleteOccurrence(stateFileUri);
        }
    }
}
