package com.instabug.library.diagnostics.diagnostics_db;

import android.annotation.SuppressLint;
import android.content.Context;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.apichecker.ReturnableRunnable;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;
import com.instabug.library.internal.storage.cache.dbv2.IBGCursor;
import com.instabug.library.internal.storage.cache.dbv2.IBGWhereArg;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.MonitoredSingleThreadExecutor;
import com.instabug.library.util.threading.PoolProvider;

import java.util.Arrays;
import java.util.List;

public class DiagnosticsDbManager {


    @VisibleForTesting
    public static SQLiteOpenHelper dbHelper;
    @Nullable
    private static DiagnosticsDbManager instance;
    @Nullable
    private Boolean databaseTransactionsEnabled;
    private final MonitoredSingleThreadExecutor executor;

    @Nullable
    private SQLiteDatabase database;

    public static synchronized void init(DiagnosticsDbHelper helper) {
        if (instance == null) {
            instance = new DiagnosticsDbManager();
            dbHelper = helper;
        }
    }

    @SuppressLint("RESOURCE_LEAK")
    public static synchronized DiagnosticsDbManager getInstance() throws IllegalStateException {
        if (instance == null) {
            if (Instabug.getApplicationContext() != null) {
                DiagnosticsDbManager.init(new DiagnosticsDbHelper(Instabug.getApplicationContext()));
            } else {
                throw new IllegalStateException(Constants.LOG_TAG + " is not initialized, call init(..) method first.");
            }
        }
        return instance;
    }

    private DiagnosticsDbManager() {
        executor = PoolProvider.getDiagnosticsDatabaseExecutor();
    }

    private synchronized void openDatabase() {
        if (database == null || !database.isOpen()) {
            database = dbHelper.getWritableDatabase();
        }
    }

    private synchronized boolean databaseInitializedAndOpen() {
        return database != null && database.isOpen();
    }

    @VisibleForTesting
    @SuppressLint("ERADICATE_FIELD_NOT_NULLABLE")
    public static synchronized void tearDown() {
        if (dbHelper != null) {
            dbHelper.close();
            dbHelper = null;
        }
        if (instance != null) {
            if (instance.database != null) {
                instance.database.close();
                instance.database = null;
            }
            instance = null;
        }
    }


    public long insert(final @NonNull String table, final @Nullable String nullColumnHack,
                       final @NonNull IBGContentValues values) {
        Long id = executor.executeAndGet(new ReturnableRunnable<Long>() {
            @Override
            @SuppressLint({"ERADICATE_PARAMETER_NOT_NULLABLE", "ERADICATE_NULLABLE_DEREFERENCE"})
            public Long run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return database.insertOrThrow(table, nullColumnHack, values.toContentValues());
                    } else {
                        logOperationFailedWarning("DB insertion failed, database not initialized");
                        return -1L;
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB insertion failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB insertion failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    return -1L;
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB insertion failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB insertion failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    return -1L;
                }
            }
        });

        return id == null ? -1 : id;

    }

    public long insertWithOnConflict(final @NonNull String tableName,
                                     final @Nullable String nullColumnHack,
                                     final @NonNull IBGContentValues values) {
        Long id = executor.executeAndGet(new ReturnableRunnable<Long>() {
            @Override
            @SuppressLint({"ERADICATE_PARAMETER_NOT_NULLABLE", "ERADICATE_NULLABLE_DEREFERENCE"})
            public Long run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return database.insertWithOnConflict(tableName, nullColumnHack, values.toContentValues(), SQLiteDatabase.CONFLICT_IGNORE);
                    } else {
                        logOperationFailedWarning("DB insertion with on conflict failed database is not initialized");
                        return -1L;
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB insertion with on conflict failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB insertion with on conflict failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    return -1L;
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB insertion with on conflict failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB insertion with on conflict failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    return -1L;
                }
            }
        });

        return id == null ? -1 : id;
    }

    public void execSQL(final @NonNull String sql) {
        executor.execute(new Runnable() {
            @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
            @Override
            public void run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        database.execSQL(sql);
                    } else {
                        logOperationFailedWarning("DB execution a sql failed");
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB execution a sql failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB execution a sql failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB execution a sql failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB execution a sql failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                }
            }
        });
    }

    public long insertWithOnConflictReplace(final @NonNull String tableName,
                                            final @Nullable String nullColumnHack,
                                            final @NonNull IBGContentValues values) {
        Long id = executor.executeAndGet(new ReturnableRunnable<Long>() {
            @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
            @Override
            public Long run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return database.insertWithOnConflict(tableName, nullColumnHack, values.toContentValues(), SQLiteDatabase.CONFLICT_REPLACE);
                    } else {
                        logOperationFailedWarning("DB insertion with on conflict replace failed");
                        return -1L;
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB insertion with on conflict replace failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB insertion with on conflict replace failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    return -1L;
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB insertion with on conflict replace failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB insertion with on conflict replace failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    return -1L;
                }
            }
        });

        return id == null ? -1 : id;

    }

    public int delete(final @NonNull String table,
                      final @Nullable String whereClause,
                      final @Nullable List<IBGWhereArg> whereArgs) {

        Integer columnsAffected = executor.executeAndGet(new ReturnableRunnable<Integer>() {
            @Override
            @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
            public Integer run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return database.delete(table, whereClause, IBGWhereArg.argsListToStringArray(whereArgs));
                    } else {
                        logOperationFailedWarning("DB deletion failed");
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB deletion failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB deletion failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB deletion failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB deletion failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                }
                return 0;
            }
        });
        return columnsAffected == null ? 0 : columnsAffected;
    }

    @Nullable
    public IBGCursor query(final String table, final @Nullable String[] columns,
                           final @Nullable String selection,
                           final @Nullable List<IBGWhereArg> selectionArgs,
                           final @Nullable String groupBy,
                           final @Nullable String having,
                           final @Nullable String orderBy) {

        return executor.executeAndGet(new ReturnableRunnable<IBGCursor>() {
            @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
            @Nullable
            @Override
            public IBGCursor run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return new IBGCursor(database.query(table, columns, selection, IBGWhereArg.argsListToStringArray(selectionArgs), groupBy, having, orderBy));
                    } else {
                        logOperationFailedWarning("DB query faile");
                        return null;
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB query failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB query faile due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    return null;
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB query failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB query faile due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    return null;
                }
            }
        });
    }

    @SuppressWarnings("ConstantConditions")
    public int update(final @NonNull String table,
                      final @NonNull IBGContentValues values,
                      final @Nullable String whereClause,
                      final @Nullable List<IBGWhereArg> whereArgs) {

        Integer columnsAffected = executor.executeAndGet(new ReturnableRunnable<Integer>() {
            @Nullable
            @Override
            @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
            public Integer run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return database.update(table, values.toContentValues(), whereClause, IBGWhereArg.argsListToStringArray(whereArgs));
                    } else {
                        logOperationFailedWarning("DB update failed");
                        return -1;
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB update failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB update failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    return -1;
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB update failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB update failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    return -1;
                }
            }
        });

        return columnsAffected == null ? 0 : columnsAffected;
    }

    @Nullable
    public IBGCursor query(final String table,
                           final @Nullable String[] columns,
                           final @Nullable String selection,
                           final @Nullable List<IBGWhereArg> selectionArgs,
                           final @Nullable String groupBy,
                           final @Nullable String having,
                           final @Nullable String orderBy,
                           final @Nullable String limit) {

        return executor.executeAndGet(new ReturnableRunnable<IBGCursor>() {
            @Nullable
            @Override
            @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
            public IBGCursor run() {
                openDatabase();
                try {
                    if (databaseInitializedAndOpen()) {
                        return new IBGCursor(database.query(table, columns, selection, IBGWhereArg.argsListToStringArray(selectionArgs), groupBy, having, orderBy,
                                limit));
                    } else {
                        logOperationFailedWarning("DB query failed");
                        return null;
                    }
                } catch (Exception ex) {
                    IBGDiagnostics.reportNonFatal(ex, "DB query failed: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    logOperationFailedWarning("DB query failed due to: " + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
                    return null;
                } catch (OutOfMemoryError oom) {
                    IBGDiagnostics.reportNonFatal(oom, "DB query failed: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    logOperationFailedWarning("DB query failed due to: " + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
                    return null;
                }
            }
        });
    }

    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public synchronized void beginTransaction() {
        openDatabase();
        try {
            if (databaseInitializedAndOpen()) {
                if (isDatabaseTransactionsEnabled()) {
                    database.beginTransaction();
                }
            } else {
                logOperationFailedWarning("DB transaction failed");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, "DB transaction failed: " + ex.getMessage());
            logOperationFailedWarning("DB transaction failed due to:" + ex.getMessage());
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, "DB transaction failed: " + oom.getMessage());
            logOperationFailedWarning("DB transaction failed due to: " + oom.getMessage());
        }
    }

    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public synchronized void endTransaction() {
        String dbEndTransactionFailed = "DB end transaction not successful due to: ";
        try {
            if (databaseInitializedAndOpen()) {
                if (isDatabaseTransactionsEnabled()) {
                    database.endTransaction();
                }
            } else {
                logOperationFailedWarning("DB end transaction not successful");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, dbEndTransactionFailed + ex.getMessage());
            logOperationFailedWarning(dbEndTransactionFailed + ex.getMessage());
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, dbEndTransactionFailed + oom.getMessage());
            logOperationFailedWarning(dbEndTransactionFailed + oom.getMessage());
        }
    }


    private synchronized boolean isDatabaseTransactionsEnabled() {
        if (databaseTransactionsEnabled == null && Instabug.getApplicationContext() != null) {
            databaseTransactionsEnabled = !InstabugCore.isDatabaseTransactionDisabled();
        }
        return databaseTransactionsEnabled != null ? databaseTransactionsEnabled : false;
    }


    private synchronized void logOperationFailedWarning(String message) {
        if (database == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Attempted to do operation on an uninitialized " +
                    "database. Falling back silently");
        } else if (!database.isOpen()) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Attempted to do operation on a closed database. " +
                    "Falling back silently");
        } else {
            InstabugSDKLogger.w(Constants.LOG_TAG, message);
        }
    }

    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    public synchronized void setTransactionSuccessful() {
        String dbTransactionFailed = "DB transaction not successful due to: ";
        try {
            if (databaseInitializedAndOpen()) {
                if (isDatabaseTransactionsEnabled()) {
                    database.setTransactionSuccessful();
                }
            } else {
                logOperationFailedWarning("DB transaction not successful");
            }
        } catch (Exception ex) {
            IBGDiagnostics.reportNonFatal(ex, dbTransactionFailed + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
            logOperationFailedWarning(dbTransactionFailed + ex.getMessage() + Arrays.toString(ex.getStackTrace()));
        } catch (OutOfMemoryError oom) {
            IBGDiagnostics.reportNonFatal(oom, dbTransactionFailed + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
            logOperationFailedWarning(dbTransactionFailed + oom.getMessage() + Arrays.toString(oom.getStackTrace()));
        }
    }

    public synchronized boolean deleteDatabase(Context context) {
        dbHelper.close();
        return context.deleteDatabase(dbHelper.getDatabaseName());
    }

}
