package com.instabug.library.core;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.ColorInt;
import androidx.annotation.ColorRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.annotation.WorkerThread;

import com.instabug.library.BuildConfig;
import com.instabug.library.Constants;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.InstabugColorTheme;
import com.instabug.library.InstabugFeaturesManager;
import com.instabug.library.LogLevel;
import com.instabug.library.OnSdkInvokedCallback;
import com.instabug.library.Platform;
import com.instabug.library.ReflectionUtils;
import com.instabug.library.SessionManager;
import com.instabug.library.core.eventbus.AutoScreenRecordingEventBus;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.core.plugin.PluginsManager;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.encryption.EncryptorVersions;
import com.instabug.library.experiments.ExperimentsManager;
import com.instabug.library.internal.orchestrator.Action;
import com.instabug.library.internal.orchestrator.ActionsOrchestrator;
import com.instabug.library.internal.servicelocator.CoreServiceLocator;
import com.instabug.library.internal.servicelocator.ServiceLocator;
import com.instabug.library.internal.servicelocator.application.ApplicationServiceLocator;
import com.instabug.library.internal.storage.ProcessedBytes;
import com.instabug.library.internal.storage.cache.db.userAttribute.UserAttributeCacheManager;
import com.instabug.library.internal.video.InternalAutoScreenRecorderHelper;
import com.instabug.library.internal.video.ScreenRecordingServiceAction;
import com.instabug.library.invocation.InstabugInvocationEvent;
import com.instabug.library.invocation.util.InstabugVideoRecordingButtonPosition;
import com.instabug.library.model.Report;
import com.instabug.library.model.common.Session;
import com.instabug.library.model.v3Session.IBGInMemorySession;
import com.instabug.library.model.v3Session.IBGSessionMapper;
import com.instabug.library.sessionV3.configurations.IBGSessionCrashesConfigurations;
import com.instabug.library.sessionV3.di.IBGSessionServiceLocator;
import com.instabug.library.sessionV3.manager.IBGSessionManager;
import com.instabug.library.sessionV3.sync.IBGSessionBatchEvaluatorImpl;
import com.instabug.library.sessionV3.sync.SessionBatchingFilter;
import com.instabug.library.settings.InstabugMinimalPersistableSettings;
import com.instabug.library.settings.PerSessionSettings;
import com.instabug.library.settings.PersistableSettings;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.tracking.ComposeLifeCycleMonitor;
import com.instabug.library.tracking.CurrentViewProvider;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.user.UserManager;
import com.instabug.library.util.BitmapUtils;
import com.instabug.library.util.FileUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.PbiFooterUtils;
import com.instabug.library.visualusersteps.GroupViewTouchedViewProcessor;
import com.instabug.library.visualusersteps.TouchedViewExtractor;
import com.instabug.library.visualusersteps.TouchedViewsProcessor;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;

/**
 * Created by tarek on 4/24/17.
 */

public class InstabugCore {

    private static boolean isApmPluginNotLoaded = false;
    private static boolean isCrashPluginNotLoaded = false;

    public static boolean isFeaturesFetchedBefore() {
        return InstabugFeaturesManager.getInstance().isFeaturesFetchedBefore();
    }

    public static Feature.State getFeatureState(@IBGFeature.Companion.IBGFeature String feature) {
        return InstabugFeaturesManager.getInstance().getFeatureState(feature);
    }

    public static boolean isFeatureAvailable(@IBGFeature.Companion.IBGFeature String feature) {
        return InstabugFeaturesManager.getInstance().isFeatureAvailable(feature);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static boolean isDatabaseTransactionDisabled() {
        return InstabugFeaturesManager.getInstance().isDatabaseTransactionDisabled();
    }

    public static boolean isExperimentalFeatureAvailable(@IBGFeature.Companion.IBGFeature String feature) {
        return InstabugFeaturesManager.getInstance().isExperimentalFeatureAvailable(feature);
    }

    @Nullable
    public static LinkedHashMap<Uri, String> getExtraAttachmentFiles() {
        return SettingsManager.getInstance().getExtraAttachmentFiles();
    }

    public static String getSDKVersion() {
        return BuildConfig.SDK_VERSION;
    }

    public static int getStartedActivitiesCount() {
        return CoreServiceLocator.getStartedActivitiesCounter().getCount();
    }

    /**
     * Start the Vital SDK components that doesn't require the SDK to be built
     */
    public static void startVitalComponents(@Nullable Application application) {
        if (application != null) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Starting vital components");
            InstabugInternalTrackingDelegate.init(application);
        }
    }

    public static int getPrimaryColor() {
        return SettingsManager.getInstance().getPrimaryColor();
    }

    public static InstabugColorTheme getTheme() {
        return SettingsManager.getInstance().getTheme();
    }

    public static String getIdentifiedUserEmail() {
        return UserManager.getIdentifiedUserEmail();
    }

    @Nullable
    public static String getEnteredEmail() {
        return UserManager.getEnteredEmail();
    }

    public static void setEnteredEmail(String enteredEmail) {
        UserManager.setEnteredEmail(enteredEmail);
    }

    @Nullable
    public static String getIdentifiedUsername() {
        return UserManager.getIdentifiedUsername();
    }

    public static String getUserData() {
        return SettingsManager.getInstance().getUserData();
    }

    public static void setIdentifiedUserEmail(String userEmail) {
        UserManager.setIdentifiedUserEmail(userEmail);
    }

    public static void setEnteredUsername(String enteredUsername) {
        UserManager.setEnteredUsername(enteredUsername);
    }

    @Nullable
    public static String getEnteredUsername() {
        return UserManager.getEnteredUsername();
    }

    @Nullable
    public static OnSdkInvokedCallback getOnSdkInvokedCallback() {
        return SettingsManager.getInstance().getOnSdkInvokedCallback();
    }

    public static String getTagsAsString() {
        return SettingsManager.getInstance().getTagsAsString();
    }

    public static void setLastContactedAt(long time) {
        SettingsManager.getInstance().setLastContactedAt(time);
    }

    public static boolean isUserLoggedOut() {
        return SettingsManager.getInstance().isUserLoggedOut();
    }

    public static boolean isForegroundBusy() {
        boolean isPromptOptionsScreenShow =
                SettingsManager.getInstance().isPromptOptionsScreenShown();
        boolean isRequestPermissionScreenShown =
                SettingsManager.getInstance().isRequestPermissionScreenShown();
        boolean isOnBoardingShowing =
                SettingsManager.getInstance().isOnboardingShowing();
        boolean isForegroundBusy = PluginsManager.isForegroundBusy();
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InstabugCore#isForegroundBusy] isPromptOptionsScreenShow: " + isPromptOptionsScreenShow);
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InstabugCore#isForegroundBusy] isRequestPermissionScreenShown: " + isRequestPermissionScreenShown);
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InstabugCore#isForegroundBusy] isOnBoardingShowing: " + isOnBoardingShowing);
        InstabugSDKLogger.d(Constants.LOG_TAG, "[InstabugCore#isForegroundBusy] isForegroundBusy: " + isForegroundBusy);
        return isPromptOptionsScreenShow
                || isRequestPermissionScreenShown
                || isOnBoardingShowing
                || isForegroundBusy;
    }

    /*
     * access plugin manager and check if foreground not busy
     *
     * */
    @WorkerThread
    public static boolean isForegroundNotBusy() {
        return !(SettingsManager.getInstance().isPromptOptionsScreenShown()
                || SettingsManager.getInstance().isRequestPermissionScreenShown()
                || SettingsManager.getInstance().isOnboardingShowing()
                || PluginsManager.foregroundBusy());
    }


    public static void setFeatureState(String feature, Feature.State state) {
        InstabugFeaturesManager.getInstance().setFeatureState(feature, state);
    }

    public static int getSessionCount() {
        return SettingsManager.getInstance().getSessionsCount();
    }

    public static long getFirstRunAt() {
        return SettingsManager.getInstance().getFirstRunAt().getTime();
    }

    public static Plugin getXPlugin(Class clazz) {
        return PluginsManager.getXPlugin(clazz);
    }

    public static boolean isAppOnForeground() {
        return SettingsManager.getInstance().isAppOnForeground();
    }

    @Nullable
    public static Activity getTargetActivity() {
        return InstabugInternalTrackingDelegate.getInstance().getTargetActivity();
    }

    @Nullable
    public static Object getLastSeenView() {
        return InstabugInternalTrackingDelegate.getInstance().getLastSeenView();
    }

    public static boolean isFeatureEnabled(@IBGFeature.Companion.IBGFeature String feature) {
        return InstabugFeaturesManager.getInstance().getFeatureState(feature) == Feature.State.ENABLED;
    }

    public static void setPushNotificationState(Feature.State state) {
        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.PUSH_NOTIFICATION, state);
    }

    public static void setBugReportingState(Feature.State state) {
        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.BUG_REPORTING, state);

        if (state == Feature.State.DISABLED) {
            AutoScreenRecordingEventBus.getInstance().post(ScreenRecordingServiceAction.CustomeActions.STOP_DELETE);
        } else {
            InternalAutoScreenRecorderHelper.getInstance().start();
        }
    }

    public static void setMessagingState(Feature.State state) {
        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.IN_APP_MESSAGING, state);
    }

    public static void setRepliesState(Feature.State state) {
        InstabugFeaturesManager.getInstance().setFeatureState(IBGFeature.REPLIES, state);
    }


    public static boolean encryptBeforeMarshmallow(@NonNull String filePath) {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
            return encrypt(filePath);
        }
        return true;
    }

    public static boolean encrypt(@NonNull String filePath) {
        try {
            return FileUtils.encryptFile(filePath);
        } catch (Exception | UnsatisfiedLinkError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't Encrypt attachment", e);
        }
        return false;
    }

    public static boolean decrypt(@NonNull String filePath) {
        try {
            return FileUtils.decryptFile(filePath);
        } catch (Exception | UnsatisfiedLinkError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't Decrypt attachment", e);
        }
        return false;
    }

    public static ProcessedBytes decryptOnTheFly(String filePath) {
        try {
            if (FileUtils.isEncryptedFile(filePath)) {
                return FileUtils.decryptOnTheFly(filePath);
            } else {
                // File is not encrypted
                File file = new File(filePath);
                byte[] fileBytes = new byte[(int) file.length()];
                FileUtils.read(file, fileBytes);
                return new ProcessedBytes(fileBytes, true);
            }
        } catch (Exception | UnsatisfiedLinkError e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Can't Decrypt attachment", e);
        }
        return new ProcessedBytes(new byte[0], false);
    }

    public static void setAutoScreenRecordingEnabled(boolean autoScreenRecordingEnabled) {
        SettingsManager.getInstance().setAutoScreenRecordingEnabled(autoScreenRecordingEnabled);
    }

    public static boolean isAutoScreenRecordingEnabled() {
        return SettingsManager.getInstance().isAutoScreenRecordingEnabled();
    }

    public static <T extends Plugin> void setPluginState(Class<T> clazz, @Plugin.PluginState int state) {
        Plugin plugin = InstabugCore.getXPlugin(clazz);
        if (plugin != null) {
            plugin.setState(state);
        }
    }

    public static long getLastSeenTimestamp() {
        return SettingsManager.getInstance().getLastSeenTimestamp();
    }

    public static void setLastSeenTimestamp(long currentTimeMillis) {
        SettingsManager.getInstance().setLastSeenTimestamp(currentTimeMillis);
    }

    public static void setInitialScreenShotAllowed(boolean initialScreenShotAllowed) {
        SettingsManager.setInitialScreenShotAllowed(initialScreenShotAllowed);
    }

    public static void isInitialScreenShotAllowed() {
        SettingsManager.isInitialScreenShotAllowed();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static HashMap<String, String> retrieveAllSDKAttributes() {
        return UserAttributeCacheManager.retrieveAllSDKAttributes();
    }

    @NonNull
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static ServiceLocator getApplicationScopeServiceLocator() {
        return ApplicationServiceLocator.get();
    }

    /**
     * This method using internal SDK Orchestrator to do some work on the bg-thread.
     */
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static void doOnBackground(@NonNull final Runnable runnable) {
        if (runnable == null) return;

        ActionsOrchestrator.obtainOrchestrator()
                .addWorkerThreadAction(new Action() {
                    @Override
                    public void run() throws Exception {
                        runnable.run();
                    }
                }).orchestrate();
    }

    public static boolean isFirstRunAfterEncryptorUpdate() {
        return SettingsManager.getInstance().isFirstRunAfterEncryptorUpdate();
    }

    public static void setFirstRunAfterEncryptorUpdate(boolean isFirstRun) {
        SettingsManager.getInstance().setFirstRunAfterEncryptorUpdate(isFirstRun);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static void setPushNotificationToken(String token) {
        SettingsManager.setPushNotificationToken(token);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static String getPushNotificationToken() {
        return SettingsManager.getPushNotificationToken();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static void setPushNotificationTokenSent(boolean isSent) {
        SettingsManager.setPushNotificationTokenSent(isSent);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static boolean isPushNotificationTokenSent() {
        return SettingsManager.isPushNotificationTokenSent();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    public static boolean isUsersPageEnabled() {
        return SettingsManager.getInstance().isUsersPageEnabled();
    }

    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static Session getRunningSession() {
        Session v3Session = IBGSessionServiceLocator.getStateManager().getCurrentSession();
        if (v3Session != null) return v3Session;

        return SessionManager.getInstance().getCurrentSession();
    }

    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static String getLatestV3SessionId() {
        return IBGSessionManager.INSTANCE.getLatestSessionId();
    }

    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static IBGInMemorySession getRunningV3Session() {
        return IBGSessionManager.INSTANCE.getCurrentSession();
    }

    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static IBGInMemorySession getLatestV3Session() {
        return IBGSessionManager.INSTANCE.getLatestSession();
    }

    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static String getLatestSessionReplayId() {
        final IBGInMemorySession session = getLatestV3Session();
        if (session != null && Instabug.getAppToken() != null) {
            return IBGSessionMapper.INSTANCE.getCompositeSessionId(session, Instabug.getAppToken());
        }
        return null;
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static IBGSessionCrashesConfigurations getV3SessionCrashesConfigurations() {
        return IBGSessionServiceLocator.getSessionCrashesConfigurations();
    }

    /**
     * For internal SDK Use only.
     * Used to totally disable the SDK and all features in the current app instance.
     * <p>
     * Only used now for NDK .so file not found issue.
     */
    @RestrictTo(RestrictTo.Scope.LIBRARY)
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public static void setTemporaryDisabled() {
        InstabugSDKLogger.d(Constants.LOG_TAG, "setTemporaryDisabled disable the SDK internally");
        InstabugFeaturesManager.getInstance().setTemporaryDisabled();
        Method disableInternalMethod = ReflectionUtils.getMethod(Instabug.class, "disableInternal");
        try {
            if (disableInternalMethod != null) {
                disableInternalMethod.invoke(null);
            }
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static String getCurrentView() {
        int currentPlatform = SettingsManager.getInstance().getCurrentPlatform();
        if (currentPlatform == Platform.RN
                || currentPlatform == Platform.FLUTTER
                || currentPlatform == Platform.UNITY) {
            return CurrentViewProvider.getInstance().getCrossPlatformCurrentView();
        }
        return CoreServiceLocator.getActiveScreenEvaluator()
                .getActiveScreenName(CoreServiceLocator.getScreensRoot());
    }

    @Nullable
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static Report.OnReportCreatedListener getOnReportCreatedListener() {
        return SettingsManager.getInstance().getOnReportCreatedListener();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    @Platform
    public static int getPlatform() {
        return SettingsManager.getInstance().getCurrentPlatform();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static boolean isAPMEnabled() {
        if(isApmPluginNotLoaded) return false;
        try {
            Plugin plugin = PluginsManager.getXPlugin(Class.forName("com.instabug.apm.APMPlugin"));
            if (plugin != null) {
                return plugin.isFeatureEnabled();
            }
        } catch (ClassNotFoundException e) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "APM Plugin Not Loaded");
            isApmPluginNotLoaded = true;
        }
        return false;
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static boolean isCrashReportingEnabled() {
        if(isCrashPluginNotLoaded) return false;
        try {
            Plugin plugin = PluginsManager.getXPlugin(Class.forName("com.instabug.crash.CrashPlugin"));
            if (plugin != null) {
                return plugin.isFeatureEnabled();
            }
        } catch (ClassNotFoundException e) {
            InstabugSDKLogger.v(Constants.LOG_TAG, "Crash Plugin Not Loaded");
            isCrashPluginNotLoaded = true;
        }
        return false;
    }

    public static Locale getLocale(@Nullable Context context) {
        return SettingsManager.getInstance().getInstabugLocale(context);
    }

    public static int getEncryptorVersion() {
        PersistableSettings persistableSettings = PersistableSettings.getInstance();
        if (persistableSettings != null) {
            return persistableSettings.getEncryptorVersion();
        }
        return EncryptorVersions.ENCRYPTOR_V1;
    }

    public static void saveEncryptorVersion(int version) {
        PersistableSettings persistableSettings = PersistableSettings.getInstance();
        if (persistableSettings != null) {
            persistableSettings.saveEncryptorVersion(version);
        }
    }

    public static Feature.State getEncryptionState() {
        return SettingsManager.getInstance().getFeatureState(IBGFeature.ENCRYPTION, false);
    }

    public static long getFirstSeen() {
        return SettingsManager.getInstance().getFirstSeen();
    }

    public static void reportError(Throwable error, String message) {
        IBGDiagnostics.reportNonFatal(error, message);
    }

    public static boolean isDbEncryptionEnabled() {
        return InstabugFeaturesManager.getInstance().getDbEncryptionState() == Feature.State.ENABLED;
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static boolean canPrintLog(@LogLevel int logLevel) {
        int permittedLevel = SettingsManager.getInstance().getLogLevel();
        return permittedLevel != LogLevel.NONE && logLevel <= permittedLevel;
    }

    @Nullable
    @WorkerThread
    public static List<String> getExperiments(float percentage) {
        ExperimentsManager experimentsManager =
                com.instabug.library.experiments.di.ServiceLocator.getExperimentsManager();
        if (experimentsManager != null) {
            return experimentsManager.getExperiments(percentage);
        }
        return null;
    }

    public static boolean isLastSDKStateEnabled(@NonNull Context context) {
        return new InstabugMinimalPersistableSettings(context).isLastSDKStateEnabled();
    }

    public static void cleanVisualUserSteps() {
        CoreServiceLocator.getReproStepsProxy().clean();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static boolean isV3SessionEnabled() {
        return IBGSessionServiceLocator.getSessionConfigurations().isV3SessionEnabled();
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static void notifyV3SessionDataReadiness(SessionBatchingFilter batchingFilter) {
        IBGSessionServiceLocator.getSyncManger().sync(batchingFilter);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static void forceV3SessionsEvaluation(SessionBatchingFilter batchingFilter) {
        IBGSessionBatchEvaluatorImpl.INSTANCE.evaluate(batchingFilter);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static ComposeLifeCycleMonitor getComposeLifeCycleMonitor() {
        return CoreServiceLocator.getComposeLifeCycleMonitor();
    }

    public static void handlePbiFooter(@Nullable View view) {
        PbiFooterUtils.handlePbiFooter(view);
    }

    public static void handlePbiFooterThemeColor(@Nullable View view, @ColorRes int lightColor, @ColorRes int darkColor) {
        PbiFooterUtils.handleThemeColor(view, lightColor, darkColor);
    }

    public static void setPbiFooterThemeColor(@Nullable View view, @ColorInt int targetColor) {
        PbiFooterUtils.setThemeColor(view, targetColor);
    }

    public static boolean isFullScreen() {
        return SettingsManager.getInstance().isFullscreen();
    }

    public static void setTouchedViewExtractorExtension(TouchedViewExtractor viewExtractor) {
        CoreServiceLocator.setTouchedViewExtractorExtension(viewExtractor);
    }

    @Nullable
    public static Uri saveDrawable(Drawable drawable) {
        long currentTime = System.currentTimeMillis();
        try {
            return BitmapUtils.saveDrawableBitmap(drawable, currentTime);
        } catch (Throwable e) {
            IBGDiagnostics.reportNonFatalAndLog(e, "something went wrong while saving drawable into file", Constants.LOG_TAG);
            return null;
        }
    }

    @NonNull
    public static String composeProminentLabelFor(@Nullable String text, boolean withIcon) {
        return TouchedViewsProcessor.getInstance().composeProminentLabelForButtonWith(text, withIcon);
    }

    @Nullable
    public static String composeProminentLabelForViewGroup(@NonNull ViewGroup viewGroup, @NonNull String template) {
        return new GroupViewTouchedViewProcessor(viewGroup, template).getCapturedLabel();
    }


    public static void setVideoRecordingButtonPosition(InstabugVideoRecordingButtonPosition
                                                               videoRecordingButtonPosition) {
        SettingsManager.getInstance().setVideoRecordingButtonPosition(videoRecordingButtonPosition);
    }

    public static InstabugInvocationEvent[] getInvocationEvents() {

        InstabugInvocationEvent[] invocationEvents = SettingsManager.getInstance().getInstabugInvocationEvents();
        InstabugInvocationEvent[] invocationEventsClone;
        if (invocationEvents != null) {
            invocationEventsClone = invocationEvents.clone();
            PerSessionSettings.getInstance().clearInvocationEvents();
        } else {
            invocationEventsClone = new InstabugInvocationEvent[]{InstabugInvocationEvent.SHAKE};
        }
        return invocationEventsClone;
    }

    @NonNull
    public static ArrayList<PluginPromptOption> getPluginsPromptOptions() {
        return PluginsManager.getPluginsPromptOptions();
    }

}