package com.instabug.bug;

import static com.instabug.bug.Constants.AUTO_SCREEN_RECORDING_ENABLED_SERVICE_NOT_AVAILABLE_ERROR;

import android.content.Context;
import android.os.Build;
import android.text.Html;
import android.text.SpannableStringBuilder;
import android.text.Spanned;

import androidx.annotation.Nullable;

import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.bug.invocation.Option;
import com.instabug.bug.proactivereporting.configs.ProactiveReportingConfigProvider;
import com.instabug.bug.settings.AttachmentsTypesParams;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.userConsent.ActionType;
import com.instabug.bug.utils.RegexUtils;
import com.instabug.chat.ChatsDelegate;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.video.InternalAutoScreenRecorderHelper;
import com.instabug.library.internal.video.MediaProjectionHelper;
import com.instabug.library.invocation.InvocationMode;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.Locale;

public class BugReportingWrapper {
    public static final int DISCLAIMER_TEXT_MAX_CHARS = 100;

    public static void setReportTypes(@BugReporting.ReportType int... promptOptions) {
        boolean bugEnabled = false, feedbackEnabled = false, isAskQuestionEnabled = false;
        for (int promptOption : promptOptions) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "setReportTypes: " + promptOption);
            switch (promptOption) {
                case BugReporting.ReportType.BUG:
                    bugEnabled = true;
                    break;
                case BugReporting.ReportType.FEEDBACK:
                    feedbackEnabled = true;
                    break;
                case BugReporting.ReportType.QUESTION:
                    isAskQuestionEnabled = true;
                    break;
                default:
                    // do nothing
            }
        }
        BugSettings.getInstance().setReportTypeStatus(Constants.ReportType.BUG, bugEnabled);
        BugSettings.getInstance().setReportTypeStatus(Constants.ReportType.FEEDBACK, feedbackEnabled);
        BugSettings.getInstance().setReportTypeStatus(Constants.ReportType.ASK_QUESTION, isAskQuestionEnabled);

        InvocationManager.getInstance().notifyInvocationOptionChanged();

    }


    public static void setOptions(int[] options) {

        for (int option : options) {
            InstabugSDKLogger.d(Constants.LOG_TAG, "setOptions: " + option);
            switch (option) {
                case Option.COMMENT_FIELD_REQUIRED:
                    BugSettings.getInstance().setCommentFieldRequired(true);
                    break;
                case Option.EMAIL_FIELD_HIDDEN:
                    BugSettings.getInstance().setEmailFieldVisibility(false);
                    BugSettings.getInstance().setEmailFieldRequired(false);
                    break;
                case Option.EMAIL_FIELD_OPTIONAL:
                    BugSettings.getInstance().setEmailFieldVisibility(true);
                    BugSettings.getInstance().setEmailFieldRequired(false);
                    break;
                case Option.DISABLE_POST_SENDING_DIALOG:
                    BugSettings.getInstance().setSuccessDialogEnabled(false);
                    break;
                default:
                    break;
            }
        }
    }

    public static void show(@BugReporting.ReportType int type) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[BugReportingWrapper#show], type: " + type);
        boolean isForegroundBusy = InstabugCore.isForegroundBusy();
        InstabugSDKLogger.d(Constants.LOG_TAG, "[BugReportingWrapper#show] isForegroundBusy: " + isForegroundBusy);
        if (isForegroundBusy) return;
        boolean isFeatureEnabled = InstabugCore.isFeatureEnabled(IBGFeature.BUG_REPORTING);
        InstabugSDKLogger.d(Constants.LOG_TAG, "[BugReportingWrapper#show] isFeatureEnabled: " + isFeatureEnabled);
        if (isFeatureEnabled) {
            switch (type) {
                case BugReporting.ReportType.BUG:
                    InvocationManager.getInstance().invoke(InvocationMode.NEW_BUG);
                    break;
                case BugReporting.ReportType.FEEDBACK:
                    InvocationManager.getInstance().invoke(InvocationMode.NEW_FEEDBACK);
                    break;
                case BugReporting.ReportType.QUESTION:
                    InvocationManager.getInstance().invoke(InvocationMode.ASK_QUESTION);
                    break;
                default:
                    // do nothing
                    break;
            }
        }
    }

    public static void show(int type, int[] invocationOptions) {
        setOptions(invocationOptions);
        show(type);
    }

    public static void setState(Feature.State state) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "setState: " + state);
        BugSettings.getInstance().setBugReportingState(state);
        InstabugCore.setBugReportingState(state);
        ServiceLocator.getReproProxy().evaluate(ServiceLocator.getConfigurationsProvider());
        InvocationManager.getInstance().notifyInvocationOptionChanged();
    }

    public static void setAttachmentTypesEnabled(boolean initialScreenshot,
                                                 boolean extraScreenshot,
                                                 boolean imageFromGallery,
                                                 boolean screenRecording) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "setAttachementTypes: initialScreenshot: " + initialScreenshot
                + " extraScreenshot: " + extraScreenshot + " imageFromGallery: " + imageFromGallery +
                "screenRecording: " + screenRecording);

        AttachmentsTypesParams attachmentTypesParams = new AttachmentsTypesParams(initialScreenshot,
                extraScreenshot, imageFromGallery, screenRecording);
        BugSettings.getInstance().setAttachmentsTypesParams(attachmentTypesParams);

        ChatsDelegate.setAttachmentTypesEnabled(extraScreenshot, imageFromGallery, screenRecording);
    }

    public static void setAutoScreenRecordingEnabled(boolean autoScreenRecordingEnabled) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "setAutoScreenRecordingEnabled: " + autoScreenRecordingEnabled);

        Context context = Instabug.getApplicationContext();
        if (context != null && autoScreenRecordingEnabled && !MediaProjectionHelper.INSTANCE.isMediaProjectionServiceAvailable(context)) {
            InstabugSDKLogger.e(Constants.LOG_TAG, AUTO_SCREEN_RECORDING_ENABLED_SERVICE_NOT_AVAILABLE_ERROR);
            return;
        }

        if (autoScreenRecordingEnabled && InstabugCore.isAutoScreenRecordingEnabled()) {
            return;
        }
        InstabugCore.setAutoScreenRecordingEnabled(autoScreenRecordingEnabled);
        if (autoScreenRecordingEnabled)
            InternalAutoScreenRecorderHelper.getInstance().

                    start();

    }

    public static void setDisclaimerText(String disclaimerText) {
        if (disclaimerText == null || disclaimerText.equals("")) {
            return;
        }
        String replacement = String.format("<font color=\"#%06X\"><a href=\"$2\">$1</a></font>",
                (0xFFFFFF & SettingsManager.getInstance().getPrimaryColor()));
        // 1. Replace the valid markdowns with the HTML Anchor tag <a>
        String disclaimerTextWithValidLinks =
                RegexUtils.reformatMarkdownText(disclaimerText, replacement);
        // 2. Span the text to replace the anchor tags with text to count the character and trim if necessary
        Spanned spanned = getSpannedFromString(disclaimerTextWithValidLinks);
        // 3. Check if the text needs to be trimmed after spanning
        if (spanned.length() > DISCLAIMER_TEXT_MAX_CHARS) {
            InstabugSDKLogger.w(Constants.LOG_TAG, String.format(Locale.getDefault(),
                    "The maximum limit of Disclaimer text is reached. Please note that maximum characters count is %d",
                    DISCLAIMER_TEXT_MAX_CHARS));
            spanned = (Spanned) spanned.subSequence(0, DISCLAIMER_TEXT_MAX_CHARS);
            if (spanned instanceof SpannableStringBuilder) {
                ((SpannableStringBuilder) spanned).append("...");
            }
        }
        BugSettings.getInstance().setDisclaimerText(spanned);
    }

    private static Spanned getSpannedFromString(String htmlDisclaimer) {
        Spanned spanned;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            spanned = Html.fromHtml(htmlDisclaimer, Html.FROM_HTML_MODE_LEGACY);
        } else {
            spanned = Html.fromHtml(htmlDisclaimer);
        }
        return spanned;
    }

    public static void setCommentMinimumCharacterCount(int limit, final @BugReporting.IBGBugReportingType int... reportTypes) {
        for (@BugReporting.IBGBugReportingType int type : reportTypes) {
            setReportTypeMinCharLimit(type, limit);
        }
    }

    private static void setReportTypeMinCharLimit(@BugReporting.IBGBugReportingType int reportType, int limit) {
        switch (reportType) {
            case BugReporting.IBGBugReportingType.BUG:
                BugSettings.getInstance().setReportTypeCommentMinCharLimit(Constants.ReportType.BUG, limit);
                break;
            case BugReporting.IBGBugReportingType.FEEDBACK:
                BugSettings.getInstance().setReportTypeCommentMinCharLimit(Constants.ReportType.FEEDBACK, limit);
                break;
            case BugReporting.IBGBugReportingType.QUESTION:
                BugSettings.getInstance().setReportTypeCommentMinCharLimit(Constants.ReportType.ASK_QUESTION, limit);
                break;
            case BugReporting.IBGBugReportingType.FRUSTRATING_EXPERIENCE:
                BugSettings.getInstance().setReportTypeCommentMinCharLimit(Constants.ReportType.FRUSTRATING_EXPERIENCE, limit);
                break;
            default:
                // do nothing
                break;
        }
    }

    public static void addUserConsent(
            @Nullable String key,
            @Nullable String description,
            boolean isMandatory,
            boolean isChecked,
            @Nullable @ActionType String consentType
    ) {
        ServiceLocator.getUserConsentsManager().addUserConsent(key, description, isMandatory, isChecked, consentType);
    }

    public static void setProactiveReportingConfigurations(ProactiveReportingConfigs proactiveReportingConfigurations) {
        ProactiveReportingConfigProvider proactiveReportingConfigProvider = ServiceLocator.getProactiveReportingConfigProvider();
        if(!proactiveReportingConfigProvider.getProactiveReportingBEAvailability() && proactiveReportingConfigurations.isEnabled()) {
            InstabugSDKLogger.e(Constants.LOG_TAG, Constants.PROACTIVE_REPORTING_ENABLE_BE_DISABLED);
        } else {
            proactiveReportingConfigProvider.setProactiveReportingApiAvailability(proactiveReportingConfigurations.isEnabled());
            proactiveReportingConfigProvider.setModalsGapApiValue(proactiveReportingConfigurations.getModalsGap());
            proactiveReportingConfigProvider.setDetectionGapApiValue(proactiveReportingConfigurations.getDetectionGap());
        }
    }
}
