package com.instabug.bug.view.reporting;

import static com.instabug.bug.Constants.ReportType.ASK_QUESTION;
import static com.instabug.bug.Constants.ReportType.BUG;
import static com.instabug.bug.Constants.ReportType.FEEDBACK;
import static com.instabug.bug.Constants.ReportType.FRUSTRATING_EXPERIENCE;
import static com.instabug.bug.Constants.ReportType.NOT_AVAILABLE;
import static com.instabug.bug.Constants.ReportingProcess.PROCESS_ASK_QUESTION;
import static com.instabug.bug.Constants.ReportingProcess.PROCESS_BUG;
import static com.instabug.bug.Constants.ReportingProcess.PROCESS_FEEDBACK;
import static com.instabug.bug.Constants.ReportingProcess.PROCESS_FRUSTRATING_EXPERIENCE;
import static com.instabug.bug.Constants.ReportingProcess.PROCESS_HANGING_BUG;

import android.net.Uri;

import com.instabug.bug.BugPlugin;
import com.instabug.bug.Constants;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.view.BugReportingContract;
import com.instabug.library.InstabugState;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.InstabugStateEventBus;
import com.instabug.library.core.eventbus.ScreenRecordingEventBus;
import com.instabug.library.core.plugin.Plugin;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.internal.video.ScreenRecordingEvent;
import com.instabug.library.model.Attachment;

import io.reactivexport.disposables.CompositeDisposable;
import io.reactivexport.functions.Consumer;


/**
 * Created by mohamedzakaria on 6/8/17.
 */

public class BaseReportingContainerPresenter extends BasePresenter<BugReportingContract.View> implements BugReportingContract.Presenter {
    private final CompositeDisposable compositeDisposable;

    public BaseReportingContainerPresenter(final BugReportingContract.View view) {
        super(view);
        compositeDisposable = new CompositeDisposable();
        compositeDisposable.add(InstabugStateEventBus.getInstance().getEventObservable().subscribe(new Consumer<InstabugState>() {
            @Override
            public void accept(InstabugState instabugState) throws Exception {
                if (instabugState == InstabugState.DISABLED) {
                    BugPlugin plugin = (BugPlugin) InstabugCore.getXPlugin(BugPlugin.class);
                    if (plugin != null) {
                        plugin.setState(Plugin.STATE_BACKGROUND);
                    }
                    ScreenRecordingEventBus.getInstance().post(new ScreenRecordingEvent(ScreenRecordingEvent.RECORDING_CANCELED, null));
                    view.dismissReport();
                }
            }
        }));
    }

    @Override
    public void handleBugProcess(@Constants.ReportingProcess int currentProcess) {

        if (view != null) {
            BugReportingContract.View viewRef = view.get();
            if (viewRef != null) {
                switch (currentProcess) {
                    case PROCESS_FEEDBACK:
                        viewRef.startFeedbackSender();
                        break;

                    case PROCESS_BUG:
                        viewRef.startBugReporter();
                        break;

                    case PROCESS_HANGING_BUG:
                        viewRef.startWithHangingBug();
                        break;

                    case PROCESS_ASK_QUESTION:
                        prepareAskAQuestion();
                        break;

                    case PROCESS_FRUSTRATING_EXPERIENCE:
                        prepareFrustratingExperience();
                        break;

                    default:
                        break;

                }
            }
        }
    }

    private void prepareAskAQuestion() {

        if (LiveBugManager.getInstance().getBug() == null)
            return;
        LiveBugManager.getInstance().getBug().setType(Constants.ReportType.ASK_QUESTION);

        String initialScreenshot = LiveBugManager.getInstance().getBug().getInitialScreenshotPath();

        if (!LiveBugManager.getInstance().getBug().hasMainScreenshot()
                && initialScreenshot != null) {
            LiveBugManager.getInstance().getBug().addAttachment(Uri.parse(initialScreenshot),
                    Attachment.Type.MAIN_SCREENSHOT);
        }

        if (view != null) {
            BugReportingContract.View view = this.view.get();
            if (view != null) {
                view.startAskQuestion();
            }
        }
        handlePoweredByFooter();
    }

    private void prepareFrustratingExperience() {
        if (LiveBugManager.getInstance().getBug() == null)
            return;
        LiveBugManager.getInstance().getBug().setType(Constants.ReportType.FRUSTRATING_EXPERIENCE);

        String initialScreenshot = LiveBugManager.getInstance().getBug().getInitialScreenshotPath();

        if (!LiveBugManager.getInstance().getBug().hasMainScreenshot()
                && initialScreenshot != null) {
            LiveBugManager.getInstance().getBug().addAttachment(Uri.parse(initialScreenshot),
                    Attachment.Type.MAIN_SCREENSHOT);
        }

        if (view != null) {
            BugReportingContract.View view = this.view.get();
            if (view != null) {
                view.startFrustratingExperience();
            }
        }
        handlePoweredByFooter();
    }

    @Override
    public void handlePoweredByFooter() {
        if (view != null) {
            BugReportingContract.View viewRef = view.get();
            if (viewRef != null) {
                viewRef.handlePoweredByFooter();
            }
        }
    }


    @Override
    public void startNavigation() {
        if (view != null) {
            BugReportingContract.View view = this.view.get();
            if (view != null && LiveBugManager.getInstance().getBug() != null) {
                String type = LiveBugManager.getInstance().getBug().getType();
                switch (type) {
                    case BUG:
                        view.navigateToBugReporting();
                        break;
                    case FEEDBACK:
                        view.navigateToFeedback();
                        break;
                    case ASK_QUESTION:
                        view.startAskQuestion();
                        break;
                    case FRUSTRATING_EXPERIENCE:
                        view.startFrustratingExperience();
                        break;
                    case NOT_AVAILABLE:
                        break;

                }

            }
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (compositeDisposable != null && !compositeDisposable.isDisposed()) {
            compositeDisposable.dispose();
        }
    }
}
