// Copyright 2016 Instabug, Inc.
package com.instabug.bug.invocation.util;

import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;

import com.instabug.bug.invocation.InvocationSettings;
import com.instabug.library.Constants;
import com.instabug.library.util.InstabugSDKLogger;

import io.reactivexport.annotations.Nullable;

/**
 * Created on 7/26/2016.
 *
 * @author Hossam
 */
public class ShakeDetector implements SensorEventListener {

    private static final int SENSOR_OUTPUT_THROTTLING_TIME_THRESHOLD_IN_MILLIS = 400;
    private static final int INVOCATION_TRIGGER_THROTTLING_THRESHOLD_IN_MILLIS = 2000;
    private final SensorManager sensorManager;
    @Nullable
    private final Sensor accelerometer;
    private long coordinatesLastUpdateTimeInMillis = 0;
    private long lastDetectionTriggerTimeInMillis = 0;
    private float lastX, lastY, lastZ;
    private OnShakeDetectedListener onShakeDetectedListener;
    private int accelerationThreshold = InvocationSettings.SHAKE_DEFAULT_THRESHOLD;

    public ShakeDetector(Context context, OnShakeDetectedListener onShakeDetectedListener) {
        this.sensorManager = (SensorManager) context.getSystemService(Context.SENSOR_SERVICE);
        this.accelerometer  = sensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        this.onShakeDetectedListener = onShakeDetectedListener;
    }

    @Override
    public void onSensorChanged(SensorEvent sensorEvent) {
        //sensorEvent have sensor property called sensor 
        Sensor sensor = sensorEvent.sensor;
        //we should make sure that the sensor is the accelerometer sensor 
        if (sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
            //now we can get the x,y and z values you can go to docs to learn more about how these values are stored 
            float x = sensorEvent.values[0];
            float y = sensorEvent.values[1];
            float z = sensorEvent.values[2];
            //get the current timestamp 
            long currentTime = System.currentTimeMillis();
            //compare the current timestamp and last update time stamp if greater than 300 milliseconds 
            //calculate the speed by the following equation 
            long coordinatesUpdateDiffTime = currentTime - coordinatesLastUpdateTimeInMillis;
            boolean sensorOutputTimeThresholdHadPassed =
                    coordinatesUpdateDiffTime > SENSOR_OUTPUT_THROTTLING_TIME_THRESHOLD_IN_MILLIS;
            if (sensorOutputTimeThresholdHadPassed) {
                //calculate speed
                float speed = Math.abs(x + y + z - lastX - lastY - lastZ) / coordinatesUpdateDiffTime * 10000;
                //invoke onShake event if the speed value greater than SHAKE_THRESHOLD
                long triggerDiffTime = currentTime - lastDetectionTriggerTimeInMillis;
                boolean triggerTimeThresholdHasPassed
                        = triggerDiffTime > INVOCATION_TRIGGER_THROTTLING_THRESHOLD_IN_MILLIS;
                boolean speedIsGreaterThanAccelerationThreshold = speed > accelerationThreshold;
                if (speedIsGreaterThanAccelerationThreshold && triggerTimeThresholdHasPassed) {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "shake detected, invoking shakeDetectedListener");
                    onShakeDetectedListener.onShakeDetected();
                }
                coordinatesLastUpdateTimeInMillis = currentTime;
                lastX = x;
                lastY = y;
                lastZ = z;
            }
        }
    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int i) {

    }

    public void setListener(OnShakeDetectedListener onShakeDetectedListener) {
        this.onShakeDetectedListener = onShakeDetectedListener;
    }

    public void start() {
        lastDetectionTriggerTimeInMillis = System.currentTimeMillis();
        if (accelerometer != null){
            sensorManager.registerListener(this, accelerometer, SensorManager.SENSOR_DELAY_NORMAL);
        }else {
            InstabugSDKLogger.e(Constants.LOG_TAG," accelerometer == null");
        }

    }

    public void stop() {
        sensorManager.unregisterListener(this);
    }

    public void setAccelerationThreshold(int accelerationThreshold) {
        this.accelerationThreshold = accelerationThreshold;
    }

    /**
     * Created by Hossam on 7/26/2016.
     * Interface definition for a callback to be invoked when the device has been shaken.
     */
    public interface OnShakeDetectedListener {
        /**
         * Called when the device has been shaken.
         */
        void onShakeDetected();
    }
}
