package com.instabug.chat.ui.base;

import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.ViewStub;
import android.widget.ImageButton;
import android.widget.TextView;

import com.instabug.bug.R;
import com.instabug.chat.util.ViewUtilsKt;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.BaseContract;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.model.IBGTheme;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.SystemServiceUtils;
import com.instabug.library.util.ThemeApplier;

import androidx.annotation.CallSuper;
import androidx.annotation.LayoutRes;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentActivity;

import kotlin.Unit;

/**
 * Created by tarek on 9/25/16.
 */
public abstract class ToolbarFragment<P extends BaseContract.Presenter> extends InstabugBaseFragment<P> {

    @Nullable
    protected ImageButton toolbarImageButtonDone, toolbarImageButtonClose;
    @Nullable
    private View instabugToolbar;
    @Nullable
    private TextView titleTextView;

    @CallSuper
    @Override
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        initToolbarViews();

        ViewStub content = findViewById(R.id.instabug_content);
        if (content != null) {
            content.setLayoutResource(getContentLayout());
            content.setOnInflateListener((stub, inflated) -> fitSystemWindow(inflated));
            content.inflate();
        }

        initContentViews(rootView, savedInstanceState);

        setTitle(getTitle());
        ThemeApplier.setScreenBackground(rootView, getIBGTheme());
    }

    private void fitSystemWindow(View content) {

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            View statusBarFiller = findViewById(R.id.status_bar_filler);
            if (statusBarFiller != null && content != null) {
                ViewUtilsKt.addSystemWindowInsetToPadding(content, true, false, true, true, insets -> {
                    statusBarFiller.setVisibility(View.VISIBLE);
                    statusBarFiller.getLayoutParams().height = Math.max(insets.top, 0);
                    statusBarFiller.setBackgroundColor(SettingsManager.getInstance().getPrimaryColor());
                    return Unit.INSTANCE;
                });
            }
        }

    }

    private void initToolbarViews() {
        toolbarImageButtonDone = findViewById(R.id.instabug_btn_toolbar_right);
        instabugToolbar = findViewById(R.id.instabug_toolbar);
        titleTextView = findViewById(R.id.instabug_fragment_title);

        if (toolbarImageButtonDone != null) {
            toolbarImageButtonDone.setOnClickListener(v -> {
                FragmentActivity activity = getActivity();
                if (activity != null) {
                    SystemServiceUtils.hideInputMethod(activity);
                }
                onDoneButtonClicked();
            });
        }
        toolbarImageButtonClose = findViewById(R.id.instabug_btn_toolbar_left);
        if (toolbarImageButtonClose != null) {
            toolbarImageButtonClose.setOnClickListener(v -> onCloseButtonClicked());
        }
        styleToolbar();
        makeToolBarFitSystemWidow(instabugToolbar);
    }

    private void makeToolBarFitSystemWidow(View instabugToolbar) {
        if (instabugToolbar != null && Build.VERSION.SDK_INT > Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            ViewUtilsKt.addSystemWindowInsetToPadding(instabugToolbar, true, false, true, false);
        }
    }

    private void styleToolbar() {
        if (instabugToolbar != null)
            instabugToolbar.setBackgroundColor(InstabugCore.getPrimaryColor());
        ThemeApplier.applyTitleStyle(titleTextView, getIBGTheme());
    }

    @Override
    protected int getLayout() {
        return R.layout.instabug_fragment_toolbar;
    }

    /**
     * @return content layout resource id.
     */
    protected abstract
    @LayoutRes
    int getContentLayout();

    /**
     * Init fragment content views
     *
     * @param rootView           fragment root view.
     * @param savedInstanceState saved instance state
     */
    protected abstract void initContentViews(View rootView, @Nullable Bundle savedInstanceState);

    /**
     * set fragment title
     *
     * @param title fragment title
     */
    protected void setTitle(String title) {
        if (rootView == null) {
            return;
        }
        TextView titleTextView = findViewById(R.id.instabug_fragment_title);
        if (titleTextView != null)
            titleTextView.setText(title);
    }

    /**
     * @return Fragment title
     */
    protected abstract String getTitle();

    protected abstract void onDoneButtonClicked();

    protected void onCloseButtonClicked() {
        FragmentActivity activity = getActivity();
        if (activity != null) {
            SystemServiceUtils.hideInputMethod(activity);
            activity.onBackPressed();
        } else Log.w("ToolbarFragment",
                "onCloseButtonClicked can't be executed due to null getActivity() reference");
    }

    @Nullable
    protected IBGTheme getIBGTheme() {
        return presenter != null && presenter.getTheme() != null ? presenter.getTheme() : null;
    }

}
