package com.instabug.chat.notification

import android.annotation.SuppressLint
import android.content.Context
import android.content.res.ColorStateList
import android.graphics.PorterDuff
import android.graphics.Typeface
import android.os.Build
import android.view.View
import android.widget.Button
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.annotation.VisibleForTesting
import androidx.core.content.res.ResourcesCompat
import com.instabug.bug.R
import com.instabug.chat.Constants
import com.instabug.chat.notification.InAppNotificationColors.CONTAINER_DARK_BACKGROUND_COLOR
import com.instabug.chat.notification.InAppNotificationColors.DISMISS_BUTTON_TEXT_COLOR
import com.instabug.chat.notification.InAppNotificationColors.SENDER_MESSAGE_DARK_TEXT_COLOR
import com.instabug.chat.notification.InAppNotificationColors.SENDER_MESSAGE_LIGHT_TEXT_COLOR
import com.instabug.chat.notification.InAppNotificationColors.SENDER_NAME_LIGHT_TEXT_COLOR
import com.instabug.chat.notification.InAppNotificationColors.WHITE_COLOR
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.IBGFeature
import com.instabug.library.Instabug
import com.instabug.library.InstabugColorTheme
import com.instabug.library.core.InstabugCore
import com.instabug.library.model.IBGTheme
import com.instabug.library.settings.SettingsManager
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.ThemeApplier

fun applyNotificationBarTheme(view: View?) {
    view?.run {
        getCustomFont(view.context).also { font ->
            styleReplyButton(font)
            styleDismissButton(font)
            styleSenderNameTextView(font)
            styleSenderMessageTextView(font)
            setLayoutBackground()
        }
    }
}

private fun View.styleReplyButton(customTypeface: Typeface?) =
    findViewById<Button>(R.id.replyButton)?.let {
        setReplyButtonStyle(customTypeface, it)
        setReplyButtonColors(it)
    }

private fun setReplyButtonStyle(customTypeface: Typeface?, replyButton: Button) =
    getIbgTheme()
        ?.run { setTextViewStyle(replyButton, ctaTextFont, ctaTextStyle, customTypeface) }
        ?: customTypeface?.let(replyButton::setTypeface)

private fun setReplyButtonColors(replyButton: Button) {
    setBackgroundTintList(replyButton, SettingsManager.getInstance().primaryColor)
    replyButton.setTextColor(WHITE_COLOR)
}

private fun View.styleDismissButton(customTypeface: Typeface?) =
    findViewById<Button>(R.id.dismissButton)?.let {
        setDismissButtonStyle(customTypeface, it)
        setDismissButtonColors(it)
    }

private fun setDismissButtonStyle(customTypeface: Typeface?, dismissButton: Button) =
    getIbgTheme()
        ?.run { setTextViewStyle(dismissButton, ctaTextFont, ctaTextStyle, customTypeface) }
        ?: customTypeface?.let(dismissButton::setTypeface)

private fun setDismissButtonColors(dismissButton: Button) {
    setBackgroundTintList(dismissButton, WHITE_COLOR)
    dismissButton.setTextColor(DISMISS_BUTTON_TEXT_COLOR)
}

private fun View.setLayoutBackground() {
    val ibgTheme = getIbgTheme()
    findViewById<RelativeLayout>(R.id.instabug_notification_layout)?.setBackgroundColor(
        if (ibgTheme != null && ibgTheme.backgroundColor != ThemeApplier.DEFAULT_COLOR) {
            ibgTheme.backgroundColor
        } else {
            getDefaultColor(WHITE_COLOR, CONTAINER_DARK_BACKGROUND_COLOR)
        }
    )
}

private fun View.styleSenderNameTextView(customTypeface: Typeface?) =
    findViewById<TextView>(R.id.senderNameTextView)?.let {
        setSenderNameStyle(customTypeface, it)
        setSenderNameTextColor(it)
    }

private fun setSenderNameStyle(customTypeface: Typeface?, senderNameTextView: TextView) =
    getIbgTheme()?.run {
        setTextViewStyle(
            senderNameTextView,
            primaryTextFont,
            primaryTextStyle,
            customTypeface
        )
    } ?: customTypeface?.let(senderNameTextView::setTypeface)

private fun setSenderNameTextColor(senderNameTextView: TextView) {
    val ibgTheme = getIbgTheme()
    val textColor = if (ibgTheme != null && ibgTheme.primaryTextColor != ThemeApplier.DEFAULT_COLOR) {
        ibgTheme.primaryTextColor
    } else {
        getDefaultColor(SENDER_NAME_LIGHT_TEXT_COLOR, WHITE_COLOR)
    }
    senderNameTextView.setTextColor(textColor)
}

private fun View.styleSenderMessageTextView(customTypeface: Typeface?) =
    findViewById<TextView>(R.id.senderMessageTextView)?.let {
        setSenderMessageStyle(it, customTypeface)
        setSenderMessageColor(it)
    }

private fun setSenderMessageStyle(
    senderMessageTextView: TextView,
    customTypeface: Typeface?
) = getIbgTheme()?.run {
    setTextViewStyle(
        senderMessageTextView,
        secondaryTextFont,
        secondaryTextStyle,
        customTypeface
    )
} ?: customTypeface?.let(senderMessageTextView::setTypeface)

private fun setSenderMessageColor(senderMessageTextView: TextView) {
    val ibgTheme = getIbgTheme()
    senderMessageTextView.setTextColor(
        if (ibgTheme != null && ibgTheme.secondaryTextColor != ThemeApplier.DEFAULT_COLOR) {
            ibgTheme.secondaryTextColor
        } else {
            getDefaultColor(SENDER_MESSAGE_LIGHT_TEXT_COLOR, SENDER_MESSAGE_DARK_TEXT_COLOR)
        }
    )
}

private fun getDefaultColor(lightColor: Int, darkColor: Int): Int {
    val colorTheme = Instabug.getTheme()
    return if (colorTheme == InstabugColorTheme.InstabugColorThemeLight) {
        lightColor
    } else {
        darkColor
    }
}

@SuppressLint("NewApi")
private fun setBackgroundTintList(replyButton: Button, color: Int) {
    if (BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP) {
        replyButton.backgroundTintList = ColorStateList.valueOf(color)
    } else {
        replyButton.background.setColorFilter(color, PorterDuff.Mode.MULTIPLY)
    }
}

private fun getIbgTheme(): IBGTheme? {
    return SettingsManager.getInstance().ibgTheme
}

private fun setTextViewStyle(
    view: TextView,
    ibgFont: Typeface?,
    ibgStyle: Int,
    customFont: Typeface?
) {
    val font = ibgFont ?: customFont
    view.setTypeface(font ?: view.typeface, ibgStyle)
}

@VisibleForTesting
fun getCustomFont(context: Context?): Typeface? =
    if (!InstabugCore.isFeatureEnabled(IBGFeature.CUSTOM_FONT)) {
        null
    } else {
        runCatching {
            context?.let {
                ResourcesCompat.getFont(
                    it,
                    R.font.instabug_custom_font
                )
            }
        }
            .onFailure {
                InstabugSDKLogger.w(
                    Constants.LOG_TAG,
                    "Chats notification view: custom font not overridden"
                )
            }.getOrNull()
    }
