package com.instabug.chat.screenrecording;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import com.instabug.bug.internal.video.InternalScreenRecordHelper;
import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.eventbus.ChatTriggeringEventBus;
import com.instabug.chat.eventbus.TriggeredChat;
import com.instabug.chat.model.Attachment;
import com.instabug.chat.model.Chat;
import com.instabug.chat.model.Message;
import com.instabug.chat.network.InstabugMessageUploaderJob;
import com.instabug.chat.ui.ChatActivityLauncher;
import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.eventbus.ScreenRecordingEventBus;
import com.instabug.library.internal.storage.cache.InMemoryCache;
import com.instabug.library.internal.video.ScreenRecordingContract;
import com.instabug.library.internal.video.ScreenRecordingEvent;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.InstabugDateFormatter;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.List;

import androidx.annotation.Nullable;
import io.reactivexport.disposables.Disposable;
import io.reactivexport.functions.Consumer;

/**
 * Created by vezikon on 6/7/17.
 */
@SuppressLint("ERADICATE_FIELD_NOT_INITIALIZED")
public class ExternalScreenRecordHelper implements ScreenRecordingContract {

    private static ExternalScreenRecordHelper INSTANCE;

    private String chatId;
    @Nullable
    private String messageId;
    @Nullable
    private Disposable disposable;
    @Nullable
    private Disposable chatTriggeringDisposable;

    public static ExternalScreenRecordHelper getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new ExternalScreenRecordHelper();
        }

        return INSTANCE;
    }

    public void start(final String chatId) {
        this.chatId = chatId;

        InternalScreenRecordHelper.getInstance().init();

        if (disposable == null || disposable.isDisposed()) {
            disposable = ScreenRecordingEventBus.getInstance().subscribe(screenRecordEvent -> {
                if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_FILE_READY) {
                    startWithHangingChat(screenRecordEvent.getVideoUri());
                    clear();
                } else if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_PERMISSION_DENIED) {
                    startWithHangingChat(screenRecordEvent.getVideoUri());
                    InternalScreenRecordHelper.getInstance().release();
                    clear();
                } else if (screenRecordEvent.getStatus() == ScreenRecordingEvent.RECORDING_ERROR) {
                    startWithHangingChat(null);
                    InternalScreenRecordHelper.getInstance().release();
                    clear();
                }
            });
        }

        chatTriggeringDisposable = ChatTriggeringEventBus.getInstance().subscribe(new Consumer<TriggeredChat>() {

            @Override
            public void accept(TriggeredChat triggeredChat) {
                if (chatId.equalsIgnoreCase(triggeredChat.getOldChatId())) {
                    setChatId(triggeredChat.getNewChatId());
                }
            }
        });

    }

    private void setChatId(String chatId) {
        this.chatId = chatId;
    }

    public boolean isRecording() {
        return InternalScreenRecordHelper.getInstance().isRecording();
    }

    private void startWithHangingChat(@Nullable Uri video) {

        if (video != null) {
            attachFileToChat(chatId, video);
            updateHangingChat(video);
        }

        Activity activity = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
        if (activity != null) {
            Intent intent = ChatActivityLauncher.chatProcessIntent(activity, chatId);
            activity.startActivity(intent);
        }
    }

    private void attachFileToChat(String chatId, Uri uri) {
        final Message message = new Message(UserManagerWrapper.getUserName(), UserManagerWrapper.getUserEmail(), InstabugCore.getPushNotificationToken()).setChatId(chatId)
                .setBody("").setMessagedAt(InstabugDateFormatter.getCurrentUTCTimeStampInSeconds())
                .setReadAt(InstabugDateFormatter.getCurrentUTCTimeStampInSeconds()).setDirection
                        (Message.Direction.INBOUND);

        //adding inbound attachments
        if (uri != null) {
            Attachment attachment = new Attachment();
            attachment.setName(uri.getLastPathSegment());
            attachment.setLocalPath(uri.getPath());
            attachment.setType(Attachment.AttachmentType.TYPE_VIDEO_RECORD);
            attachment.setState(Attachment.AttachmentState.STATE_OFFLINE);
            attachment.setVideoEncoded(false);

            assignHangingMessageId(message.getId());

            //Block sending the message till the video is ready
            message.setMessageState(Message.MessageState.STAY_OFFLINE);

            message.getAttachments().add(attachment);
        }

        Chat chat = ChatsCacheManager.getChat(chatId);
        if (chat != null && chat.getMessages() != null) {

            if (chat.getChatState() == Chat.ChatState.WAITING_ATTACHMENT_MESSAGE) {
                chat.setChatState(Chat.ChatState.SENT);
            } else if (chat.getChatState() != Chat.ChatState.SENT) {
                chat.setChatState(Chat.ChatState.READY_TO_BE_SENT);
            }

            chat.getMessages().add(message);
            InMemoryCache<String, Chat> chatCache = ChatsCacheManager.getCache();
            if (chatCache != null) {
                chatCache.put(chat.getId(), chat);
            }
        }
    }

    private void assignHangingMessageId(String messageId) {
        this.messageId = messageId;
    }

    private void updateHangingChat(Uri videoUri) {
        Chat chat = ChatsCacheManager.getChat(chatId);

        if (chat != null) {
            List<Message> messages = chat.getMessages();
            String hangingMessageId = messageId;

            for (int i = 0; i < messages.size(); i++) {
                Message message = messages.get(i);
                InstabugSDKLogger.d(Constants.LOG_TAG, "getting message with ID: " + message.getId());

                if (message.getId().equals(hangingMessageId)) {
                    updateHangingAttachment(message, videoUri);
                    message.setMessageState(Message.MessageState.READY_TO_BE_SENT);
                }
            }

            InMemoryCache<String, Chat> cache = ChatsCacheManager.getCache();
            if (cache != null) {
                cache.put(chat.getId(), chat);
            }

            InstabugSDKLogger.d(Constants.LOG_TAG, "video is encoded and updated in its message");

            Context context = Instabug.getApplicationContext();
            if (context != null) {
                InstabugMessageUploaderJob.getInstance().start();
            }

        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Hanging Chat is null and can't be updated");
        }
    }

    private void updateHangingAttachment(Message message, Uri videoUri) {
        List<Attachment> attachments = message.getAttachments();
        for (Attachment attachment : attachments) {
            if (attachment.getType() != null &&
                    attachment.getType().equals(Attachment.AttachmentType.TYPE_VIDEO_RECORD)) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Setting attachment type to Video");
                attachment.setName(videoUri.getLastPathSegment());
                attachment.setLocalPath(videoUri.getPath());
                attachment.setVideoEncoded(true);
                return;
            }
        }
    }

    private void unsubscribe() {
        if (disposable != null && !disposable.isDisposed()) {
            disposable.dispose();
        }

        if (chatTriggeringDisposable != null && !chatTriggeringDisposable.isDisposed()) {
            chatTriggeringDisposable.dispose();
        }
    }

    @Nullable
    @Override
    public Uri getAutoScreenRecordingFileUri() {
        return InternalScreenRecordHelper.getInstance().getAutoScreenRecordingFileUri();
    }

    @Override
    public void delete() {
        InternalScreenRecordHelper.getInstance().delete();
    }

    @Override
    public void clear() {
        unsubscribe();
        InternalScreenRecordHelper.getInstance().clear();
    }

    @Override
    public boolean isEnabled() {
        return InternalScreenRecordHelper.getInstance().isEnabled();
    }
}
