package com.instabug.bug.reportingpromptitems;

import static com.instabug.bug.Constants.PromptItemOrder;
import static com.instabug.library.util.LocaleUtils.getLocaleStringResource;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.Constants;
import com.instabug.bug.Constants.MainReportCategory;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.R;
import com.instabug.bug.ReportingActivityLauncher;
import com.instabug.bug.model.ReportCategory;
import com.instabug.library.InstabugCustomTextPlaceHolder;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.invocation.InvocationMode;
import com.instabug.bug.invocation.invocationdialog.InstabugDialogActivity;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.PlaceHolderUtils;

import java.util.ArrayList;

public class BugReportingPromptItem extends BaseReportingPromptItem {

    @Override
    public PluginPromptOption getItemOption(final Context context) {
        PluginPromptOption promptOption = new PluginPromptOption();
        promptOption.setOrder(PromptItemOrder.ORDER_BUG_REPORT_OPTION);
        promptOption.setInvocationMode(InvocationMode.NEW_BUG);
        promptOption.setIcon(R.drawable.ibg_core_ic_report_bug);
        promptOption.setPromptOptionIdentifier(PluginPromptOption.PromptOptionIdentifier.BUG_REPORT);
        promptOption.setTitle(getReportTitle(context));
        promptOption.setDescription(getReportDescription(context));
        promptOption.setOnInvocationListener(new PluginPromptOption.OnInvocationListener() {
            @Override
            public void onInvoke(@Nullable Uri initialScreenshotUri, String... reportCategories) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "[BugReportingPromptItem#onInvoke] Invoking ...");
                invoke(context, initialScreenshotUri, reportCategories);
            }
        });
        promptOption.setInitialScreenshotRequired(true);
        promptOption.setSubOptions(getSubOptions(Constants.ReportType.BUG));
        return promptOption;
    }


    @NonNull
    @VisibleForTesting
    String getReportTitle(Context context) {
        return PlaceHolderUtils.getPlaceHolder(
                InstabugCustomTextPlaceHolder.Key.REPORT_BUG,
                getLocaleStringResource(InstabugCore.getLocale(context),
                        R.string.IBGPromptOptionsReportBug, context));
    }

    @Override
    @SuppressLint("ERADICATE_RETURN_NOT_NULLABLE")
    String getReportDescription(Context context) {
        return PlaceHolderUtils.getPlaceHolder(
                InstabugCustomTextPlaceHolder.Key.REPORT_BUG_DESCRIPTION,
                getLocaleStringResource(InstabugCore.getLocale(context),
                        R.string.ib_bug_report_bug_description, context));
    }


    protected void invoke(Context context, @Nullable Uri initialScreenshotUri, String... reportCategories) {
        InstabugSDKLogger.d(Constants.LOG_TAG, "[BugReportingPromptItem#invoke] invoking...");
        callOnSdkInvoked();
        InstabugSDKLogger.v(Constants.LOG_TAG, "Handle invocation request new bug");
        createReportWithInitialScreenshot(initialScreenshotUri);
        if (LiveBugManager.getInstance().getBug() != null) {
            LiveBugManager.getInstance().getBug().setCategories(new ArrayList<String>());
            LiveBugManager.getInstance().getBug()
                    .addCategory(MainReportCategory.REPORT_BUG);
            for (String reportCategory : reportCategories) {
                LiveBugManager.getInstance().getBug()
                        .addCategory(reportCategory);
            }
        }
        startViewHierarchyInspection();
        // to close the dialog if opened
        Intent intent = InstabugDialogActivity.getIntent(context, null, null, null, true);
        context.startActivity(intent);
        handleNewReport(context);
    }

    @Override
    void handleNewReport(Context context) {
        context.startActivity(ReportingActivityLauncher.newBugProcessIntent(context));
    }

    @Override
    protected PluginPromptOption getSubOption(ReportCategory reportCategory,
                                              @Nullable PluginPromptOption parent,
                                              @Constants.ReportType String reportType, int order) {
        PluginPromptOption subOption = super.getSubOption(reportCategory, parent, reportType, order);
        subOption.setInvocationMode(InvocationMode.NEW_BUG);
        subOption.setPromptOptionIdentifier(PluginPromptOption.PromptOptionIdentifier.BUG_REPORT);
        return subOption;
    }
}
