package com.instabug.chat.annotation.shape;

import android.annotation.SuppressLint;
import android.graphics.Canvas;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;

import androidx.annotation.ColorInt;

import com.instabug.chat.annotation.ControlButton;
import com.instabug.chat.annotation.DirectionRectF;
import com.instabug.library.util.DrawingUtility;

import java.util.List;

/**
 * Created by Tarek360 on 11/24/16.
 */

public class RectShape extends Shape {

    protected int rotation;
    protected Path path;
    protected final static int HIT_RADIUS = 50;

    public RectShape(@ColorInt int color, float strokeWidth, int rotation) {
        super(color, strokeWidth);
        path = new Path();
        this.rotation = rotation;
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void draw(Canvas canvas, DirectionRectF bounds, DirectionRectF lastBounds) {

        if (isRotated()) {
            if (!bounds.hasBounds) {
                lastBounds.hasBounds = true;
                float cx = bounds.centerX();
                float cy = bounds.centerY();

                PointF point0 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.left, bounds.top));
                PointF point1 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.right, bounds.top));
                PointF point2 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.right, bounds.bottom));
                PointF point3 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.left, bounds.bottom));

                lastBounds.point0.set(point0);
                lastBounds.point1.set(point1);
                lastBounds.point2.set(point2);
                lastBounds.point3.set(point3);
            }
        }
        draw(canvas, bounds);
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    protected void draw(Canvas canvas, DirectionRectF bounds) {
        drawRect(canvas, bounds);
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private void drawRect(Canvas canvas, DirectionRectF bounds) {
        canvas.drawPath(getPath(bounds), paint);
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public boolean isTouched(PointF touchedPoint, DirectionRectF bounds) {

        if (!isRotated()) {
            RectF inner = new RectF(bounds);
            inner.inset(HIT_RADIUS, HIT_RADIUS);

            RectF outer = new RectF(bounds);
            outer.inset(-HIT_RADIUS, -HIT_RADIUS);

            return outer.contains(touchedPoint.x, touchedPoint.y)
                    && !inner.contains(touchedPoint.x, touchedPoint.y);
        } else {


            RectF fingerRect = new RectF(touchedPoint.x - HIT_RADIUS, touchedPoint.y - HIT_RADIUS,
                    touchedPoint.x + HIT_RADIUS, touchedPoint.y + HIT_RADIUS);

            List<PointF> pathPoints = DrawingUtility.getPoints(path);

            for (PointF p : pathPoints) {
                if (fingerRect.contains(p.x, p.y)) {
                    return true;
                }
            }
            return false;
        }
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void adjustBounds(DirectionRectF newBounds, DirectionRectF bounds, boolean isUndo) {
        if (!isRotated() || isUndo) {
            bounds.set(newBounds);
        }
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void drawControlButtons(Canvas canvas, DirectionRectF bounds,
                                   ControlButton[] controlButtons) {

        if (!isRotated()) {
            controlButtons[0].setCenterPoint(bounds.left, bounds.top);
            controlButtons[1].setCenterPoint(bounds.right, bounds.top);
            controlButtons[2].setCenterPoint(bounds.right, bounds.bottom);
            controlButtons[3].setCenterPoint(bounds.left, bounds.bottom);
        } else {
            controlButtons[0].setCenterPoint(bounds.point0.x, bounds.point0.y);
            controlButtons[1].setCenterPoint(bounds.point1.x, bounds.point1.y);
            controlButtons[2].setCenterPoint(bounds.point2.x, bounds.point2.y);
            controlButtons[3].setCenterPoint(bounds.point3.x, bounds.point3.y);
        }

        int color = paint.getColor();

        for (int i = 0; i < controlButtons.length; i++) {
            controlButtons[i].setColor(color);
            controlButtons[i].draw(canvas);
        }
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void drawBorder(Canvas canvas, PointF topLeftPoint, PointF topRightPoint,
                           PointF bottomRightPoint, PointF bottomLeftPoint) {
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void translateBy(DirectionRectF newBounds, DirectionRectF bounds, int dx, int dy) {

        newBounds.point0.set(bounds.point0.x + dx, bounds.point0.y + dy);

        newBounds.point1.set(bounds.point1.x + dx, bounds.point1.y + dy);
        newBounds.point2.set(bounds.point2.x + dx, bounds.point2.y + dy);
        newBounds.point3.set(bounds.point3.x + dx, bounds.point3.y + dy);

        newBounds.left = bounds.left + dx;
        newBounds.top = bounds.top + dy;
        newBounds.right = bounds.right + dx;
        newBounds.bottom = bounds.bottom + dy;
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void adjustPoint0(float eventX, float eventY, DirectionRectF bounds) {
        if (isRotated()) {
            adjustPoint0(eventX, eventY, bounds, true);
            reBound(bounds);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private void adjustPoint0(float eventX, float eventY, DirectionRectF bounds, boolean reset) {
        bounds.point0.set(eventX, eventY);

        float p0p1 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point2, bounds.point1, bounds.point0);

        if (bounds.point0.x < bounds.point1.x || bounds.point0.y < bounds.point1.y) {
            DrawingUtility.pointOnCircle(p0p1, rotation, bounds.point0, bounds.point1);
        } else if (bounds.point0.x > bounds.point1.x || bounds.point0.y > bounds.point1.y) {
            DrawingUtility.pointOnCircle(p0p1, 180 + rotation, bounds.point0, bounds.point1);
        }

        float p0p3 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point2, bounds.point3, bounds.point0);

        if (bounds.point0.y < bounds.point3.y || bounds.point0.x > bounds.point3.x) {
            DrawingUtility.pointOnCircle(p0p3, 90 + rotation, bounds.point0, bounds.point3);
        } else if (bounds.point0.y > bounds.point3.y || bounds.point0.x < bounds.point3.x) {
            DrawingUtility.pointOnCircle(p0p3, 270 + rotation, bounds.point0, bounds.point3);
        }
        if (reset) {
            adjustPoint2(bounds.point2.x, bounds.point2.y, bounds, false);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void adjustPoint1(float eventX, float eventY, DirectionRectF bounds) {
        if (isRotated()) {
            adjustPoint1(eventX, eventY, bounds, true);
            reBound(bounds);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private void adjustPoint1(float eventX, float eventY, DirectionRectF bounds, boolean reset) {
        bounds.point1.set(eventX, eventY);

        float p1p0 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point3, bounds.point0, bounds.point1);

        if (bounds.point1.x > bounds.point0.x || bounds.point1.y > bounds.point0.y) {
            DrawingUtility.pointOnCircle(p1p0, 180 + rotation, bounds.point1, bounds.point0);
        } else if (bounds.point1.x < bounds.point0.x || bounds.point1.y < bounds.point0.y) {
            DrawingUtility.pointOnCircle(p1p0, rotation, bounds.point1, bounds.point0);
        }

        float p1p2 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point3, bounds.point2, bounds.point1);

        if (bounds.point1.y < bounds.point2.y || bounds.point1.x > bounds.point2.x) {
            DrawingUtility.pointOnCircle(p1p2, 90 + rotation, bounds.point1, bounds.point2);
        } else if (bounds.point1.y > bounds.point2.y || bounds.point1.x < bounds.point2.x) {
            DrawingUtility.pointOnCircle(p1p2, 270 + rotation, bounds.point1, bounds.point2);
        }

        if (reset) {
            adjustPoint3(bounds.point3.x, bounds.point3.y, bounds, false);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void adjustPoint2(float eventX, float eventY, DirectionRectF bounds) {
        if (isRotated()) {
            adjustPoint2(eventX, eventY, bounds, true);
            reBound(bounds);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private void adjustPoint2(float eventX, float eventY, DirectionRectF bounds, boolean reset) {
        bounds.point2.set(eventX, eventY);

        float p2p1 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point0, bounds.point1, bounds.point2);

        if (bounds.point2.y > bounds.point1.y || bounds.point2.x < bounds.point1.x) {
            DrawingUtility.pointOnCircle(p2p1, 270 + rotation, bounds.point2, bounds.point1);
        } else if (bounds.point2.y < bounds.point1.y || bounds.point2.x > bounds.point1.x) {
            DrawingUtility.pointOnCircle(p2p1, 90 + rotation, bounds.point2, bounds.point1);
        }

        float p2p3 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point0, bounds.point3, bounds.point2);

        if (bounds.point2.x > bounds.point3.x || bounds.point2.y > bounds.point3.y) {
            DrawingUtility.pointOnCircle(p2p3, 180 + rotation, bounds.point2, bounds.point3);
        } else if (bounds.point2.x < bounds.point3.x || bounds.point2.y < bounds.point3.y) {
            DrawingUtility.pointOnCircle(p2p3, rotation, bounds.point2, bounds.point3);
        }

        if (reset) {
            adjustPoint0(bounds.point0.x, bounds.point0.y, bounds, false);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public void adjustPoint3(float eventX, float eventY, DirectionRectF bounds) {
        if (isRotated()) {
            adjustPoint3(eventX, eventY, bounds, true);
            reBound(bounds);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private void adjustPoint3(float eventX, float eventY, DirectionRectF bounds, boolean reset) {
        bounds.point3.set(eventX, eventY);

        float p3p0 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point1, bounds.point0, bounds.point3);

        if (bounds.point3.y > bounds.point0.y || bounds.point3.x < bounds.point0.x) {
            DrawingUtility.pointOnCircle(p3p0, 270 + rotation, bounds.point3, bounds.point0);
        } else if (bounds.point3.y < bounds.point0.y || bounds.point3.x > bounds.point0.x) {
            DrawingUtility.pointOnCircle(p3p0, 90 + rotation, bounds.point3, bounds.point0);
        }

        float p3p2 =
                (float) DrawingUtility.lineToPointDistance2D(bounds.point1, bounds.point2, bounds.point3);

        if (bounds.point3.x < bounds.point2.x || bounds.point3.y < bounds.point2.y) {
            DrawingUtility.pointOnCircle(p3p2, rotation, bounds.point3, bounds.point2);
        } else if (bounds.point3.x > bounds.point2.x || bounds.point3.y > bounds.point2.y) {
            DrawingUtility.pointOnCircle(p3p2, 180 + rotation, bounds.point3, bounds.point2);
        }

        if (reset) {
            adjustPoint1(bounds.point1.x, bounds.point1.y, bounds, false);
        }
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private void reBound(DirectionRectF bounds) {
        RectF rectF = new RectF();
        path.computeBounds(rectF, true);
        bounds.set(rectF);
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    private boolean isRotated() {
        return !(rotation == 0 || rotation == 180 || rotation == 90);
    }

    @Override
    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    public Path getPath(DirectionRectF bounds) {
        if (isRotated()) {
            if (!bounds.hasBounds) {
                bounds.hasBounds = true;
                float cx = bounds.centerX();
                float cy = bounds.centerY();

                PointF point0 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.left, bounds.top));
                PointF point1 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.right, bounds.top));
                PointF point2 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.right, bounds.bottom));
                PointF point3 =
                        DrawingUtility.rotatePoint(cx, cy, rotation, new PointF(bounds.left, bounds.bottom));

                bounds.point0.set(point0);
                bounds.point1.set(point1);
                bounds.point2.set(point2);
                bounds.point3.set(point3);
            }
        }

        preparePath(bounds);

        return path;
    }

    @SuppressLint("ERADICATE_INCONSISTENT_SUBCLASS_PARAMETER_ANNOTATION")
    protected void preparePath(DirectionRectF bounds) {
        path.reset();
        if (!isRotated()) {
            path.addRect(bounds, Path.Direction.CW);
        } else {
            path.moveTo(bounds.point0.x, bounds.point0.y);
            path.lineTo(bounds.point1.x, bounds.point1.y);
            path.lineTo(bounds.point2.x, bounds.point2.y);
            path.lineTo(bounds.point3.x, bounds.point3.y);
            path.close();
        }
    }
}

