package com.instabug.chat;

import android.content.Context;
import android.os.Build;
import android.os.Bundle;

import androidx.annotation.DrawableRes;
import androidx.annotation.RequiresApi;

import com.instabug.chat.cache.ChatsCacheManager;
import com.instabug.chat.network.InstabugPushNotificationTokenJob;
import com.instabug.chat.notification.NotificationManager;
import com.instabug.chat.settings.AttachmentTypesState;
import com.instabug.chat.settings.ChatSettings;
import com.instabug.chat.ui.ChatActivityLauncher;
import com.instabug.library.Feature;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.OnSdkDismissCallback;


import com.instabug.library.core.InstabugCore;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.library.invocation.InvocationMode;
import com.instabug.library.util.threading.PoolProvider;

import java.util.Map;

public class ChatsDelegate {

    /**
     * show chats list
     *
     * @since 5.0
     */
    public static void showChats() {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                if (isReadyToRun()) {
                    ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
                    if (plugin != null) {
                        Context context = plugin.getAppContext();
                        if (context != null) {
                            context.startActivity(ChatActivityLauncher
                                    .chatsProcessIntent(context));
                        }
                    }
                }
            }
        });
    }


    /**
     * open new chat view
     *
     * @since 5.0
     */
    @Deprecated
    public static void openNewChat() {
        if (isReadyToRun()) {
            InvocationManager.getInstance().forceInvoke(InvocationMode.ASK_QUESTION);
        }
    }


    /**
     * Sets the callback that gets executed when SDK is dismissed<br/>
     * WARNING: This runs on your application's main UI thread. Please do not include
     * any blocking operations to avoid ANRs.
     *
     * @param onSdkDismissedCallback to be called when Instabug SDK is dismissed
     * @throws IllegalStateException if Instabug object wasn't built using
     *                               {@link Instabug.Builder#build()} before this method was called
     * @since 8.0.0
     */
    public static void setOnSdkDismissCallback(OnSdkDismissCallback onSdkDismissedCallback) {
        ChatSettings.setOnSdkDismissCallback(onSdkDismissedCallback);
    }

    /**
     * Get current unread count of messages for this user
     *
     * @return number of messages that are unread for this user, 0 if the In-App chat feature
     * disabled
     */
    public static int getUnreadMessagesCount() throws IllegalStateException {
        if (isReadyToRun()) {
            return ChatsCacheManager.getUnreadCount();
        }
        return 0;
    }

    /**
     * Enabled/disable chat notification
     *
     * @param enable whether chat notification is reburied or not
     * @since 5.0
     */
    public static void enableNotification(boolean enable) {
        if (isReadyToRun()) {
            ChatSettings.enableNotification(enable);
        }
    }

    /**
     * Set whether new system notification received will play the default sound from
     * RingtoneManager or not (Default is {@code false})
     *
     * @param enable desired state of conversation sounds
     * @since 5.0
     */
    @Deprecated
    public static void enableSystemNotificationSound(boolean enable) {
        if (isReadyToRun()) {
            Replies.setSystemReplyNotificationSoundEnabled(enable);
        }
    }

    /**
     * Set whether new system notification received will play the default sound from
     * RingtoneManager or not (Default is {@code false})
     *
     * @param enable desired state of conversation sounds
     * @since 5.0
     */
    public static void enableInAppNotificationSound(boolean enable) {
        if (isReadyToRun()) {
            ChatSettings.enableInAppNotificationSound(enable);
        }
    }

    /**
     * Set the push notification's icon that will be shown with Instabug notifications
     *
     * @param notificationIcon the notification icon resource
     * @since 5.0
     */
    @Deprecated
    public static void setNotificationIcon(@DrawableRes int notificationIcon) {
        if (isReadyToRun()) {
            Replies.setNotificationIcon(notificationIcon);
        }
    }


    /**
     * Set the GCM registration token to Instabug
     *
     * @param token the GCM registration token
     * @since 5.0
     */
    public static void setPushNotificationRegistrationToken(String token) {
        if (isReadyToRun()) {
            if (token != null
                    && !token.isEmpty()
                    && !token.equalsIgnoreCase(InstabugCore.getPushNotificationToken())) {
                InstabugCore.setPushNotificationToken(token);
                InstabugCore.setPushNotificationTokenSent(false);

                ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
                if (plugin != null) {
                    Context context = plugin.getAppContext();
                    if (context != null && InstabugCore.getFeatureState(IBGFeature.PUSH_NOTIFICATION)
                            == Feature.State.ENABLED) {
                        InstabugPushNotificationTokenJob.getInstance().start();
                    }
                }
            }
        }
    }

    /**
     * Check if the received push notification is related to instabug or not
     *
     * @param data the push notification data bundle, false if the In-App chat feature
     *             disabled
     * @since 5.0
     */
    @Deprecated
    public static boolean isInstabugNotification(Bundle data) {
        if (isReadyToRun()) {
            return Replies.isInstabugNotification(data);
        }
        return false;
    }

    /**
     * Check if the received push notification is related to instabug or not
     *
     * @param data the push notification data bundle, false if the In-App chat feature
     *             disabled
     * @since 5.0
     */
    @Deprecated
    public static boolean isInstabugNotification(Map<String, String> data) {
        if (isReadyToRun()) {
            return Replies.isInstabugNotification(data);
        }
        return false;
    }

    /**
     * Show in-app Messaging's notifications
     *
     * @param data the data bundle related to Instabug
     * @since 5.0
     */
    public static void showNotification(Bundle data) {
        if (isReadyToRun()) {
            NotificationManager.getInstance().showNotification(data);
        }
    }

    /**
     * Show in-app Messaging's notifications
     *
     * @param data the data bundle related to Instabug
     * @since 5.0
     */
    public static void showNotification(Map<String, String> data) {
        if (isReadyToRun()) {
            NotificationManager.getInstance().showNotification(data);
        }
    }

    /**
     * Set whether new messages received will trigger a small sound notification or not
     * (Default is {@code false})
     *
     * @param shouldPlaySounds desired state of conversation sounds
     * @since 5.0
     */
    @Deprecated
    public static void enableConversationSound(boolean shouldPlaySounds) {
        if (isReadyToRun()) {
            ChatSettings.enableConversationSound(shouldPlaySounds);
        }
    }

    /**
     * Enabled/disable annotation step for the images attachment
     *
     * @param shouldSkipImageAttachmentAnnotation annotation step for the images attachment is
     *                                            enabled or disabled
     * @since 5.0.0
     */
    public static void skipImageAttachmentAnnotation(boolean shouldSkipImageAttachmentAnnotation) {
        if (isReadyToRun()) {
            ChatSettings.skipImageAttachmentAnnotation(shouldSkipImageAttachmentAnnotation);
        }
    }

    private static boolean isReadyToRun() {
        ChatPlugin plugin = (ChatPlugin) InstabugCore.getXPlugin(ChatPlugin.class);
        if (plugin != null) {
            return plugin.isAppContextAvailable() && isChatFeatureEnabled();
        }
        return false;
    }

    private static boolean isChatFeatureEnabled() {
        return InstabugCore.getFeatureState(IBGFeature.IN_APP_MESSAGING) == Feature.State.ENABLED;
    }

    /**
     * Set a notification channel id to a notification channel that notifications
     * can be posted to.
     *
     * @param pushNotificationChannelId an id to a notification channel that notifications
     *                                  can be posted to.
     * @throws IllegalStateException if Instabug object wasn't built using
     *                               {@link Instabug.Builder#build()} before this method was called
     * @since 5.0
     */
    @RequiresApi(api = Build.VERSION_CODES.O)
    public static void setPushNotificationChannelId(String pushNotificationChannelId)
            throws IllegalStateException {
        Replies.setPushNotificationChannelId(pushNotificationChannelId);
    }

    public static void setNewMessageHandler(Runnable runnable) {
        ChatSettings.setNewMessageRunnable(runnable);
    }

    public static void setAttachmentTypesEnabled(boolean extraScreenshot,
                                                 boolean imageFromGallery,
                                                 boolean screenRecording) {
        AttachmentTypesState attachmentTypesState = new AttachmentTypesState(
                extraScreenshot, imageFromGallery, screenRecording);

        ChatSettings.setAttachmentsTypesParams(attachmentTypesState);
    }

    public static void dismissSystemNotification() {
        NotificationManager.getInstance().dismissSystemNotification(Instabug.getApplicationContext());
    }
}