package com.instabug.bug.view.reporting

import android.content.res.ColorStateList
import android.graphics.drawable.StateListDrawable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.CheckBox
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.appcompat.widget.AppCompatImageView
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.DrawableCompat
import androidx.recyclerview.widget.RecyclerView
import com.instabug.bug.R
import com.instabug.bug.userConsent.UserConsent
import com.instabug.library.core.InstabugCore
import com.instabug.library.model.IBGTheme
import com.instabug.library.util.ThemeApplier
import com.instabug.library.util.ThemeApplier.applySecondaryTextStyle

fun interface OnMandatoryCheckStateChanged {
    operator fun invoke()
}

class ConsentItemViewHolder(private val view: View) : RecyclerView.ViewHolder(view) {

    private val consentCheckBox by lazy { view.findViewById<CheckBox>(R.id.cb_consent) }
    private val requiredLayout by lazy { view.findViewById<LinearLayout>(R.id.layout_required) }
    private val requiredImage by lazy { view.findViewById<AppCompatImageView>(R.id.required_iv) }
    private val checkBoxTextView by lazy { view.findViewById<TextView>(R.id.tv_checkbox) }
    private val consentRootView by lazy { view.findViewById<LinearLayout>(R.id.consent_root_layout) }
    private val requiredTextView by lazy { view.findViewById<TextView>(R.id.required_tv) }

    fun bind(
        item: UserConsent,
        onMandatoryCheckStateChanged: OnMandatoryCheckStateChanged,
        theme: IBGTheme?
    ) {
        renderConsentCheckBox(item, onMandatoryCheckStateChanged)
        requiredLayout.visibility = if (item.isMandatory) View.VISIBLE else View.GONE
        consentRootView.setOnClickListener { consentCheckBox.run { isChecked = !isChecked } }
        checkBoxTextView.text = item.description
            ?: view.context.getText(R.string.ibg_consent_default_description)

        applyTheme(theme, checkBoxTextView, requiredTextView, requiredImage)
    }

    private fun applyTheme(theme: IBGTheme?, checkBoxTv: TextView, requiredTv: TextView, requiredIv: ImageView) {
        checkBoxTv.applySecondaryTextStyle(theme)
        requiredTv.applySecondaryTextStyle(theme)
        theme?.secondaryTextColor.takeIf {
            it != ThemeApplier.DEFAULT_COLOR
        }?.let { color ->
            requiredIv.setColorFilter(color)
        }
    }

    private fun renderConsentCheckBox(
        item: UserConsent,
        onMandatoryCheckStateChanged: OnMandatoryCheckStateChanged
    ) = item.also {
        consentCheckBox.apply {
            setIBGSelectorDrawable()
            isChecked = it.isChecked
            setOnCheckedChangeListener { _, approvedByUser ->
                it.isChecked = approvedByUser
                if (it.isMandatory) onMandatoryCheckStateChanged()
            }
        }
    }

    private fun CheckBox.setIBGSelectorDrawable() = StateListDrawable().apply {
        addState(
            intArrayOf(-android.R.attr.state_checked),
            ContextCompat.getDrawable(context, R.drawable.ic_unchecked)
        )
        val checkedDrawable = ContextCompat.getDrawable(context, R.drawable.ic_checked)?.also {
            DrawableCompat.setTintList(it, ColorStateList.valueOf(InstabugCore.getPrimaryColor()))
        }
        addState(intArrayOf(android.R.attr.state_checked), checkedDrawable)
    }.let(::setButtonDrawable)
}

class ConsentsRecyclerViewAdapter(
    private val items: List<UserConsent>,
    private val onCheckedStateChanged: OnMandatoryCheckStateChanged,
    private val theme: IBGTheme?
) : RecyclerView.Adapter<ConsentItemViewHolder>() {

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ConsentItemViewHolder =
        LayoutInflater
            .from(parent.context)
            .inflate(R.layout.ibg_bug_consent_list_item, parent, false)
            .let(::ConsentItemViewHolder)

    override fun onBindViewHolder(holder: ConsentItemViewHolder, position: Int) =
        holder.bind(items[position], onCheckedStateChanged, theme)

    override fun getItemCount(): Int = items.count()
}