package com.instabug.bug.view.reporting;

import static android.content.Context.MEDIA_PROJECTION_SERVICE;
import static android.text.Spanned.SPAN_INCLUSIVE_INCLUSIVE;
import static com.google.android.material.bottomsheet.BottomSheetBehavior.BottomSheetCallback;
import static com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_COLLAPSED;
import static com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_DRAGGING;
import static com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_EXPANDED;
import static com.google.android.material.bottomsheet.BottomSheetBehavior.STATE_SETTLING;
import static com.instabug.bug.Constants.REQUEST_SCREEN_RECORDING_MIC_PERMISSIONS;
import static com.instabug.bug.LiveBugManager.DURATION_NOT_EXISTS;
import static com.instabug.bug.view.reporting.BaseReportingPresenter.SHOW_THANK_YOU_ACTIVITY;
import static com.instabug.library.InstabugCustomTextPlaceHolder.Key;
import static com.instabug.library.InstabugCustomTextPlaceHolder.Key.REPORT_ADD_ATTACHMENT_HEADER;
import static com.instabug.library.InstabugCustomTextPlaceHolder.Key.REPORT_REPRO_STEPS_DISCLAIMER_BODY;
import static com.instabug.library.InstabugCustomTextPlaceHolder.Key.REPORT_REPRO_STEPS_DISCLAIMER_LINK;
import static com.instabug.library.util.DrawableUtils.getRotateDrawable;
import static com.instabug.library.util.ThemeApplier.DEFAULT_COLOR;
import static com.instabug.library.util.ThemeApplier.applyPrimaryTextStyle;
import static com.instabug.library.util.ThemeApplier.applySecondaryTextStyle;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.graphics.PorterDuff;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.media.projection.MediaProjectionManager;
import android.net.Uri;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.os.Bundle;
import android.os.Handler;
import android.os.SystemClock;
import android.text.Editable;
import android.text.SpannableString;
import android.text.Spanned;
import android.text.method.LinkMovementMethod;
import android.text.style.StyleSpan;
import android.text.style.TypefaceSpan;
import android.util.DisplayMetrics;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ScrollView;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.StringRes;
import androidx.annotation.VisibleForTesting;
import androidx.coordinatorlayout.widget.CoordinatorLayout;
import androidx.core.content.ContextCompat;
import androidx.core.text.TextUtilsCompat;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentTransaction;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.bottomsheet.BottomSheetBehavior;
import com.instabug.bug.Constants;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.R;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.internal.video.VideoPlayerFragment;
import com.instabug.bug.screenrecording.ExternalScreenRecordHelper;
import com.instabug.bug.settings.BugSettings;
import com.instabug.bug.userConsent.UserConsent;
import com.instabug.bug.view.AttachmentsAdapter;
import com.instabug.bug.view.BugReportingActivityCallback;
import com.instabug.bug.view.InstabugThanksActivity;
import com.instabug.bug.view.annotation.BugAnnotationFragment;
import com.instabug.bug.view.extrafields.ExtraFieldsFragment;
import com.instabug.bug.view.visualusersteps.steppreview.VisualUserStepArgs;
import com.instabug.library.Feature;
import com.instabug.library.FragmentVisibilityChangedListener;
import com.instabug.library.IBGFeature;
import com.instabug.library.Instabug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.ui.InstabugBaseFragment;
import com.instabug.library.internal.video.RequestPermissionActivityLauncher;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.ui.custom.InstabugAlertDialog;
import com.instabug.library.ui.custom.InstabugEditText;
import com.instabug.library.user.UserManagerWrapper;
import com.instabug.library.util.AccessibilityUtils;
import com.instabug.library.util.DeviceStateProvider;
import com.instabug.library.util.DisplayUtils;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.LocaleUtils;
import com.instabug.library.util.OrientationUtils;
import com.instabug.library.util.PlaceHolderUtils;
import com.instabug.library.util.SimpleTextWatcher;
import com.instabug.library.util.SystemServiceUtils;
import com.instabug.library.util.ThemeApplier;
import com.instabug.library.util.threading.PoolProvider;
import com.instabug.library.view.IBGProgressDialog;
import com.instabug.library.view.ViewUtils;

import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.util.List;

/**
 * @author Hossam
 */
@SuppressLint({"ERADICATE_FIELD_NOT_INITIALIZED", "ERADICATE_FIELD_NOT_NULLABLE"})
public abstract class BaseReportingFragment extends InstabugBaseFragment<IBaseReportingView.Presenter>
        implements AttachmentsAdapter.AttachmentOnClickListener, View.OnClickListener,
        IBaseReportingView.View, View.OnFocusChangeListener {

    private static int lastState = -1;
    private static final String MESSAGE_NULL_BUG = "Bug is null";
    private static final String MESSAGE_NULL_PRESENTER = "Presenter is null";
    private EditText emailEditText;
    private EditText messageEditText;
    private TextView disclaimerTextView;
    protected TextView reproStepsDisclaimerTextView;
    private RecyclerView attachmentsList;
    private LinearLayout addAttachments;
    private LinearLayout bugReportingInputsContainer;
    @VisibleForTesting
    ScrollView scrollView;
    @Nullable
    private String issueMessage;
    private boolean isVideoPlayBtnClicked;
    private BroadcastReceiver refreshAttachmentsBroadcastReceiver;
    @Nullable
    private IBGProgressDialog preparingProgressDialog;
    private AttachmentsAdapter attachmentsAdapter;
    private Callbacks listener;
    @Nullable
    private BugReportingActivityCallback activityCallback;
    @Nullable
    private BottomSheetBehavior<View> attachmentBottomSheetBehavior;
    @Nullable
    private ImageView arrowHandler;
    private int enabledAttachmentCount = 0;
    private boolean isKeyboardOpen = false;
    private boolean isBottomSheetDragDisabled = false;
    private long lastClickTime = 0;
    @Nullable
    private Runnable attachmentClickRunnable;
    private final Handler attachmentClickHandler = new Handler();
    @Nullable
    private View focusedView;
    private RecyclerView consentsRecyclerView;
    private MenuItem submitMenuItem;

    // Arrow handler Accessibility delegates
    private final AccessibilityDelegateCompat collapseDelegate = new AccessibilityDelegateCompat() {

        @Override
        public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
            super.onInitializeAccessibilityNodeInfo(host, info);
            info.setText(getLocalizedString(R.string.ibg_bug_report_arrow_handler_collapse_description));
        }
    };

    private final AccessibilityDelegateCompat expandDelegate = new AccessibilityDelegateCompat() {

        @Override
        public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
            super.onInitializeAccessibilityNodeInfo(host, info);
            info.setText(getLocalizedString(R.string.ibg_bug_report_arrow_handler_expand_description));
        }
    };

    ViewTreeObserver.OnGlobalLayoutListener keyboardObserver = new ViewTreeObserver.OnGlobalLayoutListener() {
        @Override
        public void onGlobalLayout() {
            if (getActivity() != null && rootView != null) {
                Rect r = new Rect();
                getActivity().getWindow().getDecorView().getWindowVisibleDisplayFrame(r);
                int screenHeight = getActivity().getWindow().getDecorView().getRootView
                        ().getHeight();

                // r.bottom is the position above soft keypad or device button.
                // if keypad is shown, the r.bottom is smaller than that before.
                int keypadHeight = screenHeight - r.bottom;

                if (keypadHeight > screenHeight * 0.15) { // 0.15 ratio is perhaps enough
                    // to determine keypad height.
                    // keyboard is opened
                    isKeyboardOpen = true;
                    if (attachmentBottomSheetBehavior != null)
                        attachmentBottomSheetBehavior.setState(STATE_COLLAPSED);
                    isBottomSheetDragDisabled = true;
                    if (arrowHandler != null)
                        arrowHandler.setVisibility(View.INVISIBLE);
                } else {
                    // keyboard is closed
                    isBottomSheetDragDisabled = false;
                    isKeyboardOpen = false;
                    if (enabledAttachmentCount > 1 && arrowHandler != null) {
                        arrowHandler.setVisibility(View.VISIBLE);
                    }
                }
            }
        }
    };

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        postponeEnterTransition();
        super.onCreate(savedInstanceState);
        if (getArguments() != null)
            issueMessage = getArguments().getString(Constants.EXTRA_REPORTING_MESSAGE);
        setHasOptionsMenu(true);
        initRefreshAttachmentsReceiver();
        if (presenter == null) {
            presenter = createPresenter();
        }

    }

    private void setVideoRecordingStatus(int visibility) {
        if (presenter != null && !presenter.shouldShowScreenRecording()) {
            View videoAttachmentView = findViewById(R.id.instabug_attach_video);
            View videoAttachmentCollapseIcon = findViewById(R.id.ib_bug_attachment_collapsed_video_icon);
            if (videoAttachmentView != null)
                videoAttachmentView.setVisibility(View.GONE);
            if (videoAttachmentCollapseIcon != null)
                videoAttachmentCollapseIcon.setVisibility(View.GONE);
        } else {
            View videoAttachmentView = findViewById(R.id.instabug_attach_video);
            if (videoAttachmentView != null)
                videoAttachmentView.setVisibility(visibility);
        }
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        try {
            listener = (BaseReportingFragment.Callbacks) context;
            if (getActivity() instanceof BugReportingActivityCallback) {
                activityCallback = (BugReportingActivityCallback) context;
            }
        } catch (ClassCastException e) {
            throw new ClassCastException(getClass().getSimpleName().toString()
                    + " must implement BaseReportingFragment.Callbacks");
        }
    }

    @Override
    public void onDetach() {
        super.onDetach();
        activityCallback = null;
        listener = null;
    }

    @Override
    protected int getLayout() {
        return R.layout.ibg_bug_frgament_reporting_layout;
    }

    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    protected void initViews(View rootView, @Nullable Bundle savedInstanceState) {
        if (getActivity() instanceof ReportingContainerActivity) {
            ReportingContainerActivity activity = (ReportingContainerActivity) getActivity();
            activity.setToolbarNavigationContentDescription(getCloseButtonContentDescription());
            activity.setToolbarUpIconClose();
        }
        scrollView = findViewById(R.id.ib_bug_scroll_view);
        InstabugEditText ibMessageEdit = findViewById(R.id.instabug_edit_text_message);
        messageEditText = ibMessageEdit.getEditText();
        messageEditText.setOnFocusChangeListener(this);
        InstabugEditText emailEdit = findViewById(R.id.instabug_edit_text_email);
        emailEditText = emailEdit.getEditText();
        emailEditText.setOnFocusChangeListener(this);
        attachmentsList = findViewById(R.id.instabug_lyt_attachments_list);
        disclaimerTextView = findViewById(R.id.instabug_text_view_disclaimer);
        reproStepsDisclaimerTextView = findViewById(R.id.instabug_text_view_repro_steps_disclaimer);
        addAttachments = findViewById(R.id.instabug_add_attachment);
        consentsRecyclerView = findViewById(R.id.instabug_lyt_consent_list);

        IBaseReportingView.Presenter presenter = this.presenter;
        if (AccessibilityUtils.isTalkbackEnabled()) {
            ViewCompat.setAccessibilityDelegate(addAttachments, new AccessibilityDelegateCompat() {
                @Override
                public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                    super.onInitializeAccessibilityNodeInfo(host, info);
                    info.setContentDescription(getLocalizedString(R.string.ibg_bug_report_add_attachment_content_description));
                }
            });
        }
        bugReportingInputsContainer = findViewById(R.id.instabug_bug_reporting_edit_texts_container);

        initAttachmentsActionBar();

        if (getContext() != null) {
            attachmentsList.setLayoutManager(new LinearLayoutManager(getContext(), LinearLayoutManager.HORIZONTAL,
                    TextUtilsCompat.getLayoutDirectionFromLocale(InstabugCore.getLocale(getContext())) == ViewCompat.LAYOUT_DIRECTION_RTL));
            ViewCompat.setLayoutDirection(attachmentsList, ViewCompat.LAYOUT_DIRECTION_LTR);
            attachmentsAdapter = new AttachmentsAdapter(getContext(), null, this);
        }
        String emailHint = PlaceHolderUtils.getPlaceHolder(Key.EMAIL_FIELD_HINT,
                getLocalizedString(R.string.instabug_str_email_hint));
        emailEditText.setHint(emailHint);

        if (AccessibilityUtils.isTalkbackEnabled()) {
            ViewCompat.setAccessibilityDelegate(emailEditText, new AccessibilityDelegateCompat() {
                @Override
                public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                    super.onInitializeAccessibilityNodeInfo(host, info);
                    info.setText(emailHint);
                    info.setShowingHintText(true);
                }
            });

            ViewCompat.setAccessibilityDelegate(messageEditText, new AccessibilityDelegateCompat() {
                @Override
                public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                    super.onInitializeAccessibilityNodeInfo(host, info);
                    if (presenter != null)
                        info.setText(presenter.getMessageHint());
                    info.setShowingHintText(true);
                }
            });
        }

        reproStepsDisclaimerTextView.setOnClickListener(this);

        if (!BugSettings.getInstance().isEmailFieldVisible()) {
            emailEdit.setVisibility(View.GONE);
        }
        if (presenter != null && presenter.getMessageHint() != null)
            messageEditText.setHint(presenter.getMessageHint());

        if (issueMessage != null) {
            messageEditText.setText(issueMessage);
        }

        // update email field
        if (BugSettings.getInstance().isEmailFieldVisible()) {
            PoolProvider.postIOTask(() -> {
                String userEmail = UserManagerWrapper.getEmailForBugReport();
                if (LiveBugManager.getInstance().getBug() != null) {
                    State state = LiveBugManager.getInstance().getBug().getState();
                    String userEmailFromState = state != null ? state.getUserEmail() : null;
                    String email = null;
                    if (userEmailFromState != null && !userEmailFromState.isEmpty())
                        email = userEmailFromState;
                    else if (userEmail != null && !userEmail.isEmpty())
                        email = userEmail;

                    if (email != null) setEmailEditTextOnMainThread(email);
                }
                addEmailTextWatcher();
            });


        }

        if (presenter != null) {
            presenter.handleVisualUserStepsDisclaimer(getDisclaimer(), getDisclaimerLink());
            presenter.handleInstabugDisclaimer();
        }
        this.presenter = presenter;

        hidKeyboard();

        // Handle small screen sizes when DisclaimerText is shown
        if (shouldAdjustViewsHeights()) {
            float desiredTextSize = DisplayUtils.dpToPx(getResources(), 5);
            int desiredPadding = DisplayUtils.dpToPxIntRounded(getResources(), 14);
            // Email edittext
            emailEditText.setTextSize(desiredTextSize);
            emailEditText.setPadding(desiredPadding, desiredPadding, desiredPadding, desiredPadding);
            // Message edittext
            messageEditText.setTextSize(desiredTextSize);
            messageEditText.setPadding(desiredPadding, desiredPadding, desiredPadding, desiredPadding);

            emailEditText.setMinimumHeight(0);
            emailEditText.setLines(1);
        }

        messageEditText.addTextChangedListener(new SimpleTextWatcher() {
            @Override
            public void afterTextChanged(Editable s) {
                if (getActivity() != null && presenter != null && messageEditText != null) {
                    presenter.onMessageChanged(messageEditText.getText().toString());
                }
            }
        });

        applyTheme(emailEdit, ibMessageEdit);
    }


    @VisibleForTesting
    public void applyTheme(@Nullable InstabugEditText ibEmailEditText, @Nullable InstabugEditText descriptionEditText) {
        if (theme != null) {
            int bgColor = theme.getBackgroundColor();

            if (bgColor != ThemeApplier.DEFAULT_COLOR) {
                CoordinatorLayout container = findViewById(R.id.ib_reporting_container);
                if (container != null) {
                    container.setBackgroundColor(bgColor);
                }
            }

            if (ibEmailEditText != null) {
                applyPrimaryTextStyle(ibEmailEditText.getLabelTextView(), theme);
            }

            if (descriptionEditText != null) {
                applyPrimaryTextStyle(descriptionEditText.getLabelTextView(), theme);
            }

            applySecondaryTextStyle(emailEditText, theme);
            if (theme != null && theme.getSecondaryTextColor() != ThemeApplier.DEFAULT_COLOR) {
                emailEditText.setHintTextColor(theme.getSecondaryTextColor());
                messageEditText.setHintTextColor(theme.getSecondaryTextColor());
            }
            applySecondaryTextStyle(messageEditText, theme);

            applySecondaryTextStyle(reproStepsDisclaimerTextView, theme);

            Typeface ctaFont = null;
            if (theme != null) {
                ctaFont = theme.getCtaTextFont();
            }
            if (ctaFont != null || (theme != null && theme.getCtaTextStyle() != Typeface.NORMAL)) {
                String disclaimer = reproStepsDisclaimerTextView.getText().toString();
                SpannableString spannableString = new SpannableString(reproStepsDisclaimerTextView.getText());
                int disclaimerLinkStartPosition = disclaimer.indexOf(getDisclaimerLink());
                if (disclaimerLinkStartPosition != -1) {
                    if (VERSION.SDK_INT >= VERSION_CODES.P && ctaFont != null) {
                        spannableString.setSpan(new TypefaceSpan(ctaFont), disclaimerLinkStartPosition, spannableString.length(), SPAN_INCLUSIVE_INCLUSIVE);
                    }
                    spannableString.setSpan(new StyleSpan(theme.getCtaTextStyle()), disclaimerLinkStartPosition, spannableString.length(), SPAN_INCLUSIVE_INCLUSIVE);
                }
                reproStepsDisclaimerTextView.setText(spannableString);
            }
        }
    }

    private void addEmailTextWatcher() {
        PoolProvider.postMainThreadTask(() -> {
            if (emailEditText != null)
                emailEditText.addTextChangedListener(new SimpleTextWatcher() {
                    @Override
                    public void afterTextChanged(Editable s) {
                        if (emailEditText != null) {
                            final String email = emailEditText.getText().toString();
                            if (presenter != null)
                                presenter.onEmailChanged(email);
                        }
                    }
                });
        });
    }

    private void setEmailEditTextOnMainThread(String userEmail) {
        PoolProvider.postMainThreadTask(() -> {
            if (emailEditText != null) emailEditText.setText(userEmail);
        });

    }

    private boolean shouldAdjustViewsHeights() {
        return DisplayUtils.isSmallDevice()
                && BugSettings.getInstance().getDisclaimerText() != null
                && !BugSettings.getInstance().getDisclaimerText().toString().equals("");
    }


    private String getDisclaimer() {
        return PlaceHolderUtils.getPlaceHolder(
                getContext(),
                REPORT_REPRO_STEPS_DISCLAIMER_BODY,
                R.string.IBGReproStepsDisclaimerBody);
    }


    private String getDisclaimerLink() {
        return PlaceHolderUtils.getPlaceHolder(
                getContext(),
                REPORT_REPRO_STEPS_DISCLAIMER_LINK,
                R.string.IBGReproStepsDisclaimerLink);
    }

    private void collapseAttachmentBar() {
        if (rootView == null) return;
        if (findViewById(R.id.instabug_add_attachment) != null)
            findViewById(R.id.instabug_add_attachment).setVisibility(View.VISIBLE);
        if (presenter != null && presenter.shouldShowScreenRecording()) {
            setVideoRecordingStatus(View.INVISIBLE);
        } else {
            setVideoRecordingStatus(View.GONE);
        }
    }

    private void expandAttachmentBar() {
        if (rootView == null) return;
        if (presenter != null && presenter.shouldShowScreenRecording()) {
            if (findViewById(R.id.instabug_add_attachment) != null)
                findViewById(R.id.instabug_add_attachment).setVisibility(View.INVISIBLE);
            setVideoRecordingStatus(View.VISIBLE);
        } else {
            if (findViewById(R.id.instabug_add_attachment) != null)
                findViewById(R.id.instabug_add_attachment).setVisibility(View.GONE);
            setVideoRecordingStatus(View.GONE);
        }
    }

    @SuppressLint("WrongConstant")
    private void initAttachmentsActionBar() {
        try {
            int backgroundColor = DEFAULT_COLOR;

            if (theme != null) {

                backgroundColor = theme.getBackgroundColor();
            }

            View bottomSheet = findViewById(R.id.instabug_attachment_bottom_sheet);
            if (bottomSheet != null && backgroundColor != ThemeApplier.DEFAULT_COLOR) {
                bottomSheet.setBackgroundColor(backgroundColor);
            }
            applyPrimaryTextStyle(findViewById(R.id.instabug_add_attachment_label), theme);

            handleBottomSheetArrow();

            ImageView instabugAddAttachmentIcon = findViewById(R.id.instabug_add_attachment_icon);
            attachmentBottomSheetBehavior = BottomSheetBehavior.from(bottomSheet);
            attachmentBottomSheetBehavior.setPeekHeight(ViewUtils.convertDpToPx(Instabug.getApplicationContext(), 100));
            LinearLayout addAttachmentContainer = findViewById(R.id.instabug_add_attachment);
            if (addAttachmentContainer != null) {
                addAttachmentContainer.setOnClickListener(this);
            }

            LinearLayout bottomSheetArrowContainer = findViewById(R.id.ib_bottomsheet_arrow_layout);
            if (bottomSheetArrowContainer != null) {
                bottomSheetArrowContainer.setOnClickListener(this);
                if (backgroundColor != ThemeApplier.DEFAULT_COLOR) {
                    bottomSheetArrowContainer.setBackgroundColor(backgroundColor);
                }
            }
            if (instabugAddAttachmentIcon != null) {
                colorizeImg(instabugAddAttachmentIcon, SettingsManager.getInstance().getPrimaryColor());
            }

            applyPrimaryTextStyle(findViewById(R.id.instabug_attach_video_label), theme);
            applyPrimaryTextStyle(findViewById(R.id.instabug_attach_screenshot_label), theme);
            applyPrimaryTextStyle(findViewById(R.id.instabug_attach_gallery_image_label), theme);

            setListenersAndIconColors();

            if (addAttachmentContainer != null)
                addAttachmentContainer.setVisibility(View.INVISIBLE);

            if (enabledAttachmentCount > 1) {
                scrollView.setPadding(0, 0, 0, ViewUtils.convertDpToPx(Instabug.getApplicationContext(), 130));
                if (attachmentBottomSheetBehavior != null)
                    attachmentBottomSheetBehavior.setBottomSheetCallback(new BottomSheetCallback() {
                        @Override
                        public void onStateChanged(@NonNull View bottomSheet, int newState) {
                            if (scrollView == null || newState == STATE_SETTLING) return;
                            lastState = newState;

                            if (newState == STATE_COLLAPSED && !isKeyboardOpen) {
                                scrollView.setPadding(0, 0, 0, ViewUtils.convertDpToPx(Instabug.getApplicationContext(), 0));
                            } else if (newState == STATE_EXPANDED) {
                                scrollView.setPadding(0, 0, 0, ViewUtils.convertDpToPx(Instabug.getApplicationContext(), 130));
                            }

                            if ((newState == STATE_DRAGGING && isBottomSheetDragDisabled) || isKeyboardOpen) {
                                collapseAttachmentBar();
                                return;
                            }

                            if (newState == STATE_COLLAPSED) {
                                collapseAttachmentBar();
                            } else {
                                expandAttachmentBar();
                            }
                        }

                        @Override
                        public void onSlide(@NonNull View bottomSheet, float slideOffset) {
                            if (arrowHandler != null) {
                                arrowHandler.setRotation((1 - slideOffset) * 180);
                                if (AccessibilityUtils.isTalkbackEnabled()) {
                                    if (slideOffset == 0) {
                                        ViewCompat.setAccessibilityDelegate(arrowHandler, expandDelegate);
                                    } else if (slideOffset == 1) {
                                        ViewCompat.setAccessibilityDelegate(arrowHandler, collapseDelegate);
                                    }
                                }
                            }
                        }
                    });
            } else {
                scrollView.setPadding(0, 0, 0, ViewUtils.convertDpToPx(Instabug.getApplicationContext(), 0));
                if (attachmentBottomSheetBehavior != null)
                    attachmentBottomSheetBehavior.setBottomSheetCallback(new BottomSheetCallback() {
                        @Override
                        public void onStateChanged(@NonNull View bottomSheet, int newState) {
                            if (attachmentBottomSheetBehavior != null)
                                attachmentBottomSheetBehavior.setState(STATE_COLLAPSED);
                        }

                        @Override
                        public void onSlide(@NonNull View bottomSheet, float slideOffset) {

                        }
                    });
            }
            if (attachmentBottomSheetBehavior != null)
                attachmentBottomSheetBehavior.setState(lastState == -1
                        ? STATE_EXPANDED : lastState);


            if (lastState == STATE_COLLAPSED) {
                collapseAttachmentBar();
                if (attachmentBottomSheetBehavior != null)
                    attachmentBottomSheetBehavior.setState(STATE_COLLAPSED);
                if (arrowHandler != null) {
                    arrowHandler.setRotation(180);

                    if (AccessibilityUtils.isTalkbackEnabled()) {
                        ViewCompat.setAccessibilityDelegate(arrowHandler, expandDelegate);
                    }
                }
            } else {
                expandAttachmentBar();
                if (arrowHandler != null) {
                    arrowHandler.setRotation(0);

                    if (AccessibilityUtils.isTalkbackEnabled()) {
                        ViewCompat.setAccessibilityDelegate(arrowHandler, collapseDelegate);
                    }
                }
            }

            addCustomPlaceHolders();

            if (getActivity() != null) {
                if (OrientationUtils.isInLandscape(getActivity())) {
                    collapseAttachmentBar();
                    if (attachmentBottomSheetBehavior != null)
                        attachmentBottomSheetBehavior.setState(STATE_COLLAPSED);
                    if (arrowHandler != null) arrowHandler.setRotation(180);
                }
            }
            handleAttachmentsDragging();
            this.arrowHandler = arrowHandler;
        } catch (
                Exception e) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Error in initAttachmentsActionBar", e);
        }

    }

    private void handleBottomSheetArrow() {
        ImageView arrowHandler = findViewById(R.id.arrow_handler);
        if (arrowHandler != null) {
            arrowHandler.setRotation(0);
            arrowHandler.setOnClickListener(this);
            if (AccessibilityUtils.isTalkbackEnabled()) {
                ViewCompat.setAccessibilityDelegate(arrowHandler, collapseDelegate);
            }
        }
    }

    private String getAddAttachmentLabel() {
        return PlaceHolderUtils.getPlaceHolder(getContext(),
                REPORT_ADD_ATTACHMENT_HEADER,
                R.string.instabug_str_add_attachment);
    }

    /**
     * if only one attachment is disabled, bottom sheet will not be draggable
     */
    private void handleAttachmentsDragging() {
        if (arrowHandler != null && enabledAttachmentCount == 1) {
            arrowHandler.setVisibility(View.GONE);
            if (findViewById(R.id.instabug_add_attachment) != null)
                findViewById(R.id.instabug_add_attachment).setVisibility(View.GONE);
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        announceScreenVideoDuration();
        registerKeyboardObserver();
    }

    private void announceScreenVideoDuration() {
        long duration = LiveBugManager.getInstance().getAndResetRecordingDuration();
        if (duration != DURATION_NOT_EXISTS && AccessibilityUtils.isTalkbackEnabled()) {
            String message = getLocalizedString(R.string.ibg_screen_recording_duration_ended_for_accessibility, duration);
            AccessibilityUtils.sendTextEvent(message);
        }
    }

    private void setListenersAndIconColors() {
        initScreenRecordViews();
        initExtraScreenShotViews();
        initGalleryAttachmentViews();
    }

    private void initScreenRecordViews() {
        if (presenter != null && presenter.shouldShowScreenRecording()) {
            enabledAttachmentCount++;
            if (findViewById(R.id.instabug_attach_video) != null)
                findViewById(R.id.instabug_attach_video).setOnClickListener(this);

            ImageView icon = findViewById(R.id.instabug_attach_video_icon);
            ImageView collapsedIcon = findViewById(R.id.ib_bug_attachment_collapsed_video_icon);

            colorizeImg(icon, SettingsManager.getInstance().getPrimaryColor());
            if (getContext() != null) {
                colorizeImg(collapsedIcon, SettingsManager.getInstance().getPrimaryColor());
            }
        } else {
            setVideoRecordingStatus(View.GONE);
            if (findViewById(R.id.ib_bug_attachment_collapsed_video_icon) != null)
                findViewById(R.id.ib_bug_attachment_collapsed_video_icon).setVisibility(View.GONE);
            if (findViewById(R.id.ib_bug_videorecording_separator) != null)
                findViewById(R.id.ib_bug_videorecording_separator).setVisibility(View.GONE);
        }
    }

    private void initExtraScreenShotViews() {
        if (presenter != null && presenter.shouldShowExtraScreenshot()) {
            enabledAttachmentCount++;
            if (findViewById(R.id.instabug_attach_screenshot) != null)
                findViewById(R.id.instabug_attach_screenshot).setOnClickListener(this);

            ImageView icon = findViewById(R.id.instabug_attach_screenshot_icon);
            ImageView collapsedIcon = findViewById(R.id.ib_bug_attachment_collapsed_screenshot_icon);

            colorizeImg(icon, SettingsManager.getInstance().getPrimaryColor());
            if (getContext() != null) {
                colorizeImg(collapsedIcon, SettingsManager.getInstance().getPrimaryColor());
            }
        } else {
            if (findViewById(R.id.instabug_attach_screenshot) != null)
                findViewById(R.id.instabug_attach_screenshot).setVisibility(View.GONE);
            if (findViewById(R.id.ib_bug_attachment_collapsed_screenshot_icon) != null)
                findViewById(R.id.ib_bug_attachment_collapsed_screenshot_icon).setVisibility(View.GONE);
            if (findViewById(R.id.ib_bug_screenshot_separator) != null)
                findViewById(R.id.ib_bug_screenshot_separator).setVisibility(View.GONE);
        }
    }

    private void initGalleryAttachmentViews() {
        if (BugSettings.getInstance().getAttachmentsTypesParams().isAllowAttachImageFromGallery()) {
            enabledAttachmentCount++;
            View attacheGalleryImg = findViewById(R.id.instabug_attach_gallery_image);
            ImageView icon = findViewById(R.id.instabug_attach_gallery_image_icon);
            ImageView collapsedIcon = findViewById(R.id.ib_bug_attachment_collapsed_photo_library_icon);

            if (attacheGalleryImg != null) {
                attacheGalleryImg.setOnClickListener(this);
            }
            if (getContext() != null) {
                colorizeImg(collapsedIcon, SettingsManager.getInstance().getPrimaryColor());
            }
            colorizeImg(icon, SettingsManager.getInstance().getPrimaryColor());
        } else {
            View attacheGalleryImg = findViewById(R.id.instabug_attach_gallery_image);
            View collapseAttachmentsIcon = findViewById(R.id.ib_bug_attachment_collapsed_photo_library_icon);
            if (attacheGalleryImg != null) {
                attacheGalleryImg.setVisibility(View.GONE);
            }
            if (collapseAttachmentsIcon != null) {
                collapseAttachmentsIcon.setVisibility(View.GONE);
            }
        }
    }

    private void colorizeImg(ImageView imageView, int color) {
        if (imageView == null) return;

        imageView.getDrawable().setColorFilter(color, PorterDuff.Mode.SRC_IN);
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void addCustomPlaceHolders() {
        TextView attachGalleryImgLabel = findViewById(R.id.instabug_attach_gallery_image_label);
        TextView attachScreenshotLabel = findViewById(R.id.instabug_attach_screenshot_label);
        TextView attachVideoLabel = findViewById(R.id.instabug_attach_video_label);

        if (attachGalleryImgLabel != null)
            attachGalleryImgLabel.setText(PlaceHolderUtils.getPlaceHolder(Key.ADD_IMAGE_FROM_GALLERY, getLocalizedString(R
                    .string.instabug_str_pick_media_from_gallery)));
        if (attachScreenshotLabel != null)
            attachScreenshotLabel.setText(PlaceHolderUtils.getPlaceHolder(Key.ADD_EXTRA_SCREENSHOT, getLocalizedString(R
                    .string.instabug_str_take_screenshot)));
        if (attachVideoLabel != null)
            attachVideoLabel.setText(PlaceHolderUtils.getPlaceHolder(Key.ADD_VIDEO, getLocalizedString(R.string
                    .instabug_str_record_video)));
    }

    @Override
    public void onViewCreated(@NotNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        if (getActivity() != null)
            getActivity().getWindow().setSoftInputMode(WindowManager.LayoutParams
                    .SOFT_INPUT_ADJUST_RESIZE);
        if (activityCallback != null && presenter != null) {
            activityCallback.setToolbarTitle(presenter.getTitle());
        }
    }

    @Override
    public void onStart() {
        super.onStart();
        IBaseReportingView.Presenter presenter = this.presenter;
        if (getActivity() != null && presenter != null) {
            presenter.onStart();
            LocalBroadcastManager.getInstance(getActivity()).registerReceiver(
                    refreshAttachmentsBroadcastReceiver, new IntentFilter(LiveBugManager
                            .REFRESH_ATTACHMENTS));
            presenter.refreshAttachments();
        }
        this.presenter = presenter;
    }

    @StringRes
    protected abstract int getCloseButtonContentDescription();

    @Override
    public void onStop() {
        super.onStop();
        if (getActivity() != null && presenter != null) {
            presenter.onStop();
            LocalBroadcastManager.getInstance(getActivity())
                    .unregisterReceiver(refreshAttachmentsBroadcastReceiver);
        }

        unregisterKeyboardObserver();
    }

    @Override
    public void onSaveInstanceState(@NotNull Bundle outState) {
        super.onSaveInstanceState(outState);
        if (presenter != null)
            presenter.onSaveInstanceState(outState);
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (presenter != null)
            presenter.onActivityResult(requestCode, resultCode, data);
    }

    @Override
    public void onViewStateRestored(@Nullable Bundle savedInstanceState) {
        super.onViewStateRestored(savedInstanceState);
        if (presenter != null)
            presenter.onRestoreInstanceState(savedInstanceState);
    }


    @Nullable
    protected abstract IBaseReportingView.Presenter createPresenter();

    @Override
    public void onClick(View v) {
        // mis-clicking prevention, using threshold of 1000 ms
        if (SystemClock.elapsedRealtime() - lastClickTime < 1000) {
            return;
        }
        lastClickTime = SystemClock.elapsedRealtime();

        int id = v.getId();

        if (id == R.id.instabug_attach_screenshot) {
            doIfVideoDoneEncoding(new Runnable() {
                @Override
                public void run() {
                    if (LiveBugManager.getInstance().getBug() != null) {
                        if (LiveBugManager.getInstance().getBug().getVisibleAttachmentsCount() < 4) {
                            if (presenter != null) {
                                presenter.takeScreenshot();
                            } else {
                                InstabugSDKLogger.v(Constants.LOG_TAG, MESSAGE_NULL_PRESENTER);
                            }
                        } else {
                            showAlertWhenUserReachMaxAttachmentsLimit();
                        }
                    } else {
                        InstabugSDKLogger.v(Constants.LOG_TAG, MESSAGE_NULL_BUG);
                    }
                }
            });
        } else if (id == R.id.instabug_attach_gallery_image) {
            doIfVideoDoneEncoding(new Runnable() {
                @Override
                public void run() {
                    if (LiveBugManager.getInstance().getBug() != null) {
                        if (LiveBugManager.getInstance().getBug().getVisibleAttachmentsCount() < 4) {
                            if (presenter != null) {
                                presenter.pickPhotoFromGallery();
                            } else {
                                InstabugSDKLogger.v(Constants.LOG_TAG, MESSAGE_NULL_PRESENTER);
                            }
                        } else {
                            showAlertWhenUserReachMaxAttachmentsLimit();
                        }
                    } else {
                        InstabugSDKLogger.v(Constants.LOG_TAG, MESSAGE_NULL_BUG);
                    }
                }
            });
        } else if (id == R.id.instabug_attach_video) {
            doIfVideoDoneEncoding(new Runnable() {
                @Override
                public void run() {
                    if (LiveBugManager.getInstance().getBug() != null) {
                        if (LiveBugManager.getInstance().getBug().getVisibleAttachmentsCount() < 4
                                && BugSettings.getInstance().getAttachmentsTypesParams().isAllowScreenRecording()) {
                            startScreenRecording();
                        } else {
                            showAlertWhenUserReachMaxAttachmentsLimit();
                        }
                    } else {
                        InstabugSDKLogger.v(Constants.LOG_TAG, MESSAGE_NULL_BUG);
                    }
                }
            });
        } else if (id == R.id.ib_bottomsheet_arrow_layout || id == R.id.arrow_handler) {
            hidKeyboard();
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (addAttachments != null && attachmentBottomSheetBehavior != null) {
                        if (attachmentBottomSheetBehavior.getState() == STATE_COLLAPSED) {
                            addAttachments.setVisibility(View.GONE);
                            attachmentBottomSheetBehavior.setState(STATE_EXPANDED);
                        } else {
                            attachmentBottomSheetBehavior.setState(STATE_COLLAPSED);
                        }
                    }
                }
            }, 200);
        } else if (id == R.id.instabug_add_attachment) {
            if (attachmentBottomSheetBehavior != null && attachmentBottomSheetBehavior.getState() == STATE_COLLAPSED) {
                hidKeyboard();
                new Handler().postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        if (findViewById(R.id.instabug_add_attachment) != null)
                            findViewById(R.id.instabug_add_attachment).setVisibility(View.GONE);
                        if (attachmentBottomSheetBehavior != null)
                            attachmentBottomSheetBehavior.setState(STATE_EXPANDED);
                    }
                }, 200);
            }
        } else if (id == R.id.instabug_text_view_repro_steps_disclaimer) {
            if (activityCallback != null) {
                activityCallback.openVisualUserStepsDisclaimerFragment();
            }
        }
    }

    private void doIfVideoDoneEncoding(Runnable runnable) {
        if (ExternalScreenRecordHelper.getInstance().isRecording()) {
            String warningMessage = getLocalizedString(R.string.instabug_str_video_encoder_busy)
                    + ", " + getLocalizedString(R.string.instabug_str_please_wait);
            if (getContext() != null) {
                Toast.makeText(getContext().getApplicationContext(), warningMessage, Toast.LENGTH_LONG).show();
            }
        } else {
            runnable.run();
        }
    }

    @RequiresApi(api = VERSION_CODES.LOLLIPOP)
    private void startScreenRecording() {
        //checks if the internal screen recorder is busy or not
        if (!ExternalScreenRecordHelper.getInstance().isRecording()) {
            requestAudioRecordingPermissions();
        } else {
            if (getContext() != null) {
                Toast.makeText(getContext().getApplicationContext(), R.string.instabug_str_video_encoder_busy, Toast
                        .LENGTH_SHORT).show();
            }
        }
    }

    /**
     * Handle grid onAttachmentClicked
     *
     * @param view       reference from the clicked view
     * @param attachment reference from attachment item
     */
    @Override
    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public void onAttachmentClicked(final View view, @NonNull final Attachment attachment) {
        clearFocusFromEditTexts();

        if (getActivity() != null)
            SystemServiceUtils.hideInputMethod(getActivity(), focusedView);

        final int id = view.getId();
        if (attachmentClickRunnable == null) {
            initAttachmentClickRunnable(view, attachment, id);
        }
        attachmentClickHandler.postDelayed(attachmentClickRunnable, 200);
    }

    private void initAttachmentClickRunnable(final View view, @NonNull final Attachment attachment, @IdRes final int id) {
        attachmentClickRunnable = new Runnable() {
            @Override
            public void run() {
                if (id == R.id.instabug_attachment_img_item || id == R.id.instabug_btn_image_edit_attachment) {
                    openAttachment(view, attachment);
                } else if (id == R.id.instabug_btn_remove_attachment) {
                    if (presenter != null)
                        presenter.removeAttachment(attachment);
                } else if (id == R.id.instabug_attachment_video_item && attachment.getLocalPath() != null) {
                    isVideoPlayBtnClicked = true;
                    startVideo(attachment);
                }
                if (attachmentClickHandler != null && attachmentClickRunnable != null) {
                    attachmentClickHandler.removeCallbacks(attachmentClickRunnable);
                }
                attachmentClickRunnable = null;
            }
        };
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void clearFocusFromEditTexts() {
        if (emailEditText != null) {
            emailEditText.clearFocus();
            emailEditText.setError(null);
        }
        if (messageEditText != null) {
            messageEditText.clearFocus();
            messageEditText.setError(null);
        }
    }

    @RequiresApi(api = VERSION_CODES.LOLLIPOP)
    private void requestAudioRecordingPermissions() {
        if (getActivity() == null) return;

        boolean isAudioPermissionGranted = ContextCompat.checkSelfPermission(getActivity(),
                Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;

        if (!isAudioPermissionGranted) {
            requestPermissions(
                    new String[]{Manifest.permission.RECORD_AUDIO},
                    REQUEST_SCREEN_RECORDING_MIC_PERMISSIONS);
        } else {
            requestVideoRecordingPermissions();
        }
    }

    @RequiresApi(api = VERSION_CODES.LOLLIPOP)
    @Override
    public void onRequestPermissionsResult(int requestCode, @NotNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode == REQUEST_SCREEN_RECORDING_MIC_PERMISSIONS) {//Video permission will be requested regardless the audio permission is granted or not.
            requestVideoRecordingPermissions();
        } else {
            super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        }
    }

    @RequiresApi(api = VERSION_CODES.LOLLIPOP)
    private void requestVideoRecordingPermissions() {
        if (VERSION.SDK_INT >= VERSION_CODES.LOLLIPOP && (getActivity() != null)) {
            MediaProjectionManager manager = (MediaProjectionManager) getActivity().getSystemService(MEDIA_PROJECTION_SERVICE);
            if (manager != null) {
                ReportingNavigator.requestRecordingPermission(manager, this);
            }
        }
    }

    private void showAlertWhenUserReachMaxAttachmentsLimit() {
        if (getActivity() != null) {
            new InstabugAlertDialog.Builder(getActivity())
                    .setTitle(getLocalizedString(R.string.instabug_str_alert_title_max_attachments))
                    .setMessage(getLocalizedString(R.string.instabug_str_alert_message_max_attachments))
                    .setPositiveButton(PlaceHolderUtils.getPlaceHolder(
                                    Key.BUG_ATTACHMENT_DIALOG_OK_BUTTON,
                                    getLocalizedString(R.string.instabug_str_ok))
                            , null)
                    .show();
        }
    }


    private void openAttachment(View view, @NonNull Attachment attachment) {
        if (presenter != null && !presenter.isAttachmentBeingRemoved(attachment)) {
            hidKeyboard();
            if (attachment.getLocalPath() == null) return;
            ImageView attachmentImageView = view.findViewById(R.id.instabug_img_attachment);
            if (attachmentImageView != null) {
                if (AccessibilityUtils.isTalkbackEnabled()) {
                    String title = attachmentImageView.getContentDescription().toString();
                    openStepPreviewFragment(attachment.getLocalPath(), title);
                } else {
                    if (presenter != null)
                        openAnnotationFragment(attachment, attachmentImageView, presenter.getTitle());
                }

            }
        }
    }

    private void openAnnotationFragment(Attachment attachment, ImageView attachmentImageView, String title) {
        if (attachment.getLocalPath() == null) return;
        notifyFragmentVisibilityChanged(false);
        FragmentTransaction transaction = null;
        if (getFragmentManager() != null) {
            transaction = getFragmentManager().beginTransaction();
        }
        Uri uri = Uri.fromFile(new File(attachment.getLocalPath()));
        String transitionName = ViewCompat.getTransitionName(attachmentImageView);
        if (transitionName != null && transaction != null) {
            transaction.addSharedElement(attachmentImageView, transitionName);
        }
        BitmapDrawable bitmapDrawable = (BitmapDrawable) attachmentImageView.getDrawable();
        if (bitmapDrawable != null && transaction != null) {
            transaction.replace(R.id.instabug_fragment_container,
                            BugAnnotationFragment.newInstance(title, uri, attachment.getName())
                            , "annotation")
                    .addToBackStack("annotation")
                    .commitAllowingStateLoss();
        }
    }

    private void openStepPreviewFragment(String attachmentPath, String title) {
        VisualUserStepArgs args = new VisualUserStepArgs(presenter != null ? presenter.getTitle() : title, attachmentPath, title);
        if (activityCallback != null) {
            activityCallback.openStepPreviewFragment(args);
        }
    }

    private void hidKeyboard() {
        if (getActivity() != null) {
            SystemServiceUtils.hideInputMethod(getActivity(), focusedView);
        }
    }

    @Override
    public void showToast(CharSequence text) {
        if (getActivity() != null) {
            Toast.makeText(getActivity().getApplicationContext(), text, Toast.LENGTH_SHORT).show();
        }
    }

    @Override
    public void showToast(@StringRes int resId) {
        if (getActivity() != null) {
            Toast.makeText(getActivity().getApplicationContext(), resId, Toast.LENGTH_SHORT).show();
        }
    }

    @Override
    public void setAttachments(List<Attachment> attachments) {
        attachmentsAdapter.clearAttachments();
        // finding the last image and animate its helper icons
        int lastImgIndex = -1;
        for (int i = 0; i < attachments.size(); i++) {
            if (attachments.get(i).getType() != null) {
                if (attachments.get(i).getType().equals(Attachment.Type.MAIN_SCREENSHOT)
                        || attachments.get(i).getType().equals(Attachment.Type.EXTRA_IMAGE)
                        || attachments.get(i).getType().equals(Attachment.Type.GALLERY_IMAGE)
                        || attachments.get(i).getType().equals(Attachment.Type.AUDIO)
                        || attachments.get(i).getType().equals(Attachment.Type.EXTRA_VIDEO)
                        || attachments.get(i).getType().equals(Attachment.Type.GALLERY_VIDEO)
                        || attachments.get(i).getType().equals(Attachment.Type.AUTO_SCREEN_RECORDING_VIDEO)) {
                    if (attachments.get(i).getType().equals(Attachment.Type.GALLERY_VIDEO)) {
                        attachments.get(i).setVideoEncoded(true);
                    }
                    attachmentsAdapter.addAttachment(attachments.get(i));
                }
                if (attachments.get(i).getType().equals(Attachment.Type.EXTRA_VIDEO) || attachments.get(i).getType().equals(Attachment.Type.GALLERY_VIDEO)) {
                    if (LiveBugManager.getInstance().getBug() != null) {
                        LiveBugManager.getInstance().getBug().setHasVideo(true);
                    }
                }
            }
        }
        for (int i = 0; i < attachmentsAdapter.getDataset().size(); i++) {
            if (attachmentsAdapter.getDataset().get(i).getType() != null) {
                if (attachmentsAdapter.getDataset().get(i).getType().equals(Attachment.Type.MAIN_SCREENSHOT)
                        || attachmentsAdapter.getDataset().get(i).getType().equals(Attachment.Type.GALLERY_IMAGE)
                        || attachmentsAdapter.getDataset().get(i).getType().equals(Attachment.Type.EXTRA_IMAGE)) {
                    lastImgIndex = i;
                }
            }
        }

        attachmentsAdapter.setLastImageIndex(lastImgIndex);
        attachmentsList.setAdapter(attachmentsAdapter);
        attachmentsAdapter.notifyDataSetChanged();

        if ((InstabugCore.getFeatureState(IBGFeature.MULTIPLE_ATTACHMENTS)
                == Feature.State.ENABLED)
                && BugSettings.getInstance().isAddAttachmentsButtonEnable()) {
            if (findViewById(R.id.instabug_attachment_bottom_sheet) != null)
                findViewById(R.id.instabug_attachment_bottom_sheet).setVisibility(View.VISIBLE);
        } else {
            if (findViewById(R.id.instabug_attachment_bottom_sheet) != null)
                findViewById(R.id.instabug_attachment_bottom_sheet).setVisibility(View.GONE);
        }

        // get last attachment coordinates for animation
        attachmentsList.post(new Runnable() {
            @Override
            public void run() {
                if (attachmentsList != null && attachmentsList.getLayoutManager() != null) {
                    View viewItem = attachmentsList.getLayoutManager().findViewByPosition(attachmentsAdapter.getItemCount() - 1);
                    if (viewItem != null && getActivity() != null) {
                        Rect rectf = new Rect();
                        viewItem.getGlobalVisibleRect(rectf);

                        DisplayMetrics displayMetrics = DeviceStateProvider.getDisplayMetrics(getActivity());
                        float centerX = ((rectf.right + rectf.left) / 2.0f);
                        float centerY = ((rectf.top + rectf.bottom) / 2.0f);

                        listener.loadBitmapAnimation(
                                (centerX / displayMetrics.widthPixels), // center pivotX
                                (centerY / displayMetrics.heightPixels)); // center pivotY
                    }
                }
            }
        });
        startPostponedEnterTransition();

    }

    @Override
    public boolean isVideoProgressBarVisible() {
        return attachmentsAdapter.getVideoProgressBar() != null && attachmentsAdapter
                .getVideoProgressBar().getVisibility() == View.VISIBLE;
    }

    @Override
    public boolean isVideoPlayImageViewVisible() {
        return attachmentsAdapter.getVideoPlayImageView() != null
                && attachmentsAdapter.getVideoPlayImageView().getVisibility() == View.VISIBLE;
    }

    @Override
    public void setVideoProgressBarVisibility(boolean visibility) {
        if (attachmentsAdapter.getVideoProgressBar() != null) {
            if (visibility) {
                attachmentsAdapter.getVideoProgressBar().setVisibility(View.VISIBLE);
            } else {
                attachmentsAdapter.getVideoProgressBar().setVisibility(View.GONE);
            }
        }
    }

    @Override
    public void setVideoPlayImageViewVisibility(boolean visibility) {
        if (attachmentsAdapter.getVideoPlayImageView() != null) {
            if (visibility) {
                attachmentsAdapter.getVideoPlayImageView().setVisibility(View.VISIBLE);
            } else {
                attachmentsAdapter.getVideoPlayImageView().setVisibility(View.GONE);
            }
        }
    }

    @Override
    public boolean isVideoPlayButtonClicked() {
        return isVideoPlayBtnClicked;
    }

    @Override
    public void openActivityForResult(Intent intent, int requestCode) {
        startActivityForResult(intent, requestCode);
    }

    @Override
    public void notifyFragmentVisibilityChanged(boolean isVisible) {
        if (getFragmentManager() != null && (getFragmentManager().findFragmentById(
                R.id.instabug_fragment_container) instanceof FragmentVisibilityChangedListener)) {
            ((FragmentVisibilityChangedListener) getFragmentManager().findFragmentById(
                    R.id.instabug_fragment_container)).onVisibilityChanged(isVisible);
        }
    }

    @Override
    public void startVideo(Attachment attachment) {
        if (presenter != null && !presenter.isAttachmentBeingRemoved(attachment)) {
            String videoPath = attachment.getLocalPath();
            if (videoPath != null && getFragmentManager() != null) {
                FragmentTransaction transaction =
                        getFragmentManager().beginTransaction();
                transaction.add(R.id.instabug_fragment_container, VideoPlayerFragment.newInstance
                                (videoPath),
                        "video_player").addToBackStack("play video").commitAllowingStateLoss();
            } else {
                if (!isVideoProgressBarVisible()) {
                    setVideoProgressBarVisibility(true);
                }

                if (isVideoPlayImageViewVisible()) {
                    setVideoPlayImageViewVisibility(false);
                }
            }
        }
    }

    @Override
    public void navigateToSuccessFragment() {
        hidKeyboard();
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                if (BugSettings.getInstance().successDialogEnabled() &&
                        !ServiceLocator.getConfigurationsProvider().isBugReportingUsageExceeded()) {
                    Activity ownerActivity = getActivity();
                    if (ownerActivity != null) {
                        Intent intent = new Intent(ownerActivity, InstabugThanksActivity.class);
                        ownerActivity.startActivityForResult(intent, SHOW_THANK_YOU_ACTIVITY);
                    }

                } else {
                    if (activityCallback != null)
                        listener.onFeedbackFragmentDismissed();
                }
            }
        }, 200);
    }

    @Override
    public void navigateToExtraFieldsFragment() {
        IBaseReportingView.Presenter presenter = this.presenter;
        if (presenter != null && getFragmentManager() != null) {
            String title = presenter.getTitle();
            ReportingNavigator.navigateToExtraFields(getFragmentManager(), title);
        }
        this.presenter = presenter;
    }

    @Override
    public void startGalleryPicker() {
        ReportingNavigator.navigateToPhotoPicker(this);
    }

    @Override
    public void showPreparingDialog() {
        if (preparingProgressDialog != null) {
            if (!preparingProgressDialog.isShowing()) {
                if (getFragmentManager() != null && !getFragmentManager().isStateSaved()) {
                    preparingProgressDialog.show();
                }
            }
        } else {
            if (getActivity() != null && getFragmentManager() != null) {
                preparingProgressDialog = new IBGProgressDialog.Builder()
                        .setMessage(getLocalizedString(R.string
                                .instabug_str_dialog_message_preparing))
                        .build(getActivity());
                if (!getFragmentManager().isStateSaved()) {
                    preparingProgressDialog.show();
                }
            }
        }
    }

    @Override
    public void showMediaFileSizeAlert() {
        if (getActivity() != null) {
            new InstabugAlertDialog.Builder(getActivity())
                    .setTitle(getLocalizedString(R.string.instabug_str_bugreport_file_size_limit_warning_title))
                    .setMessage(getLocalizedString(R.string.instabug_str_bugreport_file_size_limit_warning_message,
                            BaseReportingPresenter.MAX_FILE_SIZE_IN_MB))
                    .setPositiveButton(getLocalizedString(R.string.instabug_str_ok), (dialogInterface, i) -> {
                        dialogInterface.dismiss();
                    }).show();
        }
    }

    @Override
    public void showVideoLengthAlert() {
        if (getActivity() != null) {
            new InstabugAlertDialog.Builder(getActivity())
                    .setTitle(getLocalizedString(R.string.instabug_str_video_length_limit_warning_title))
                    .setMessage(getLocalizedString(R.string.instabug_str_video_length_limit_warning_message))
                    .setPositiveButton(getLocalizedString(R.string.instabug_str_ok), (dialogInterface, i) -> {
                        dialogInterface.dismiss();
                    }).show();
        }
    }

    @Override
    public void dismissPreparingDialog() {
        if (preparingProgressDialog != null && preparingProgressDialog.isShowing()) {
            preparingProgressDialog.dismiss();
        }
    }

    @Override
    public void notifyAttachmentRemoved(Attachment attachment) {
        if (attachmentsAdapter != null) {
            attachmentsAdapter.removeAttachment(attachment);
            attachmentsAdapter.notifyDataSetChanged();
        }
    }

    @Override
    public void showEmailError(String message) {
        emailEditText.requestFocus();
        emailEditText.setError(message);
    }

    @Override
    public void showCommentError(String message) {
        messageEditText.requestFocus();
        messageEditText.setError(message);
    }

    @Override
    public String getEnteredEmail() {
        return emailEditText.getText().toString();
    }

    @Override
    public void setEnteredEmail(String enteredEmail) {
        emailEditText.setText(enteredEmail);
    }

    @Override
    public void showVisualUserStepDisclaimer(Spanned disclaimer, final String body) {
        reproStepsDisclaimerTextView.setVisibility(View.VISIBLE);
        reproStepsDisclaimerTextView.setText(disclaimer);
        if (AccessibilityUtils.isTalkbackEnabled()) {
            ViewCompat.setAccessibilityDelegate(reproStepsDisclaimerTextView, new AccessibilityDelegateCompat() {
                @Override
                public void onInitializeAccessibilityNodeInfo(View host, AccessibilityNodeInfoCompat info) {
                    super.onInitializeAccessibilityNodeInfo(host, info);
                    info.setContentDescription(body);
                    // The result announcement will be "Double tap to edit steps list"
                    info.addAction(new AccessibilityNodeInfoCompat.AccessibilityActionCompat(AccessibilityNodeInfoCompat.ACTION_CLICK,
                            getLocalizedString(R.string.ibg_bug_report_visual_steps_disclaimer_action_description)));
                }
            });
        }
    }

    @Override
    public void hideVisualUserStepDisclaimer() {
        reproStepsDisclaimerTextView.setVisibility(View.GONE);
    }

    @Override
    public void showInstabugDisclaimer(Spanned spanned) {
        disclaimerTextView.setVisibility(View.VISIBLE);
        disclaimerTextView.setText(spanned);
        disclaimerTextView.setMovementMethod(LinkMovementMethod.getInstance());
    }

    @Override
    public void hideInstabugDisclaimer() {
        disclaimerTextView.setVisibility(View.GONE);
    }

    private void initRefreshAttachmentsReceiver() {
        refreshAttachmentsBroadcastReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                InstabugSDKLogger.v(Constants.LOG_TAG, "Refreshing Attachments");
                if (getActivity() != null) {
                    if (presenter != null) {
                        presenter.refreshAttachments();
                    }
                }
            }
        };
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        if (bugReportingInputsContainer != null) {
            bugReportingInputsContainer.clearFocus();
            bugReportingInputsContainer.removeAllViews();
        }
        enabledAttachmentCount = 0;
        disclaimerTextView = null;
        emailEditText = null;
        messageEditText = null;
        reproStepsDisclaimerTextView = null;
        scrollView = null;
        arrowHandler = null;
        attachmentsList = null;
        attachmentBottomSheetBehavior = null;
        attachmentsAdapter = null;
        addAttachments = null;
        bugReportingInputsContainer = null;
        focusedView = null;
        consentsRecyclerView = null;
        submitMenuItem = null;
    }

    @Override
    public void onDestroy() {
        if (attachmentClickRunnable != null && attachmentClickHandler != null) {
            attachmentClickHandler.removeCallbacks(attachmentClickRunnable);
            attachmentClickRunnable = null;
        }

        super.onDestroy();
        teardown();
    }

    private static void teardown() {
        lastState = -1;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        inflater.inflate(R.menu.instabug_bug_reporting, menu);

        boolean hasNextScreen = false;
        if (presenter != null) {
            hasNextScreen = presenter.hasExtendedReport();
        }
        final MenuItem nextMenuItem = menu.findItem(R.id.instabug_bugreporting_next);
        final MenuItem sendMenuItem = menu.findItem(R.id.instabug_bugreporting_send);
        submitMenuItem = hasNextScreen ? nextMenuItem : sendMenuItem;
        setSubmitEnabledState();
        if (hasNextScreen) {
            if (sendMenuItem != null) {
                sendMenuItem.setVisible(false);
            }

            if (nextMenuItem != null) {
                nextMenuItem.setVisible(true);
                if (AccessibilityUtils.isTalkbackEnabled()) {
                    nextMenuItem.setTitle(R.string.ibg_bug_report_next_btn_content_description);
                }

                Drawable nextMenuItemIcon = nextMenuItem.getIcon();
                if (getContext() != null && nextMenuItemIcon != null && LocaleUtils.isRTL(InstabugCore.getLocale(getContext()))) {
                    nextMenuItem.setIcon(
                            getRotateDrawable(nextMenuItemIcon, 180));
                }
            }
        } else {
            nextMenuItem.setVisible(false);
            sendMenuItem.setVisible(true);
            sendMenuItem.setTitle(getSendButtonContentDescriptionTitle());
            Drawable sendBugReportIcon = sendMenuItem.getIcon();
            if (getContext() != null && sendBugReportIcon != null && LocaleUtils.isRTL(InstabugCore.getLocale(getContext()))) {
                menu.findItem(R.id.instabug_bugreporting_send).setIcon(
                        getRotateDrawable(sendBugReportIcon, 180));
            }
        }
    }

    protected abstract int getSendButtonContentDescriptionTitle();

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // mis-clicking prevention, using threshold of 1000 ms
        IBaseReportingView.Presenter presenter = this.presenter;
        if (SystemClock.elapsedRealtime() - lastClickTime < 1000) {
            return false;
        }
        lastClickTime = SystemClock.elapsedRealtime();

        if (item.getItemId() == R.id.instabug_bugreporting_next && presenter != null) {
            presenter.onSendClicked();
        } else if (item.getItemId() == R.id.instabug_bugreporting_send && presenter != null) {
            if (getFragmentManager() != null) {
                for (Fragment fragment : getFragmentManager().getFragments()) {
                    if (fragment instanceof ExtraFieldsFragment) {
                        // do nothing
                        return super.onOptionsItemSelected(item);
                    }
                }
            }
            presenter.onSendClicked();
        } else if (item.getItemId() == android.R.id.home) {
            if (getActivity() != null)
                getActivity().onBackPressed();
        }
        this.presenter = presenter;
        return false;
    }

    private void registerKeyboardObserver() {
        if (getActivity() != null) {
            getActivity().getWindow().getDecorView().getViewTreeObserver()
                    .addOnGlobalLayoutListener(keyboardObserver);
        }
    }

    private void unregisterKeyboardObserver() {
        if (VERSION.SDK_INT >= VERSION_CODES.JELLY_BEAN) {
            if (getActivity() != null) {
                getActivity().getWindow().getDecorView().getViewTreeObserver()
                        .removeOnGlobalLayoutListener(keyboardObserver);
            }
        }
    }

    @Override
    public void requestMediaProjectionPermission() {
        if (getActivity() != null) {
            RequestPermissionActivityLauncher.start(getActivity(), false, false, null);
        }
    }

    public void openVideoRecorder() {
        if (presenter == null) return;

        presenter.openVideoRecorder();
    }

    @Override
    public void onFocusChange(View view, boolean hasFocus) {
        if (hasFocus) {
            focusedView = view;
        }
    }


    public interface Callbacks {
        void onFeedbackFragmentDismissed();

        void loadBitmapAnimation(final float pivotXValue, final float pivotYValue);

    }

    @Override
    public String getLocalizedString(@StringRes int resourceId) {
        return LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(getContext()), resourceId, getContext());
    }

    @Override
    public void showUserConsents(@Nullable List<UserConsent> consents) {
        if (consents == null || consentsRecyclerView == null) return;
        ConsentsRecyclerViewAdapter consentsRecyclerViewAdapter = new ConsentsRecyclerViewAdapter(consents, this::setSubmitEnabledState, theme);
        consentsRecyclerView.setAdapter(consentsRecyclerViewAdapter);
    }

    private void setSubmitEnabledState() {
        if (submitMenuItem != null && presenter != null)
            submitMenuItem.setEnabled(presenter.hasValidConsents());
    }

    public String getLocalizedString(@StringRes int resourceId, Object... formatArgs) {
        return LocaleUtils.getLocaleStringResource(InstabugCore.getLocale(getContext()), resourceId, getContext(), formatArgs);
    }
}