package com.instabug.bug.screenshot.viewhierarchy;

import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Rect;
import android.os.Build;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.library.R;
import com.instabug.library.internal.utils.stability.execution.ReturnableExecutable;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class ViewHierarchyInspector {

    private static final String KEY_RESOURCE_ID = "resource_id";
    private static final String KEY_HEIGHT = "height";
    private static final String KEY_WIDTH = "width";
    private static final String KEY_PADDING_TOP = "padding_top";
    private static final String KEY_PADDING_BOTTOM = "padding_bottom";
    private static final String KEY_PADDING_RIGHT = "padding_right";
    private static final String KEY_PADDING_LEFT = "padding_left";
    private static final String KEY_PADDING_END = "padding_end";
    private static final String KEY_PADDING_START = "padding_start";
    private static final String KEY_X = "x";
    private static final String KEY_Y = "y";
    private static final String KEY_W = "w";
    private static final String KEY_H = "h";
    private static final String KEY_VISIBILITY = "visibility";
    private static final String KEY_MARGIN_TOP = "margin_top";
    private static final String KEY_MARGIN_BOTTOM = "margin_bottom";
    private static final String KEY_MARGIN_LEFT = "margin_left";
    private static final String KEY_MARGIN_RIGHT = "margin_right";
    private static final String KEY_LAYOUT_GRAVITY = "gravity";

    private static final int DEFAULT_UNIDENTIFIED_VIEW_ID = -1;

    public enum Action {
        STARTED, FAILED, COMPLETED
    }

    public static ReturnableExecutable<ViewHierarchy> getInspectRootViewExecutable(final ViewHierarchy viewHierarchy) {
        return new ReturnableExecutable<ViewHierarchy>() {
            @Override
            public ViewHierarchy execute() {
                return inspectVisibleView(viewHierarchy);
            }
        };
    }

    private static ViewHierarchy inspectVisibleView(final ViewHierarchy viewHierarchy) {
        if (viewHierarchy.getView() != null && viewHierarchy.getView().getVisibility() == View.VISIBLE)
            try {
                viewHierarchy.setType(ViewHierarchyInspector.inspectViewType(viewHierarchy.getView()));
                viewHierarchy.setIconIdentifier(obtainViewIcon(viewHierarchy.getView()));
                viewHierarchy.setProperties(ViewHierarchyInspector.inspectViewProperties(viewHierarchy.getView()));
                viewHierarchy.setOriginalRect(ViewHierarchyInspector.inspectViewOriginalRect(viewHierarchy.getView()));
                viewHierarchy.setVisibleRect(ViewHierarchyInspector.inspectViewVisibleRect(viewHierarchy));
                viewHierarchy.setFrame(ViewHierarchyInspector.inspectViewFrame(viewHierarchy));
                // inspect view children if exist, not we inspect view children before capture its screenshot
                if (viewHierarchy.getView() instanceof ViewGroup) {
                    viewHierarchy.setHasChildren(true);
                    inspectViewChildren(viewHierarchy);
                } else {
                    viewHierarchy.setHasChildren(false);
                }
            } catch (JSONException e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "inspect view hierarchy got error: " + e.getMessage() + ",View hierarchy id:" + viewHierarchy.getId(), e);
            }
        return viewHierarchy;
    }

    private static void inspectViewChildren(ViewHierarchy parentViewHierarchy) {
        if (parentViewHierarchy.getView() instanceof ViewGroup) {
            ViewGroup parent = (ViewGroup) parentViewHierarchy.getView();
            for (int i = 0; i < parent.getChildCount(); i++) {
                if (parent.getChildAt(i).getId() != R.id.instabug_extra_screenshot_button
                        && parent.getChildAt(i).getId() != R.id.instabug_floating_button) {
                    ViewHierarchy childViewHierarchy = new ViewHierarchy();
                    childViewHierarchy.setRoot(false);
                    childViewHierarchy.setId(parentViewHierarchy.getId() + "-" + i);
                    childViewHierarchy.setView(parent.getChildAt(i));
                    childViewHierarchy.setParent(parentViewHierarchy);
                    childViewHierarchy.setScale(parentViewHierarchy.getScale());
                    parentViewHierarchy.addNode(inspectVisibleView(childViewHierarchy));
                }
            }
        }
    }

    private static String obtainViewIcon(View view) {
        switch (view.getClass().getSimpleName()) {
            case "LinearLayout":
                LinearLayout linearLayout = (LinearLayout) view;
                if (linearLayout.getOrientation() == LinearLayout.HORIZONTAL)
                    return "HorizontalLinearLayout";
                else
                    return "VerticalLinearLayout";
            case "RelativeLayout":
                return "RelativeLayout";
            case "FrameLayout":
                return "FrameLayout";
            case "TableLayout":
                return "TableLayout";
            case "TableRow":
                return "TableRow";
            case "Button":
                return "Button";
            case "ImageButton":
                return "ImageButton";
            case "ImageView":
                return "ImageView";
            case "EditText":
                return "EditText";
            case "TextView":
                return "TextView";
            case "AutoCompleteTextView":
                return "AutoCompleteTextView";
            case "MultiAutoCompleteTextView":
                return "MultiAutoCompleteTextView";
            case "ScrollView":
                return "ScrollView";
            case "HorizontalScrollView":
                return "HorizontalScrollView";
            case "GridView":
                return "GridView";
            case "ListView":
                return "ListView";
            case "ProgressBar":
                return "ProgressBar";
            case "RadioButton":
                return "RadioButton";
            case "VideoView":
                return "VideoView";
            case "WebView":
                return "WebView";
            case "SearchView":
                return "SearchView";
            case "ToggleButton":
                return "ToggleButton";
            default:
                return "default";
        }
    }

    private static String inspectViewType(View view) {
        return view.getClass().getSimpleName();
    }

    private static JSONObject inspectViewProperties(View view) throws JSONException {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put(KEY_RESOURCE_ID, inspectViewResourceId(view.getContext(), view.getId()))
                .put(KEY_HEIGHT, view.getHeight())
                .put(KEY_WIDTH, view.getWidth())
                .put(KEY_PADDING_TOP, view.getPaddingTop())
                .put(KEY_PADDING_BOTTOM, view.getPaddingBottom())
                .put(KEY_PADDING_RIGHT, view.getPaddingRight())
                .put(KEY_PADDING_LEFT, view.getPaddingLeft())
                .put(KEY_VISIBILITY, view.getVisibility());
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            jsonObject.put(KEY_PADDING_END, view.getPaddingEnd())
                    .put(KEY_PADDING_START, view.getPaddingStart());
        }
        jsonObject.put(KEY_X, view.getX()).put(KEY_Y, view.getY());
        if (view.getLayoutParams() instanceof LinearLayout.LayoutParams)
            inspectLinearLayoutParams(((LinearLayout.LayoutParams) view.getLayoutParams()), jsonObject);
        else if (view.getLayoutParams() instanceof FrameLayout.LayoutParams)
            inspectFrameLayoutParams(((FrameLayout.LayoutParams) view.getLayoutParams()), jsonObject);
        else if (view.getLayoutParams() instanceof RelativeLayout.LayoutParams)
            inspectRelativeLayoutParams(view.getContext(), ((RelativeLayout.LayoutParams) view.getLayoutParams()), jsonObject);

        return jsonObject;
    }

    private static String inspectViewResourceId(Context context, int id) {

        if (id == DEFAULT_UNIDENTIFIED_VIEW_ID)
            return String.valueOf(id);

        try {
            return context != null && context.getResources() != null && context.getResources().getResourceEntryName(id) != null ?
                    context.getResources().getResourceEntryName(id) : String.valueOf(id);
        } catch (Resources.NotFoundException e) {
            return String.valueOf(id);
        }
    }

    private static void inspectLinearLayoutParams(LinearLayout.LayoutParams layoutParams, JSONObject jsonObject) throws JSONException {
        jsonObject.put(KEY_LAYOUT_GRAVITY, layoutParams.gravity)
                .put(KEY_MARGIN_TOP, layoutParams.topMargin)
                .put(KEY_MARGIN_BOTTOM, layoutParams.bottomMargin)
                .put(KEY_MARGIN_LEFT, layoutParams.leftMargin)
                .put(KEY_MARGIN_RIGHT, layoutParams.rightMargin);
    }

    private static void inspectFrameLayoutParams(FrameLayout.LayoutParams layoutParams, JSONObject jsonObject) throws JSONException {
        jsonObject.put(KEY_LAYOUT_GRAVITY, layoutParams.gravity)
                .put(KEY_MARGIN_TOP, layoutParams.topMargin)
                .put(KEY_MARGIN_BOTTOM, layoutParams.bottomMargin)
                .put(KEY_MARGIN_LEFT, layoutParams.leftMargin)
                .put(KEY_MARGIN_RIGHT, layoutParams.rightMargin);
    }

    private static void inspectRelativeLayoutParams(Context context, RelativeLayout.LayoutParams layoutParams, JSONObject jsonObject) throws JSONException {
        jsonObject.put(KEY_MARGIN_TOP, layoutParams.topMargin)
                .put(KEY_MARGIN_BOTTOM, layoutParams.bottomMargin)
                .put(KEY_MARGIN_LEFT, layoutParams.leftMargin)
                .put(KEY_MARGIN_RIGHT, layoutParams.rightMargin);
        int[] rules = layoutParams.getRules();
        for (int i = 0; i < rules.length; i++) {
            String subject;
            if (rules[i] > 0) {
                subject = inspectViewResourceId(context, rules[i]);
            } else {
                subject = String.valueOf(rules[i]);
            }
            jsonObject.put(inspectRelativeLayoutParamRuleVerb(i), subject);
        }
    }

    private static String inspectRelativeLayoutParamRuleVerb(int ruleId) {
        switch (ruleId) {
            case RelativeLayout.LEFT_OF:
                return "leftOf";
            case RelativeLayout.RIGHT_OF:
                return "rightOf";
            case RelativeLayout.ABOVE:
                return "above";
            case RelativeLayout.BELOW:
                return "below";
            case RelativeLayout.ALIGN_BASELINE:
                return "alignBaseline";
            case RelativeLayout.ALIGN_LEFT:
                return "alignLeft";
            case RelativeLayout.ALIGN_TOP:
                return "alignTop";
            case RelativeLayout.ALIGN_RIGHT:
                return "alignRight";
            case RelativeLayout.ALIGN_BOTTOM:
                return "alignBottom";
            case RelativeLayout.ALIGN_PARENT_LEFT:
                return "alignParentLeft";
            case RelativeLayout.ALIGN_PARENT_TOP:
                return "alignParentTop";
            case RelativeLayout.ALIGN_PARENT_RIGHT:
                return "alignParentRight";
            case RelativeLayout.ALIGN_PARENT_BOTTOM:
                return "alignParentBottom";
            case RelativeLayout.CENTER_IN_PARENT:
                return "centerInParent";
            case RelativeLayout.CENTER_HORIZONTAL:
                return "centerHorizontal";
            case RelativeLayout.CENTER_VERTICAL:
                return "centerVertical";
            case RelativeLayout.START_OF:
                return "startOf";
            case RelativeLayout.ALIGN_START:
                return "alignStart";
            case RelativeLayout.ALIGN_END:
                return "alignEnd";
            case RelativeLayout.ALIGN_PARENT_START:
                return "alignParentStart";
            case RelativeLayout.ALIGN_PARENT_END:
                return "alignParentEnd";
            default:
                return "notIdentified";
        }
    }

    private static Rect inspectViewOriginalRect(View view) {
        int[] locationOnScreen = new int[2];
        view.getLocationOnScreen(locationOnScreen);
        return new Rect(locationOnScreen[0],
                locationOnScreen[1],
                locationOnScreen[0] + view.getWidth(),
                locationOnScreen[1] + view.getHeight());
    }

    @Nullable
    private static Rect inspectViewVisibleRect(ViewHierarchy viewHierarchy) {
        if (viewHierarchy.isRoot()) {
            return viewHierarchy.getOriginalRect();
        } else {
            if (viewHierarchy.getOriginalRect() != null && viewHierarchy.getParent() != null) {
                Rect viewVisibleRect = new Rect(
                        viewHierarchy.getOriginalRect().left,
                        viewHierarchy.getOriginalRect().top,
                        viewHierarchy.getOriginalRect().right,
                        viewHierarchy.getOriginalRect().bottom);
                Rect parentAvailableVisibleRect = new Rect(
                        inspectViewAvailableX(viewHierarchy.getParent()),
                        inspectViewAvailableY(viewHierarchy.getParent()),
                        inspectViewAvailableRight(viewHierarchy.getParent()),
                        inspectViewAvailableBottom(viewHierarchy.getParent()));
                if (viewVisibleRect.intersect(parentAvailableVisibleRect)) {
                    return viewVisibleRect;
                } else {
                    return new Rect(0, 0, 0, 0);
                }
            }
        }
        return new Rect(0, 0, 0, 0);
    }

    private static int inspectViewAvailableX(ViewHierarchy viewHierarchy) {
        int visibleLeft = 0, paddingLeft = 0, originalLeft = 0;
        if (viewHierarchy.getVisibleRect() != null) {
            visibleLeft = viewHierarchy.getVisibleRect().left;
        }
        if (viewHierarchy.getView() != null) {
            paddingLeft = viewHierarchy.getView().getPaddingLeft();
        }
        if (viewHierarchy.getOriginalRect() != null) {
            originalLeft = viewHierarchy.getOriginalRect().left;
        }
        if (paddingLeft == 0) {
            return visibleLeft;
        } else {
            return Math.max(visibleLeft, (originalLeft + paddingLeft));
        }
    }

    private static int inspectViewAvailableY(ViewHierarchy viewHierarchy) {
        int visibleTop = 0, paddingTop = 0, originalTop = 0;
        if (viewHierarchy.getVisibleRect() != null) {
            visibleTop = viewHierarchy.getVisibleRect().top;
        }
        if (viewHierarchy.getView() != null) {
            paddingTop = viewHierarchy.getView().getPaddingTop();
        }
        if (viewHierarchy.getOriginalRect() != null) {
            originalTop = viewHierarchy.getOriginalRect().top;
        }
        if (paddingTop == 0) {
            return visibleTop;
        } else {
            return Math.max(visibleTop, (originalTop + paddingTop));
        }
    }

    private static int inspectViewAvailableRight(ViewHierarchy viewHierarchy) {
        int visibleRight = 0, paddingRight = 0, originalRight = 0;
        if (viewHierarchy.getVisibleRect() != null) {
            visibleRight = viewHierarchy.getVisibleRect().right;
        }
        if (viewHierarchy.getView() != null) {
            paddingRight = viewHierarchy.getView().getPaddingRight();
        }
        if (viewHierarchy.getOriginalRect() != null) {
            originalRight = viewHierarchy.getOriginalRect().right;
        }
        if (paddingRight == 0) {
            return visibleRight;
        } else {
            return Math.min(visibleRight, (originalRight - paddingRight));
        }
    }

    private static int inspectViewAvailableBottom(ViewHierarchy viewHierarchy) {
        int visibleBottom = 0, paddingBottom = 0, originalBottom = 0;
        if (viewHierarchy.getVisibleRect() != null) {
            visibleBottom = viewHierarchy.getVisibleRect().bottom;
        }
        if (viewHierarchy.getView() != null) {
            paddingBottom = viewHierarchy.getView().getPaddingBottom();
        }
        if (viewHierarchy.getOriginalRect() != null) {
            originalBottom = viewHierarchy.getOriginalRect().bottom;
        }
        if (paddingBottom == 0) {
            return visibleBottom;
        } else {
            return Math.min(visibleBottom, (originalBottom - paddingBottom));
        }
    }

    @Nullable
    private static JSONObject inspectViewFrame(ViewHierarchy viewHierarchy) throws JSONException {
        if (viewHierarchy.getVisibleRect() != null) {
            return new JSONObject().put(KEY_X, viewHierarchy.getVisibleRect().left / viewHierarchy.getScale())
                    .put(KEY_Y, viewHierarchy.getVisibleRect().top / viewHierarchy.getScale())
                    .put(KEY_W, viewHierarchy.getVisibleRect().width() / viewHierarchy.getScale())
                    .put(KEY_H, viewHierarchy.getVisibleRect().height() / viewHierarchy.getScale());
        }
        return null;
    }

    public static JSONObject inspectRootViewFrame(Activity activity, int scaleFactor) throws JSONException {
        return new JSONObject().put(KEY_W, activity.getWindow().getDecorView().getWidth() / scaleFactor)
                .put(KEY_H, activity.getWindow().getDecorView().getHeight() / scaleFactor);
    }

    public static List<ViewHierarchy> convertViewHierarchyToList(ViewHierarchy viewHierarchy) {
        ArrayList<ViewHierarchy> viewHierarchies = new ArrayList<>();
        if (viewHierarchy != null) {
            viewHierarchies.add(viewHierarchy);
            if (viewHierarchy.hasChildren()) {
                for (ViewHierarchy childViewHierarchy : viewHierarchy.getNodes()) {
                    viewHierarchies.addAll(convertViewHierarchyToList(childViewHierarchy));
                }
            }
        }
        return viewHierarchies;
    }
}
