package com.instabug.bug.screenshot

import android.app.Activity
import android.content.Context
import com.instabug.bug.Constants
import com.instabug.bug.LiveBugManager
import com.instabug.bug.model.Bug
import com.instabug.bug.screenshot.viewhierarchy.ViewHierarchy
import com.instabug.bug.screenshot.viewhierarchy.ViewHierarchyInspector
import com.instabug.bug.screenshot.viewhierarchy.utilities.BitmapUtils
import com.instabug.bug.screenshot.viewhierarchy.utilities.ViewHierarchyDiskUtils
import com.instabug.bug.screenshot.viewhierarchy.utilities.ViewHierarchyInspectorEventBus
import com.instabug.library.R
import com.instabug.library.instacapture.screenshot.FieldHelper
import com.instabug.library.internal.storage.DiskUtils
import com.instabug.library.internal.utils.stability.execution.ReturnableExecutable
import com.instabug.library.model.Attachment
import com.instabug.library.util.InstabugSDKLogger
import com.instabug.library.util.memory.MemoryUtils
import com.instabug.library.util.threading.PoolProvider
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import kotlin.math.max


class ActivityViewInspectorTask {

    companion object {
        private const val TAG = "ActivityViewInspectorTask"
        private const val MAX_EDGE_SIZE = 640
        private const val DEFAULT_SCALE_FACTOR = 1
        private const val KEY_ID = "id"
        private const val KEY_ICON = "icon"
        private const val KEY_TYPE = "type"
        private const val KEY_PROPERTIES = "properties"
        private const val KEY_FRAME = "frame"
        private const val KEY_NODES = "nodes"
    }

    private var isCanceled = false
    private var isRunning = true

    /**
     * Inspects activity view hierarchy
     * @param activity to be inspected
     */
    fun inspectActivityView(
        activity: Activity
    ) {
        if (LiveBugManager.getInstance().bug != null) {
            LiveBugManager.getInstance().bug!!.viewHierarchyInspectionState =
                Bug.ViewHierarchyInspectionState.IN_PROGRESS
        }
        ViewHierarchyInspectorEventBus.getInstance().post(ViewHierarchyInspector.Action.STARTED)
        val rootViewHierarchy = ViewHierarchy()
        rootViewHierarchy.view = activity.window.decorView
        try {
            rootViewHierarchy.frame =
                ViewHierarchyInspector.inspectRootViewFrame(
                    activity,
                    getProperScaleFactor(activity)
                )
        } catch (e: JSONException) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG, "inspect activity frame got error" + e.message, e
            )
        }
        val ignoredViewsIds = intArrayOf(
            R.id.instabug_decor_view,
            R.id.instabug_in_app_notification,
            R.id.instabug_intro_dialog
        )
        val rootViewsInfo = FieldHelper.getRootViews(activity, ignoredViewsIds)
        if (rootViewsInfo.size > 0) rootViewHierarchy.setHasChildren(true)

        val rootViewsReturnableExecutables = ArrayList<ReturnableExecutable<ViewHierarchy>>(
            rootViewsInfo
                .size
        )
        for (i in rootViewsInfo.indices) {
            val windowRootViewHierarchy = ViewHierarchy()
            windowRootViewHierarchy.id = i.toString()
            windowRootViewHierarchy.view = rootViewsInfo[i].view
            windowRootViewHierarchy.isRoot = true
            windowRootViewHierarchy.scale = getProperScaleFactor(activity)
            rootViewsReturnableExecutables.add(
                ViewHierarchyInspector.getInspectRootViewExecutable(
                    windowRootViewHierarchy
                )
            )
        }

        try {
            flattenViewHierarchy(
                activity, rootViewsReturnableExecutables,
                rootViewHierarchy
            ) {
                captureViewHierarchiesBitmaps(
                    activity, it
                ) {
                    zipViewHierarchyImages(
                        activity,
                        rootViewHierarchy
                    ) {
                        finalizeViewHierarchyInspection(
                            rootViewHierarchy
                        )
                    }
                }
            }
        } catch (e: Exception) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "activity view inspection got error: " + e.message,
                e
            )
            LiveBugManager.getInstance().bug?.viewHierarchyInspectionState =
                Bug.ViewHierarchyInspectionState.FAILED

            ViewHierarchyInspectorEventBus.getInstance().post(ViewHierarchyInspector.Action.FAILED)
            PoolProvider.postIOTask {
                DiskUtils.cleanDirectory(
                    ViewHierarchyDiskUtils.getViewHierarchyImagesDirectory(activity)
                )
            }
        }
    }

    /**
     * Updates existing bug with the captured view hierarchy and
     * updates the view hierarchy inspection state to DONE
     * @param rootViewHierarchy including all sub view hierarchies
     */
    private fun finalizeViewHierarchyInspection(rootViewHierarchy: ViewHierarchy) {
        PoolProvider.postIOTask {
            InstabugSDKLogger.v(
                Constants.LOG_TAG,
                "Activity view inspection done successfully"
            )
            if (LiveBugManager.getInstance().bug == null) return@postIOTask
            LiveBugManager.getInstance().bug!!.viewHierarchy =
                convertViewHierarchyToJson(rootViewHierarchy).toString()
            // convertViewHierarchyToJson may take long time on slow devices hence we need to double check getBug() again not null
            if (LiveBugManager.getInstance().bug == null) return@postIOTask
            LiveBugManager.getInstance().bug!!.viewHierarchyInspectionState =
                Bug.ViewHierarchyInspectionState.DONE
            ViewHierarchyInspectorEventBus.getInstance()
                .post(ViewHierarchyInspector.Action.COMPLETED)
            isRunning = false
        }
    }

    /**
     * Wraps all view hierarchy sub views images in a single zip file
     * and clears view hierarchy images directory
     */
    private fun zipViewHierarchyImages(
        activity: Activity,
        seedViewHierarchy: ViewHierarchy,
        onTaskCompletedCallback: () -> Unit
    ) {
        PoolProvider.postIOTask {
            val uri = ViewHierarchyDiskUtils.zipViewHierarchyImages(seedViewHierarchy)
            if (uri != null) {
                InstabugSDKLogger.v(
                    Constants.LOG_TAG,
                    "viewHierarchy images zipped successfully, zip file uri: "
                            + uri.toString()
                            + ", time in MS: "
                            + System.currentTimeMillis()
                )
            }
            if (LiveBugManager.getInstance().bug != null && uri != null) {
                LiveBugManager.getInstance()
                    .bug
                    ?.addAttachment(uri, Attachment.Type.VIEW_HIERARCHY)
            }
            DiskUtils.cleanDirectory(
                ViewHierarchyDiskUtils.getViewHierarchyImagesDirectory(
                    activity
                )
            )

            onTaskCompletedCallback.invoke()
        }
    }

    /**
     * Flattens view hierarchies in rootViewsReturnableExecutables
     * and add them as nodes in rootViewHierarchy
     * @param activity current activity
     * @param rootViewsReturnableExecutables ReturnableExecutable of ViewHierarchy
     * returns flat list of ViewHierarchy
     * @param callback callback to be invoked after view hierarchy flattening is done
     */
    private fun flattenViewHierarchy(
        activity: Activity,
        rootViewsReturnableExecutables: ArrayList<ReturnableExecutable<ViewHierarchy>>,
        rootViewHierarchy: ViewHierarchy,
        callback: (List<ViewHierarchy>) -> Unit
    ) {
        PoolProvider.postIOTask {
            if (isCanceled) {
                return@postIOTask
            }
            val flatViewHierarchies: MutableList<ViewHierarchy> =
                ArrayList()
            for (returnableExecutable in rootViewsReturnableExecutables) {
                var viewHierarchy: ViewHierarchy? = null
                try {
                    viewHierarchy = returnableExecutable.execute()
                } catch (e: Exception) {
                }
                rootViewHierarchy.addNode(viewHierarchy)
                if (!MemoryUtils.isLowMemory(activity)) {
                    flatViewHierarchies.addAll(
                        ViewHierarchyInspector.convertViewHierarchyToList(
                            viewHierarchy
                        )
                    )
                }
            }
            callback.invoke(flatViewHierarchies)
        }
    }

    /**
     * Captures images of app view hierarchies passed
     * @param activity current activity
     * @param flatViewHierarchies list of view hierarchy to capture images for
     * @param onTaskCompletedCallback callback to be executed when capturing for all views is done
     */
    private fun captureViewHierarchiesBitmaps(
        activity: Activity,
        flatViewHierarchies: List<ViewHierarchy>,
        onTaskCompletedCallback: () -> Unit
    ) {
        PoolProvider.postMainThreadTask {
            if (isCanceled) {
                return@postMainThreadTask
            }
            if (flatViewHierarchies.isNotEmpty()) {
                var currentViewHierarchy: ViewHierarchy = flatViewHierarchies[0]
                if (!MemoryUtils.isLowMemory(activity)) {
                    currentViewHierarchy = BitmapUtils.captureViewHierarchy(
                        currentViewHierarchy
                    )
                    persistViewHierarchyImage(currentViewHierarchy) {
                        captureViewHierarchiesBitmaps(
                            activity,
                            flatViewHierarchies.subList(
                                1,
                                flatViewHierarchies.size
                            ), onTaskCompletedCallback
                        )
                    }
                }

            } else {
                onTaskCompletedCallback.invoke()
            }


        }
    }

    /**
     * Persists captured image of view hierarchy on disk
     * @param viewHierarchy ViewHierarchy instance containing the captured image
     */
    private fun persistViewHierarchyImage(
        viewHierarchy: ViewHierarchy, onTaskCompletedCallback: () -> Unit
    ) {
        PoolProvider.postIOTask {
            if (isCanceled) {
                return@postIOTask
            }
            if (viewHierarchy.image != null) {
                InstabugSDKLogger.v(
                    TAG, "Started saving image on disk, viewHierarchyId: " + viewHierarchy.id
                )
                ViewHierarchyDiskUtils.saveViewHierarchyImage(viewHierarchy)
                viewHierarchy.removeImage()
                InstabugSDKLogger.v(
                    TAG,
                    "view hierarchy image saved successfully, uri: " + viewHierarchy.imageUriOnDisk
                )
            }
            onTaskCompletedCallback.invoke()
        }
    }

    private fun convertViewHierarchyToJson(viewHierarchy: ViewHierarchy): JSONObject {
        val viewHierarchyJsonObject = JSONObject()
        try {
            if (viewHierarchy.id != null) viewHierarchyJsonObject.put(
                KEY_ID,
                viewHierarchy.id
            )
            if (viewHierarchy.iconIdentifier != null) viewHierarchyJsonObject.put(
                KEY_ICON,
                viewHierarchy.iconIdentifier
            )
            if (viewHierarchy.type != null) viewHierarchyJsonObject.put(
                KEY_TYPE,
                viewHierarchy.type
            )
            if (viewHierarchy.properties != null) viewHierarchyJsonObject.put(
                KEY_PROPERTIES,
                viewHierarchy.properties
            )
            if (viewHierarchy.frame != null) viewHierarchyJsonObject.put(
                KEY_FRAME,
                viewHierarchy.frame
            )
            if (viewHierarchy.nodes != null && viewHierarchy.hasChildren()) {
                val childrenJsonArray = JSONArray()
                for (child in viewHierarchy.nodes) {
                    childrenJsonArray.put(convertViewHierarchyToJson(child))
                }
                viewHierarchyJsonObject.put(KEY_NODES, childrenJsonArray)
            }
        } catch (e: JSONException) {
            InstabugSDKLogger.e(
                Constants.LOG_TAG,
                "Converting view hierarchy to json got json exception: " + e.message,
                e
            )
        }
        return viewHierarchyJsonObject
    }

    private fun getProperScaleFactor(activity: Activity): Int {
        val rootView = activity.window.decorView
        val largerEdge = max(rootView.height, rootView.width)
        return if (largerEdge > MAX_EDGE_SIZE) largerEdge / MAX_EDGE_SIZE else DEFAULT_SCALE_FACTOR
    }

    /**
     * Cancels current view hierarchy inspection
     * @param context
     */
    fun cancelViewHierarchyInspection(context: Context) {
        if (isRunning) {
            InstabugSDKLogger.d(
                Constants.LOG_TAG, "CancelViewInspection called"
            )
            isCanceled = true
            PoolProvider.postIOTask {
                DiskUtils.cleanDirectory(
                    ViewHierarchyDiskUtils.getViewHierarchyImagesDirectory(
                        context
                    )
                )

            }
        }
    }

}