package com.instabug.bug.reportingpromptitems;

import static com.instabug.bug.Constants.ReportType;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.bug.BugPlugin;
import com.instabug.bug.LiveBugManager;
import com.instabug.bug.model.ReportCategory;
import com.instabug.bug.screenshot.viewhierarchy.ActivityViewInspector;
import com.instabug.bug.settings.BugSettings;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.memory.MemoryUtils;

import java.util.ArrayList;
import java.util.List;

public abstract class BaseReportingPromptItem {

    public abstract PluginPromptOption getItemOption(final Context context);


    @SuppressLint("NULL_DEREFERENCE")
    static void callOnSdkInvoked() {
        if (InstabugCore.getOnSdkInvokedCallback() != null) {
            InstabugCore.getOnSdkInvokedCallback().onSdkInvoked();
        }
    }

    static void createReportWithInitialScreenshot(@Nullable Uri initialScreenshotUri) {
        BugPlugin plugin = (BugPlugin) InstabugCore.getXPlugin(BugPlugin.class);
        if (plugin != null && plugin.getAppContext() != null) {
            LiveBugManager.getInstance().createBug(plugin.getAppContext());
        }
        if (initialScreenshotUri != null
                && BugSettings.getInstance().getAttachmentsTypesParams()
                .isShouldTakesInitialScreenshot() && LiveBugManager.getInstance().getBug() != null) {
            LiveBugManager.getInstance().getBug().
                    setInitialScreenshotPath(initialScreenshotUri.getPath());
        }

    }

    @VisibleForTesting
    abstract String getReportTitle(Context context);

    @VisibleForTesting
    abstract String getReportDescription(Context context);

    protected abstract void invoke(Context context, Uri initialScreenshotUri, String... reportCategories);

    static void startViewHierarchyInspection() {
        Activity activity = InstabugInternalTrackingDelegate.getInstance().getTargetActivity();
        if (activity == null) {
            return;
        }
        if (LiveBugManager.getInstance().getBug() != null
                && LiveBugManager.getInstance().getBug().isRequiredViewHierarchy()
                && !MemoryUtils.isLowMemory(activity)) {
            ActivityViewInspector.inspectActivityView(activity);
        }
    }


    abstract void handleNewReport(Context context);

    protected ArrayList<PluginPromptOption> getSubOptions(@ReportType String reportType) {
        List<ReportCategory> categories = ReportCategory.getSubReportCategories(reportType);
        return getSubOptions(categories, null, reportType);
    }

    @SuppressLint("ERADICATE_NULLABLE_DEREFERENCE")
    protected ArrayList<PluginPromptOption> getSubOptions(@Nullable List<ReportCategory> categories,
                                                          @Nullable PluginPromptOption parent,
                                                          @ReportType String reportType) {
        ArrayList<PluginPromptOption> options = new ArrayList<>();
        for (int i = 0; i < (categories != null ? categories.size() : 0); i++) {
            options.add(getSubOption(categories.get(i), parent, reportType, i));
        }
        return options;
    }

    protected PluginPromptOption getSubOption(ReportCategory reportCategory,
                                              @Nullable PluginPromptOption parent,
                                              @ReportType String reportType,
                                              int order) {
        PluginPromptOption promptOption = new PluginPromptOption();
        promptOption.setOrder(order);
        promptOption.setTitle(reportCategory.getLabel());
        promptOption.setDescription(reportCategory.getDescription());
        promptOption.setInitialScreenshotRequired(true);
        promptOption.setParent(parent);
        promptOption.setSubOptions(getSubOptions(reportCategory.getSubs(), promptOption, reportType));
        return promptOption;
    }
}
