package com.instabug.bug.invocation.invoker;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Build;
import android.view.GestureDetector;
import android.view.MotionEvent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.view.GestureDetectorCompat;

import com.instabug.library.Constants;
import com.instabug.bug.invocation.InvocationListener;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.library.tracking.InstabugInternalTrackingDelegate;
import com.instabug.library.util.InstabugSDKLogger;
import com.instabug.library.util.threading.PoolProvider;

/**
 * @author mesbah.
 */
public class TwoFingerSwipeLeftInvoker implements AbstractInvoker<MotionEvent> {

    /**
     * The needed number of pointers of data contained in this swipe event.
     */
    private static final int NEEDED_CONTAINED_POINTER = 2;
    @Nullable
    private GestureDetectorCompat gestureDetectorCompat;
    @Nullable
    private GestureListener gestureListener;
    private Context context;
    private boolean goodCycle = false;
    private InvocationListener invocationListener;
    private volatile boolean active;

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    public TwoFingerSwipeLeftInvoker(Context context, InvocationListener invocationListener) {
        this.context = context;
        this.invocationListener = invocationListener;
    }

    @Override
    public synchronized void listen() {
        PoolProvider.postMainThreadTask(new Runnable() {
            @Override
            public void run() {
                Context currentContext;
                if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.Q) {
                    currentContext = context;
                } else {
                    currentContext = InstabugInternalTrackingDelegate.getInstance().getCurrentActivity();
                }
                if (currentContext != null) {
                  try {
                      gestureListener = new GestureListener();
                      gestureDetectorCompat = new GestureDetectorCompat(currentContext, gestureListener);
                      active = true;
                  } catch (Exception exception) {
                      InstabugSDKLogger.e(Constants.LOG_TAG, exception.getMessage() != null ? exception.getMessage():
                            "Couldn't initialize GestureDetector", exception);
                  }
                }
            }
        });
    }

    @Override
    public synchronized void handle(MotionEvent event) {
        if (gestureDetectorCompat == null) { // invoker is sleep
            return;
        }
        //Filter MOVE actions only, because UP and DOWN usually happen with one finger
        // (you can't accurately start the gesture with all three fingers at the exact moment)
        if ((event.getAction() & MotionEvent.ACTION_MASK) == MotionEvent.ACTION_MOVE) {
            if (event.getPointerCount() < NEEDED_CONTAINED_POINTER) {
                return;
            } else {
                goodCycle = true;
            }
        }
        gestureDetectorCompat.onTouchEvent(event);
    }

    @Override
    public synchronized void sleep() {
        gestureListener = null;
        gestureDetectorCompat = null;
        active = false;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    class GestureListener extends GestureDetector.SimpleOnGestureListener {
        @Override
        public boolean onFling(@Nullable MotionEvent e1, @NonNull MotionEvent e2, float velocityX, float velocityY) {
            if (isDirectionValid(e1, e2) && goodCycle) {
                PoolProvider.postIOTask(() -> {
                    InstabugSDKLogger.d(Constants.LOG_TAG, "Two fingers swiped left, invoking SDK");
                    InvocationManager.getInstance().setLastUsedInvoker(TwoFingerSwipeLeftInvoker.this);
                    invocationListener.onInvocationRequested();
                });

            }
            goodCycle = false;
            return false;
        }
    }

    private boolean isDirectionValid(@Nullable MotionEvent e1, MotionEvent e2) {
        return !(e1 == null || e2 == null) && e1.getX() > e2.getX() && ((e1.getX() - e2.getX()) >= Math.abs(e1.getY() - e2.getY()));
    }

}
