package com.instabug.bug.invocation.invocationdialog;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.Uri;
import android.os.Handler;

import androidx.annotation.Nullable;

import com.instabug.library.Constants;
import com.instabug.library.Instabug;
import com.instabug.library.core.plugin.PluginPromptOption;
import com.instabug.library.core.plugin.PromptOptionManager;
import com.instabug.library.core.ui.BasePresenter;
import com.instabug.library.internal.storage.DiskUtils;
import com.instabug.library.internal.storage.operation.DeleteUriDiskOperation;
import com.instabug.bug.invocation.InvocationManager;
import com.instabug.bug.invocation.invoker.AbstractInvoker;
import com.instabug.bug.invocation.invoker.ShakeInvoker;
import com.instabug.library.settings.SettingsManager;
import com.instabug.library.util.InstabugSDKLogger;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class InstabugDialogActivityPresenter extends BasePresenter<InstabugDialogActivityContract.View>
        implements InstabugDialogActivityContract.Presenter {

    private final static long AUTO_DISMISS_TIME = 10_000;
    private final InstabugDialogActivityContract.View viewInstance;
    @Nullable
    private InstabugDialogItem selectedItem;
    @Nullable
    private Handler autoDismissHandler;
    private int enterAnimation;
    private int exitAnimation;

    InstabugDialogActivityPresenter(InstabugDialogActivityContract.View view) {
        super(view);
        viewInstance = view;
        enterAnimation = viewInstance.getFadeInAnimation();
        exitAnimation = viewInstance.getSlidOutLeftAnimation();
        // This fixes https://instabug.atlassian.net/browse/MOB-2427 till we find another solution
        SettingsManager.getInstance().setProcessingForeground(false);
    }

    @Override
    @SuppressLint("NULL_DEREFERENCE")
    public void onViewStarted() {
        AbstractInvoker lastInvoker = InvocationManager.getInstance().getLastUsedInvoker();
        if (lastInvoker instanceof ShakeInvoker)
            startAutoDismiss();
    }

    @Override
    public void onViewStopped() {
        cancelAutoDismiss();
        SettingsManager.getInstance().setShouldAutoShowOnboarding(false);
    }

    private void startAutoDismiss() {
        autoDismissHandler = new Handler();
        if (viewInstance != null) {
            autoDismissHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    viewInstance.finishActivity();
                }
            }, AUTO_DISMISS_TIME);
        }
    }

    @SuppressLint("ERADICATE_PARAMETER_NOT_NULLABLE")
    private void cancelAutoDismiss() {
        if (autoDismissHandler != null) {
            autoDismissHandler.removeCallbacksAndMessages(null);
        }
    }

    @Override
    public void onItemClicked(@Nullable InstabugDialogItem item, @Nullable Uri screenshotUri) {
        selectedItem = item;
        cancelAutoDismiss();
        if (item != null) {
            ArrayList<InstabugDialogItem> subItems = item.getSubItems();
            if (subItems != null && !subItems.isEmpty()) {
                enterAnimation = viewInstance.getSlidInRightAnimation();
                exitAnimation = viewInstance.getSlidOutLeftAnimation();
                String title = getRootParent(item).getTitle();
                title = title == null ? "" : title;
                viewInstance.setContent(title, false, subItems);
            } else {
                invoke(item, screenshotUri);
            }
        }
    }

    @Override
    public int getContentEnterAnimation() {
        return enterAnimation;
    }

    @Override
    public int getContentExitAnimation() {
        return exitAnimation;
    }

    @Override
    public void onBackPressed() {
        if (selectedItem != null) {
            selectedItem = selectedItem.getParent();
        }
        enterAnimation = viewInstance.getSlidInLeftAnimation();
        exitAnimation = viewInstance.getSlidOutRightAnimation();
    }

    @Override
    public void resetPromptOptionsSelection() {
        selectedItem = null;
    }

    @Override
    public boolean isPromptOptionSelected() {
        return selectedItem != null;
    }

    @SuppressLint("NULL_DEREFERENCE")
    private void invoke(InstabugDialogItem item, @Nullable Uri screenshotUri) {
        ArrayList<PluginPromptOption> availablePromptOptions =
                InvocationManager.getInstance().getAvailablePromptOptions();
        InstabugDialogItem rootParent = getRootParent(item);
        if (rootParent.getOrder() == PluginPromptOption.NO_ORDER) {
            // chats list button
            for (PluginPromptOption promptOption : availablePromptOptions) {
                if (promptOption.getOrder() == PluginPromptOption.NO_ORDER) {
                    promptOption.invoke();
                    break;
                }
            }
        } else {
            PluginPromptOption option = PromptOptionManager.getPluginByIdentifier(item.getIdentifier(), true);
            if (option != null) {
                option.invoke(screenshotUri, getCategories(item));
            }
        }
    }

    public void dumpAttachments(Uri... uris) {
        Context context = Instabug.getApplicationContext();
        if (context != null) {
            DiskUtils diskUtils = DiskUtils.with(context);
            if (uris != null)
                for (Uri uri : uris) {
                    diskUtils.deleteOperation(new DeleteUriDiskOperation(uri))
                            .executeAsync(null);
                }
        } else InstabugSDKLogger.e(Constants.LOG_TAG, "Can't execute dumpAttachments() due to null context");
    }

    private String[] getCategories(InstabugDialogItem item) {
        List<String> categories = new ArrayList<>();
        while (item.getParent() != null) {
            categories.add(item.getTitle());
            item = item.getParent();
        }
        Collections.reverse(categories);
        return categories.toArray(new String[0]);
    }

    private InstabugDialogItem getRootParent(InstabugDialogItem item) {
        while (item.getParent() != null) {
            item = item.getParent();
        }
        return item;
    }

    void removeScreenShotIfNeeded(InstabugDialogItem dialogAction) {
        if (view != null) {
            InstabugDialogActivityContract.View viewRef = this.view.get();
            if (viewRef != null) {
                if (dialogAction != null) {
                    if (!dialogAction.isInitialScreenshotRequired()) {
                        viewRef.onInitialScreenShotNotRequired();
                    }
                }
            }
        }
    }
}
