package com.instabug.bug.configurations

import com.instabug.bug.configurations.BugReportingConfigurationsProviderImpl.USER_CONSENT_DEFAULT_VALUE
import com.instabug.bug.configurations.BugReportingConfigurationsProviderImpl.USER_CONSENT_FEATURE_KEY
import com.instabug.bug.configurations.BugReportingConfigurationsProviderImpl.USER_CONSENT_LIMIT_DEFAULT_VALUE
import com.instabug.bug.configurations.BugReportingConfigurationsProviderImpl.USER_CONSENT_LIMIT_KEY
import com.instabug.bug.di.ServiceLocator
import com.instabug.library.IssueType
import com.instabug.library.ReproMode
import com.instabug.library.core.InstabugCore
import com.instabug.library.visualusersteps.ReproRuntimeConfigurationsHandler
import org.json.JSONObject

interface BugReportingConfigurationsHandler : ReproRuntimeConfigurationsHandler {
    fun handleConfigurations(configurations: String?): Boolean
}

private const val CONFIG_KEY_BUGS = "bugs"
private const val CONFIG_KEY_REPRO_SCREENSHOTS = "rsa"
private const val CONFIG_KEY_PRODUCT_USAGE = "product_usage_exceeded"
private const val CONFIG_KEY_BUG_REPORTING = "bug_reporting"
private const val CONFIG_KEY_PROACTIVE_REPORTING = "proactive_reporting"

object BugReportingConfigurationsHandlerImpl : BugReportingConfigurationsHandler {
    private val configurationsProvider: BugReportingConfigurationsProvider
        get() = ServiceLocator.getConfigurationsProvider()

    override fun handleConfigurations(configurations: String?): Boolean {
        return try {
            configurations?.let(::JSONObject)
                ?.also(this::extractAndUpdateUsageExceeded)
                ?.also(this::extractAndUpdateRSA)
                ?.also(::extractUserConsentFeatureFlag)
                ?.also (this::extractProactiveReportingFlags)
                ?.let { true }
                ?: false
        } catch (exception: Exception) {
            InstabugCore.reportError(exception, "couldn't parse bug reporting feature flags ")
            false
        }
    }

    private fun extractUserConsentFeatureFlag(configurationObject: JSONObject) {
        configurationsProvider.isUserConsentAvailable =
            configurationObject.optBoolean(USER_CONSENT_FEATURE_KEY, USER_CONSENT_DEFAULT_VALUE)
        configurationsProvider.consentsLimit =
            configurationObject.optInt(USER_CONSENT_LIMIT_KEY, USER_CONSENT_LIMIT_DEFAULT_VALUE)
    }

    private fun extractProactiveReportingFlags(configurationObject: JSONObject) {
        ServiceLocator.getProactiveReportingConfigHandler()
            .handle(configurationObject.optJSONObject(CONFIG_KEY_BUGS)?.optJSONObject(
                CONFIG_KEY_PROACTIVE_REPORTING))
    }

    override fun handle(modesMap: Map<Int, Int>) {
        modesMap[IssueType.Bug]?.let { bugsMode ->
            with(configurationsProvider) {
                isReproStepsEnabledSDK = bugsMode > ReproMode.Disable
                isReproScreenShotsEnabledSDK = bugsMode > ReproMode.EnableWithNoScreenshots

            }
        }
    }

    private fun extractAndUpdateUsageExceeded(configObj: JSONObject) {
        configurationsProvider.isBugReportingUsageExceeded =
            configObj.optJSONObject(CONFIG_KEY_PRODUCT_USAGE)
                ?.optBoolean(CONFIG_KEY_BUG_REPORTING) ?: false
    }

    private fun extractAndUpdateRSA(configObj: JSONObject) {
        configObj.optJSONObject(CONFIG_KEY_BUGS)
            ?.takeIf { bugsObj -> bugsObj.has(CONFIG_KEY_REPRO_SCREENSHOTS) }
            ?.optBoolean(CONFIG_KEY_REPRO_SCREENSHOTS, true)
            ?.let { rsa -> configurationsProvider.isReproScreenshotsAvailable = rsa }
    }
}