package com.instabug.bug.cache;

import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_BUG_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_CATEGORIES_LIST;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_ID;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_MESSAGE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_STATE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_TEMPORARY_SERVER_TOKEN;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_TYPE;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.COLUMN_VIEW_HIERARCHY;
import static com.instabug.library.internal.storage.cache.db.InstabugDbContract.BugEntry.TABLE_NAME;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;

import androidx.annotation.Nullable;

import com.instabug.bug.Constants;
import com.instabug.bug.model.Bug;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.AttachmentsDbHelper;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.model.Attachment;
import com.instabug.library.model.State;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;
import java.util.List;

public class BugReportsPlainDbHelperImpl implements BugReportsDbHelper<ContentValues> {

    @Override
    public long insert(Bug bug) throws JSONException {
        if (bug.getId() == null) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Couldn't save the bug to DB because its ID is null");
            return NOT_FOUND;
        }
        // Gets the data repository in write mode
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            // Create a new map of values, where column names are the keys
            ContentValues values = new ContentValues();
            values.put(COLUMN_ID, bug.getId());
            values.put(COLUMN_MESSAGE, bug.getMessage());
            values.put(COLUMN_BUG_STATE, bug.getBugState().name());
            if (bug.getTemporaryServerToken() != null) {
                values.put(COLUMN_TEMPORARY_SERVER_TOKEN, bug.getTemporaryServerToken());
            }
            values.put(COLUMN_TYPE, bug.getType());
            values.put(COLUMN_CATEGORIES_LIST, bug.getCategoriesAsJSONArray().toString());
            if (bug.getViewHierarchy() != null) {
                values.put(COLUMN_VIEW_HIERARCHY, bug.getViewHierarchy());
            }
            if (bug.getState() != null && bug.getState().getUri() != null) {
                values.put(COLUMN_STATE, bug.getState().getUri().toString());
            }

            for (Attachment attachment : bug.getAttachments()) {
                long rowId = AttachmentsDbHelper.insert(attachment, bug.getId());
                if (rowId != NOT_FOUND) {
                    attachment.setId(rowId);
                }
            }

            // Insert the new row, returning the primary key value of the new row
            long rowId = db.insert(TABLE_NAME, null, values);
            db.setTransactionSuccessful();
            return rowId;
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @Override
    public List<Bug> retrieve(Context context) {
        if (context != null) {
            SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
            List<Bug> bugs = new ArrayList<>();
            Cursor c = null;
            try {
                c = db.query(
                        TABLE_NAME,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null,
                        null
                );

                if (c != null && c.moveToFirst()) {
                    do {
                        Bug bug = new Bug();
                        int idColIndex = c.getColumnIndex(COLUMN_ID);
                        bug.setId(c.getString(idColIndex));

                        int messageColIndex = c.getColumnIndex(COLUMN_MESSAGE);
                        bug.setMessage(c.getString(messageColIndex));

                        int bugStateColIndex = c.getColumnIndex(COLUMN_BUG_STATE);
                        bug.setBugState(Enum.valueOf(Bug.BugState.class, c.getString(bugStateColIndex)));

                        int tempServerTokenColIndex = c.getColumnIndex(COLUMN_TEMPORARY_SERVER_TOKEN);
                        bug.setTemporaryServerToken(c.getString(tempServerTokenColIndex));

                        int bugTypeColIndex = c.getColumnIndex(COLUMN_TYPE);
                        bug.setType(c.getString(bugTypeColIndex));

                        int bugCategoriesListColIndex = c.getColumnIndex(COLUMN_CATEGORIES_LIST);
                        bug.setCategoriesFromJSONArray(new JSONArray(c.getString(bugCategoriesListColIndex)));

                        int bugViewHierarchy = c.getColumnIndex(COLUMN_VIEW_HIERARCHY);
                        bug.setViewHierarchy(c.getString(bugViewHierarchy));

                        int stateColIndex = c.getColumnIndex(COLUMN_STATE);
                        String uriString = c.getString(stateColIndex);
                        Uri uri = null;
                        if (uriString != null)
                            uri = Uri.parse(uriString);
                        try {
                            bug.setState(State.getState(context, uri));
                        } catch (OutOfMemoryError | Exception e) {
                            InstabugCore.reportError(e, "retrieving bug state throws OOM");
                            InstabugSDKLogger.e(Constants.LOG_TAG, "Retrieving bug state throws an exception", e);
                        }
                        if (bug.getId() != null) {
                            ArrayList<Attachment> bugAttachments = AttachmentsDbHelper.retrieve(bug.getId(), db);
                            bug.setAttachments(bugAttachments);
                            bugs.add(bug);
                        }
                    } while (c.moveToNext());
                }
            } catch (Exception e) {
                InstabugSDKLogger.e(Constants.LOG_TAG, "retrieve bug reports failed: ", e);
                IBGDiagnostics.reportNonFatal(e, "retrieve bug reports failed: " + e.getMessage());
            } finally {
                if (c != null) {
                    c.close();
                }
                db.close();
            }
            return bugs;
        } else {
            InstabugSDKLogger.e(Constants.LOG_TAG, "Context is null while retrieving bugs from DB.");
            return new ArrayList<>();
        }
    }

    @Override
    public void update(String id, ContentValues cv) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{id};
        db.beginTransaction();
        try {
            db.update(TABLE_NAME, cv, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    /**
     * This method is empty because plain bugs table doesn't contain connection_error column as we don't use it anymore
     */
    @Override
    public void updateConnectionError(@Nullable String id, String network) {
    }

    @Override
    public void delete(String id) {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        String whereClause = COLUMN_ID + "=? ";
        String[] whereArgs = new String[]{id};
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, whereClause, whereArgs);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @Override
    public void deleteAll() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            db.delete(TABLE_NAME, null, null);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @Override
    public void dropTable() {
        SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        db.beginTransaction();
        try {
            db.execSQL(InstabugDbContract.BugEntry.DROP_TABLE);
            db.setTransactionSuccessful();
        } finally {
            db.endTransaction();
            db.close();
        }
    }

    @Override
    public boolean updateFrustratingExperienceExternalId(Long internalId, String externalId) {
        return false;
    }

    @Override
    public List<Bug> retrieveNormalBugs(Context context) {
        return new ArrayList<>();
    }

    @Override
    public List<Bug> retrieveProactiveReportingBugs(Context context) {
        return new ArrayList<>();
    }
}