package com.instabug.bug.cache;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;

import androidx.annotation.VisibleForTesting;

import com.instabug.bug.Constants;
import com.instabug.bug.cache.migrationpolicy.InterTableMigrationInterruptionsPolicy;
import com.instabug.bug.cache.migrationpolicy.MigrationInterruptionPolicy;
import com.instabug.bug.di.ServiceLocator;
import com.instabug.bug.model.Bug;
import com.instabug.library.diagnostics.IBGDiagnostics;
import com.instabug.library.internal.storage.cache.db.DatabaseManager;
import com.instabug.library.internal.storage.cache.db.InstabugDbContract;
import com.instabug.library.internal.storage.cache.db.SQLiteDatabaseWrapper;
import com.instabug.library.internal.storage.cache.dbv2.IBGContentValues;
import com.instabug.library.util.InstabugSDKLogger;

import org.json.JSONException;

import java.util.List;

public class BugReportsDbMigrationHelper {

    private BugReportsDbHelper<IBGContentValues> bugReportsDbHelper;
    private BugReportsDbHelper<ContentValues> bugReportsPlainDbHelper;

    public BugReportsDbMigrationHelper() {
        bugReportsDbHelper = ServiceLocator.getBugReportsDbHelper();
        bugReportsPlainDbHelper = ServiceLocator.getBugReportsPlainDbHelper();
    }

    /**
     * Migrate bugs cross tables, from the old plain table to the new encrypted table
     */
    public void migrateToEncrypted(Context context) {
        try {
            if (!isPlainBugsTableExists()) return;

            List<Bug> bugList = bugReportsPlainDbHelper.retrieve(context);
            if (!bugList.isEmpty()) {
                InstabugSDKLogger.d(Constants.LOG_TAG, "Migrating " + bugList.size() + " bugs to encrypted table");
                for (Bug bug : bugList) {
                    String id = bug.getId();
                    if (id != null) {
                        try {
                            bugReportsDbHelper.insert(bug);
                        } catch (JSONException e) {
                            IBGDiagnostics.reportNonFatal(e, String.format("Failed to migrate bug with id %s to encrypted DB, dropping it.", id));
                        }
                        bugReportsPlainDbHelper.delete(id);
                    }
                }
            }
        } catch (Exception e) {
            handleMigrationInterruptions(new InterTableMigrationInterruptionsPolicy(e));
        } finally {
            bugReportsPlainDbHelper.dropTable();
        }
    }

    /**
     * checks if the plain bugs table exists
     * select bug table table name from sqlite_master
     * @return  bug_table is Exists on the database or not
     */
    private boolean isPlainBugsTableExists() {
        final SQLiteDatabaseWrapper db = DatabaseManager.getInstance().openDatabase();
        final String selectQuery = "SELECT name FROM sqlite_master WHERE type='table' AND name = '" +
                InstabugDbContract.BugEntry.TABLE_NAME + "'";

        try (Cursor cursor = db.rawQuery(selectQuery, null)) {

            return cursor != null && cursor.moveToFirst();

        } catch (Exception exception) {
            InstabugSDKLogger.e(Constants.LOG_TAG, "can't check if plain Bugs Table exists", exception);
            IBGDiagnostics.reportNonFatal(exception, "can't check if plain Bugs Table exists due to: " + exception.getMessage());
            return false;
        }
    }

    @VisibleForTesting
    void handleMigrationInterruptions(MigrationInterruptionPolicy policy) {
        policy.migrate();
    }


    // Migrate unencrypted bugs to be ≈encrypted
    @VisibleForTesting
    void runMigration(Context context) throws JSONException {
        List<Bug> bugsList = bugReportsDbHelper.retrieve(context);
        if (bugsList.isEmpty()) {
            return;
        }
        for (Bug bug : bugsList) {
            bugReportsDbHelper.insert(bug);
        }
    }

}
