package com.instabug.apm.uitrace.handler

import android.annotation.SuppressLint
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.apm.di.Provider
import com.instabug.apm.handler.session.SessionHandler
import com.instabug.apm.uitrace.uihangs.UiHangHandler
import com.instabug.apm.uitrace.UiTraceWrapper
import com.instabug.apm.uitrace.model.UiTraceEndParams
import com.instabug.apm.uitrace.model.UiTraceInitParams
import com.instabug.apm.util.powermanagement.BatteryLevelChangeBroadcast
import com.instabug.apm.util.powermanagement.PowerManagementCallback
import com.instabug.apm.util.powermanagement.PowerSaveModeBroadcast
import com.instabug.apm.webview.webview_trace.manager.WebViewTraceManager
import com.instabug.library.BuildFieldsProvider

interface UiTraceWrapperHandler {
    fun start(wrapper: UiTraceWrapper, initModel: UiTraceInitParams)

    fun end(wrapper: UiTraceWrapper, params: UiTraceEndParams)

    fun clean(wrapper: UiTraceWrapper)
}

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
@SuppressLint("NewApi")
class UiTraceWrapperHandlerImpl(
    val batteryLevelChangeBroadcastProvider: Provider<BatteryLevelChangeBroadcast?>,
    val powerSaveModeBroadcastProvider: Provider<PowerSaveModeBroadcast?>,
    val webViewTraceManagerProvider: Provider<WebViewTraceManager?>,
    val sessionHandler: SessionHandler
) : UiTraceWrapperHandler {

    private val canCapturePowerSaving: Boolean
        get() = BuildFieldsProvider.provideBuildVersion() >= Build.VERSION_CODES.LOLLIPOP

    override fun start(wrapper: UiTraceWrapper, initModel: UiTraceInitParams) {
        wrapper.apply {
            registerBroadcastReceivers(powerManagementCallback)
            uiHangsHandler.start()
            cacheModel.also {
                it.name = initModel.name
                it.screenTitle = initModel.screenTitle
                it.sessionId = sessionHandler.currentSession?.id
                it.batteryLevel = initModel.batteryLevel
                it.powerSaveMode = initModel.isPowerSaveModeEnabled
                it.orientation = initModel.screenOrientation
                it.startTimestamp = initModel.timeStampMicros
                it.startTimeInMicros = initModel.startTimeMicro
                it.isUserDefined = initModel.isUserDefined
            }
        }
    }

    private fun registerBroadcastReceivers(callback: PowerManagementCallback) {
        batteryLevelChangeBroadcastProvider()?.register(callback)
        if (canCapturePowerSaving) powerSaveModeBroadcastProvider()?.register(callback)
    }

    override fun end(wrapper: UiTraceWrapper, params: UiTraceEndParams) {
        wrapper.updateCacheModel(params)
        wrapper.webViewTraceListener?.onUiTraceEnded()
        clean(wrapper)
    }

    private fun UiTraceWrapper.updateCacheModel(params: UiTraceEndParams) {
        cacheModel.also {
            it.refreshRate = params.refreshRate
            it.duration = params.elapsedTimeMicro - it.startTimeInMicros
            if (it.name.orEmpty() != params.activityClassName) {
                it.containerName = params.activityClassName
            }
            it.moduleName = params.moduleName
            it.setUiHangsModel(uiHangsHandler.uiHangModel)
        }
    }

    override fun clean(wrapper: UiTraceWrapper) {
        wrapper.apply {
            unregisterBroadcastReceivers(powerManagementCallback)
            uiHangsHandler.stopUiHangs()
            webViewTraceListener?.let {
                webViewTraceManagerProvider()?.unregisterWebViewEventListener(it)
            }
        }
    }

    private fun unregisterBroadcastReceivers(callback: PowerManagementCallback) {
        batteryLevelChangeBroadcastProvider()?.unregister(callback)
        if (canCapturePowerSaving) powerSaveModeBroadcastProvider()?.unregister(callback)
    }

    private fun UiHangHandler.stopUiHangs() {
        stop()
        clearUiHangModel()
    }
}