package com.instabug.apm.appflow.handler

import com.instabug.apm.appflow.model.AppFlowCacheModel
import com.instabug.apm.appflow.model.AppFlowEndReason
import com.instabug.apm.appflow.model.AppFlowInsertionCacheModel

interface AppFlowCacheHandler {

    /**
     * Insert new app flow record to database
     * it will not end active app flows with the same name, it has to be done manually through the
     * appropriate CacheHandler APIs
     * it's the responsibility of the caller to end the active app flows with the same name before
     * calling this API
     *
     */
    fun insert(insertionModel: AppFlowInsertionCacheModel): Long

    /**
     * End all active app flows with the provided name in the within the provided app launch
     * Should always be one app flow affected
     *
     * @param timeMicro: time in micro marking the end of the app flow which is going to be used in deriving the flow duration
     * @param appLaunchId: ID of the app launch that the flow can be active through
     */
    fun end(name: String, timeMicro: Long, appLaunchId: String): Int

    /**
     * End all active app flows with the provided name in the within the provided app launch
     * Should always be one app flow affected
     *
     * @param appLaunchId: ID of the app launch that the flow can be active through
     */
    fun end(name: String, appLaunchId: String): Int

    /**
     * End all abandoned active app flows the provided app launch
     *
     * @param appLaunchId: ID of the app launch that the flows can be active through
     * @param endReason: active app flows with this end reason will be marked as ended
     */
    fun endActiveFlowsWithEndReason(appLaunchId: String, @AppFlowEndReason endReason: Int): Int

    /**
     * End all active app flows with the provided name within the provided app launch
     * with the provided reason
     * Should always be one app flow affected
     *
     * @param appLaunchId: ID of the app launch that the flow can be active through
     */
    fun endWithReason(name: String, @AppFlowEndReason reason: Int, appLaunchId: String): Int

    /**
     * Add attribute to app flow, will only add it to the most recent active app flow with the
     * matching name and app launch id
     */
    fun addAttribute(name: String, key: String, value: String, appLaunchId: String): Long

    /**
     * Remove attribute from app flow, will only remove from the most recent active app flow with
     * the matching name and app launch id
     */
    fun removeAttribute(name: String, key: String, appLaunchId: String): Int

    /**
     * Update AppFlow attribute value for a given key, it will only affect the most recent active
     * app flow with the matching name and app launch id
     */
    fun updateAttributeValue(name: String, key: String, newValue: String, appLaunchId: String): Int

    /**
     * Get the total attribute count of the most recent active app flow with the matching
     * name and app launch id
     */
    fun getAttributeCount(name: String, appLaunchId: String): Int

    /**
     * Remove all app flow records and attributes from DB
     */
    fun clear()

    /**
     * Migrate all active app flows within the app launch to apm session with the passed ID
     */
    fun migrateActiveFlows(newSession: String, appLaunchId: String): Int

    /**
     * Retrieve all app flows in apm session with the passed ID
     */
    fun retrieve(sessionId: String): List<AppFlowCacheModel>

    /**
     * Get app flows count in apm session with the passed ID
     */
    fun getCount(sessionId: String): Int

    /**
     * Trim AppFlows for a session by limit per request
     *
     * @return the number of dropped app flow records
     */
    fun trimByRequestLimitForSession(sessionId: String, limit: Int): Int

    /**
     * Trim app flows by store limit, excluding the given session.
     * The passed session id will not be used in calculating the total count and no records will
     * be dropped from it
     */
    fun trimByStoreLimitExcludingSession(sessionId: String, limit: Int): Int

    /**
     * Drop flows that are not linked to a APM session except for active flows in the passed app launch.
     * Flows the are gonna be dropped are the following:
     * * Dangling app flows in other app launches
     * * Dangling ended app flows in the passed app launch
     */
    fun dropDanglingFLowsExcludingAppLaunch(launchId: String): Int

    /**
     * Sets end reason for all active app flows for a given app launch
     *
     * @param isBackgroundFlagOverride: override for isBackground flag, null if flag value shouldn't be overridden
     * @param appLaunchId : ID of the app launch that the flow can be active through
     * */
    fun setActiveFlowsEndReason(
        @AppFlowEndReason reason: Int,
        isBackgroundFlagOverride: Boolean? = null,
        appLaunchId: String
    ): Int

    /**
     * Update isBackground flag for active app flows in the provided app launch
     *
     * @return number of affected rows
     */
    fun setActiveFlowsBackgroundState(isBackground: Boolean, appLaunchId: String): Int

    /**
     * Set core session id for active app flows which have null core session id within the passed app launch
     * @return number of affected rows
     */
    fun setActiveFlowsCoreSessionId(coreSessionId: String, appLaunchId: String): Int

    /**
     * Filters unready core session ids due to unfinished APM session flow migration
     *
     * @return String list containing core session ids for APM sessions that still have active flows that are not migrated
     * the list will be empty in case no unready session ids or error
     *
     * @param sessionIds list of session ids to search in
     * @param launchId of app flows linked to searched sessions
     */
    fun filterUnReadyCoreSessionIds(sessionIds: List<String>, launchId: String): List<String>
}