package com.instabug.apm.webview.webview_trace.handler

import android.graphics.Bitmap
import android.os.Build
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import androidx.annotation.RequiresApi
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.webview.vital.InstabugWebViewVitalJSInterface
import com.instabug.apm.webview.vital.InstabugWebVitalsEventListener
import com.instabug.apm.webview.webview_trace.model.event.WebViewEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewEventId
import com.instabug.apm.webview.webview_trace.model.event.WebViewLoadDataEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewLoadDataWithUrlEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewLoadUrlEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewOverrideUrlEndEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewPostUrlEvent
import com.instabug.apm.webview.webview_trace.model.event.WebViewSizeReadyEvent
import com.instabug.apm.webview.webview_trace.util.WebViewSizeEventResolver
import com.instabug.apm.webview.webview_trace.util.WebViewTraceUrlSanitizer
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.factory.ParameterizedFactory
import java.util.concurrent.Executor

@RequiresApi(api = Build.VERSION_CODES.O)
class WebViewTraceEventListenerImpl(
    private val uiTraceId: Long,
    private val repository: WebViewTraceRepository,
    private val sizeEventResolver: WebViewSizeEventResolver,
    private val javaScriptInterface: InstabugWebViewVitalJSInterface,
    private val vitalsListenerFactory: ParameterizedFactory<InstabugWebVitalsEventListener, Long>,
    private val webViewExecutor: Executor,
    private val mainThreadExecutor: Executor,
    private val urlSanitizer: WebViewTraceUrlSanitizer
) : WebViewTraceEventListener {

    private var javaScriptInterfaceAttached: Boolean = false

    override fun onLoadUrl(
        id: Long,
        webView: WebView,
        url: String?,
        timeCapture: EventTimeMetricCapture
    ) {
        handleOnLoadUrl(id, webView, url, timeCapture)
    }

    override fun onLoadUrl(
        id: Long,
        webView: WebView,
        url: String?,
        headers: Map<String, String>?,
        timeCapture: EventTimeMetricCapture
    ) {
        handleOnLoadUrl(id, webView, url, timeCapture)
    }

    private fun handleOnLoadUrl(
        id: Long,
        webView: WebView,
        url: String?,
        timeCapture: EventTimeMetricCapture
    ) {
        dispatchToRepository(
            id,
            WebViewLoadUrlEvent(
                uiTraceId,
                urlSanitizer(url),
                timeCapture
            )
        )
        webView.handleSizeEvent(id)
        attachJavaScriptInterface(webView)
    }

    override fun onPostUrl(
        id: Long,
        webView: WebView,
        url: String?,
        postData: ByteArray?,
        timeCapture: EventTimeMetricCapture
    ) {
        dispatchToRepository(
            id,
            WebViewPostUrlEvent(
                uiTraceId,
                timeCapture
            )
        )
    }

    override fun onLoadData(id: Long, webView: WebView, timeCapture: EventTimeMetricCapture) {
        dispatchToRepository(
            id,
            WebViewLoadDataEvent(
                uiTraceId,
                timeCapture
            )
        )
    }

    override fun onLoadDataWithBaseUrl(
        id: Long,
        webView: WebView,
        timeCapture: EventTimeMetricCapture
    ) {
        dispatchToRepository(
            id,
            WebViewLoadDataWithUrlEvent(
                uiTraceId,
                timeCapture
            )
        )
    }

    override fun shouldOverrideUrlLoadingStarted(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        timeCapture: EventTimeMetricCapture
    ) {
        view?.handleSizeEvent(id)
        dispatchToRepository(
            id,
            WebViewEvent(WebViewEventId.OVERRIDE_URL_START, timeCapture)
        )
    }

    override fun shouldOverrideUrlLoadingEnded(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        override: Boolean,
        timeCapture: EventTimeMetricCapture
    ) {
        dispatchToRepository(
            id,
            WebViewOverrideUrlEndEvent(
                isUserTriggered = request?.hasGesture() ?: false,
                isForceOverride = override,
                timeCapture
            )
        )
    }

    override fun onPageStarted(
        id: Long,
        view: WebView?,
        url: String?,
        favicon: Bitmap?,
        timeCapture: EventTimeMetricCapture
    ) {
        view?.handleSizeEvent(id)
        dispatchToRepository(
            id,
            WebViewEvent(WebViewEventId.PAGE_STARTED, timeCapture)
        )
    }

    override fun onPageFinished(
        id: Long,
        view: WebView?,
        url: String?,
        timeCapture: EventTimeMetricCapture
    ) {
        view?.handleSizeEvent(id)
        dispatchToRepository(
            id,
            WebViewEvent(WebViewEventId.PAGE_FINISHED, timeCapture)
        )
        checkAndRegisterVitalsListener(id, view)
    }

    override fun onReceivedError(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        error: WebResourceError?,
        timeCapture: EventTimeMetricCapture
    ) {
        view?.handleSizeEvent(id)
        dispatchToRepository(
            id,
            WebViewEvent(WebViewEventId.ERROR_RECEIVED, timeCapture)
        )
    }

    override fun onReceivedHttpError(
        id: Long,
        view: WebView?,
        request: WebResourceRequest?,
        errorResponse: WebResourceResponse?,
        timeCapture: EventTimeMetricCapture
    ) {
        view?.handleSizeEvent(id)
        dispatchToRepository(
            id,
            WebViewEvent(WebViewEventId.HTTP_ERROR_RECEIVED, timeCapture)
        )
    }

    override fun onUiTraceEnded() = webViewExecutor.execute {
        repository.onUiTraceEnded(uiTraceId)
    }

    private fun attachJavaScriptInterface(webView: WebView) {
        mainThreadExecutor.execute {
            javaScriptInterfaceAttached = javaScriptInterface.attach(webView)
        }
    }

    private fun checkAndRegisterVitalsListener(webViewTraceId: Long, webView: WebView?) {
        webView?.takeIf { shouldCollectVitals(webViewTraceId) }
            ?.takeIf { javaScriptInterfaceAttached }
            ?.also { registerVitalsListener(it, vitalsListenerFactory.create(webViewTraceId)) }
    }

    private fun registerVitalsListener(
        view: WebView,
        listener: InstabugWebVitalsEventListener
    ) {
        mainThreadExecutor.execute { javaScriptInterface.listenToWebVitals(view, listener) }
    }

    private fun shouldCollectVitals(webViewTraceId: Long) = repository.shouldCollectVitals(webViewTraceId)

    private fun WebView.handleSizeEvent(id: Long) {
        runCatching { takeIf { repositoryWebViewSizeIsMissing(id) }
            ?.takeIf { webViewSizeIsReady() }
            ?.let { resolveWebViewSize() }
            ?.let { mapToEvent(it) }
            ?.also { dispatchToRepository(id, it) }
        }.getOrElse { IBGDiagnostics.reportNonFatal(it, "Error while resolving WebViewSize") }
    }

    private fun repositoryWebViewSizeIsMissing(id: Long) =
        !repository.isWebViewSizeResolved(id)

    private fun WebView.webViewSizeIsReady() = !(width == 0 || height == 0)

    private fun WebView.resolveWebViewSize() =
        sizeEventResolver.isFullScreen(widthPx = width, heightPx = height)

    private fun mapToEvent(isFullScreen: Boolean) =
        WebViewSizeReadyEvent(isFullScreen)

    private fun dispatchToRepository(id: Long, event: WebViewEvent) {
        repository.handleEvent(id, event)
    }
}