package com.instabug.apm.uitrace.handler

import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler
import com.instabug.apm.cache.handler.uitrace.UiTraceCacheHandler
import com.instabug.apm.cache.model.UiTraceCacheModel
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.logger.internal.Logger

interface UiTraceHandler {
    fun saveIfPossible(uiTraceCacheModel: UiTraceCacheModel)
    fun end(cacheModel: UiTraceCacheModel)
    fun removeAll()
    fun removeUiHangs()
}

class UiTraceHandlerImpl(
    private val cacheHandler: UiTraceCacheHandler,
    private val metaDataCacheHandler: SessionMetaDataCacheHandler,
    private val configurationProvider: APMConfigurationProvider,
    private val logger: Logger
) : UiTraceHandler {

    private val UiTraceCacheModel.inserted
        get() = cacheStateIncludes(UiTraceCacheModel.UI_TRACE_INSERTED)

    private val UiTraceCacheModel.countsUpdated
        get() = cacheStateIncludes(UiTraceCacheModel.COUNTS_UPDATED)

    override fun saveIfPossible(uiTraceCacheModel: UiTraceCacheModel) {
        uiTraceCacheModel
            .takeIf { it.sessionId != null }
            ?.takeUnless { it.inserted }
            ?.let(cacheHandler::insert)
            ?.takeIf { it != -1L }
            ?.let { uiTraceCacheModel.updateCacheState(UiTraceCacheModel.UI_TRACE_INSERTED) }
    }

    override fun end(cacheModel: UiTraceCacheModel) {
        insertOrUpdate(cacheModel).let {
            if (it) updateCountsAndTrim(cacheModel)
            else logger.logSDKProtected("Session meta data was not updated. Failed to update UITrace")
        }
    }

    private fun insertOrUpdate(cacheModel: UiTraceCacheModel): Boolean = with(cacheModel) {
        when {
            !inserted -> {
                saveIfPossible(cacheModel)
                inserted
            }

            sessionId != null -> cacheHandler.update(this) > 0
            else -> {
                logger.logSDKProtected("UITrace was not updated. APM session is null")
                false
            }
        }
    }

    private fun updateCountsAndTrim(cacheModel: UiTraceCacheModel) = cacheModel
        .takeIf { it.inserted }
        ?.takeUnless { it.countsUpdated }
        ?.run {
            sessionId?.let {
                val countUpdated = metaDataCacheHandler.addToUITracesTotalCount(it, 1)
                if (countUpdated) updateCacheState(UiTraceCacheModel.COUNTS_UPDATED)
                val trimmedRequestLimitCount = cacheHandler
                    .trimToLimit(it, configurationProvider.uiTraceLimitPerRequest)
                if (trimmedRequestLimitCount > 0)
                    metaDataCacheHandler.addToUITracesDroppedCount(it, trimmedRequestLimitCount)
                cacheHandler.clearPreviousUnEndedTraces(it)
                cacheHandler.trimToLimit(configurationProvider.uiTraceStoreLimit)
            }
        }

    override fun removeAll() {
        cacheHandler.removeAll()
        metaDataCacheHandler.resetUITracesCounts()
    }

    override fun removeUiHangs() = cacheHandler.removeUiHangs()
}