package com.instabug.apm.webview.vital

import android.annotation.SuppressLint
import android.os.Build
import android.webkit.JavascriptInterface
import android.webkit.WebView
import androidx.annotation.MainThread
import com.instabug.apm.constants.Constants
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.util.DeviceStateProvider

class InstabugWebViewVitalJSInterface : InstabugWebVitalsEventListener {

    private var listener: InstabugWebVitalsEventListener? = null

    /**
     * Add to WebView as JavaScriptInterface
     */
    @MainThread
    @SuppressLint("NewApi")
    fun attach(webView: WebView): Boolean {
        if (webView.settings.javaScriptEnabled && DeviceStateProvider.getOSVersion() >= Build.VERSION_CODES.O) {
            webView.addJavascriptInterface(this, jsInterfaceName)
            return true
        }
        return false
    }

    @MainThread
    fun detach(wenView: WebView) {
        wenView.removeJavascriptInterface(jsInterfaceName)
    }

    /**
     * Inject JavaScript vitals script into the WebView and dispatch events to the passed listener
     */
    @MainThread
    @SuppressLint("NewApi")
    fun listenToWebVitals(webView: WebView, listener: InstabugWebVitalsEventListener) {
        if (webView.settings.javaScriptEnabled && DeviceStateProvider.getOSVersion() >= Build.VERSION_CODES.O) {
            this.listener = listener
            webView.evaluateJavascript(vitalsScript, null)
        }
    }

    @JavascriptInterface
    override fun onCls(value: Double) {
        listener?.onCls(value)
    }

    @JavascriptInterface
    override fun onFid(value: Double) {
        listener?.onFid(value)
    }

    @JavascriptInterface
    override fun onLCP(value: Double) {
        listener?.onLCP(value)
    }

    @JavascriptInterface
    override fun onError(message: String) {
        IBGDiagnostics.reportNonFatalAndLog(Exception(message), message, Constants.LOG_TAG)
    }

    companion object {
        private const val jsInterfaceName = "IBGVitals"
        private val vitalsScript =
            """        
        (function() {
       
          var script = document.createElement('script');
          // pinned to version 4.0.0 for stability 
          script.src = 'https://unpkg.com/web-vitals@4.0.0/dist/web-vitals.iife.js';

          script.onload = function() {
            // IMPORTANT: Check *again* after our script loads, in case it overwrote or failed.
            if (window.webVitals) { // Does the webVitals object exist?

              // Check specifically for each function before calling it
              if (typeof window.webVitals.onCLS === 'function') {
                window.webVitals.onCLS(function(vital) { $jsInterfaceName.onCls(vital.value); }, { reportAllChanges: true });
              } else {
                $jsInterfaceName.onError('IBGVitals: webVitals.onCLS is not a function after script load.');
              }

              if (typeof window.webVitals.onFID === 'function') {
                window.webVitals.onFID(function(vital) { $jsInterfaceName.onFid(vital.value); }, { reportAllChanges: true });
              } else {
                $jsInterfaceName.onError('IBGVitals: webVitals.onFID is not a function after script load.'); // Log the specific error
              }

              if (typeof window.webVitals.onLCP === 'function') {
                window.webVitals.onLCP(function(vital) { $jsInterfaceName.onLCP(vital.value); }, { reportAllChanges: true });
              } else {
                $jsInterfaceName.onError('IBGVitals: webVitals.onLCP is not a function after script load.');
              }

            } else {
              $jsInterfaceName.onError('IBGVitals: window.webVitals object not found after script load.');
            }
          };

          script.onerror = function() {
             // Handle case where the script *itself* fails to load (network error, etc)
             $jsInterfaceName.onError('IBGVitals: Failed to load web-vitals script from unpkg.');
             
          };

          document.head.appendChild(script);
        })();""".trimIndent()
    }
}