package com.instabug.apm.networking.mapping.sessions;

import static com.instabug.apm.networking.mapping.applaunch.AppLaunchMappingKeys.LAUNCHES_LIST;
import static com.instabug.apm.networking.mapping.networklog.APMNetworkLogMappingKeys.NETWORK_LIST;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.APM_SESSION_DATA;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.DROPPED_OCCURRENCES_REQUEST_LIMIT;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.DROPPED_OCCURRENCES_STORE_LIMIT;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.FRAGMENT_SPANS_OBJECT;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.SESSION_APP_LAUNCHES_LIST;
import static com.instabug.apm.networking.mapping.sessions.SessionMappingKeys.SESSION_NETWORK_LOGS_LIST;

import androidx.annotation.NonNull;

import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.di.Provider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.networking.mapping.applaunch.AppLaunchMapper;
import com.instabug.apm.networking.mapping.fragment_span.FragmentSpanMapper;
import com.instabug.apm.networking.mapping.networklog.APMNetworkLogMapper;
import com.instabug.library.model.v3Session.IBGSessionData;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class SessionMapperImpl implements SessionMapper {

    private AppLaunchMapper appLaunchMapper = ServiceLocator.getAppLaunchMapper();
    private APMNetworkLogMapper networkLogMapper = ServiceLocator.getNetworkLogMapper();
    private FragmentSpanMapper fragmentSpanMapper = ServiceLocator.getFragmentSpanMapper();
    private Provider<SessionFeatureJsonFiller[]> sessionFeatureJsonFillerProvider;

    public SessionMapperImpl(Provider<SessionFeatureJsonFiller[]> sessionFeatureJsonFillerProvider) {
        this.sessionFeatureJsonFillerProvider = sessionFeatureJsonFillerProvider;
    }

    @NonNull
    @Override
    public JSONArray toJsonArray(@NonNull List<SessionCacheModel> sessions) throws JSONException {
        JSONArray jsonArray = new JSONArray();
        for (SessionCacheModel session : sessions) {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put(SessionMappingKeys.SESSION_ID, session.getCoreId());
            jsonObject.put(Constants.BaseRequestKeys.OS_VERSION, session.getOs());
            jsonObject.put(SessionMappingKeys.UID, session.getUuid());
            jsonObject.put(Constants.BaseRequestKeys.APP_VERSION, session.getAppVersion());
            jsonObject.put(SessionMappingKeys.SESSION_START_TIME, session.getStartTimestampMicros());
            jsonObject.put(SessionMappingKeys.SESSION_EXIT_CODE, session.getTerminationStatusCode());

            if (session.getDuration() > 0) {
                jsonObject.put(SessionMappingKeys.SESSION_DURATION, session.getDuration());
            }
            fillAppLaunch(session, jsonObject);
            fillNetworkLogs(session, jsonObject);
            fillFragmentSpans(session, jsonObject);
            fillFeatures(session, jsonObject);
            jsonArray.put(jsonObject);
        }
        return jsonArray;
    }

    private void fillFeatures(SessionCacheModel session, JSONObject jsonObject) {
        SessionFeatureJsonFiller[] fillers = sessionFeatureJsonFillerProvider.invoke();
        for (SessionFeatureJsonFiller filler : fillers) {
            filler.addToJsonObject(session, jsonObject);
        }
    }

    @NonNull
    @Override
    public Map<String, IBGSessionData> toSessionData(@NonNull List<SessionCacheModel> sessions) throws JSONException {
        Map<String, IBGSessionData> sessionDataMap = new HashMap<>();
        for (SessionCacheModel session : sessions) {
            JSONObject sessionObject = new JSONObject();
            fillAppLaunch(session, sessionObject);
            fillNetworkLogs(session, sessionObject);
            fillFragmentSpans(session, sessionObject);
            fillFeatures(session, sessionObject);
            if (!sessionObject.keys().hasNext()) continue;
            IBGSessionData sessionData = new IBGSessionData(APM_SESSION_DATA, sessionObject);
            sessionDataMap.put(session.getCoreId(), sessionData);
        }
        return sessionDataMap;
    }

    private void fillAppLaunch(SessionCacheModel session, JSONObject jsonObject) throws JSONException {
        JSONArray appLaunchesJsonArray = null;
        if (session.getAppLaunches() != null
                && !session.getAppLaunches().isEmpty()) {
            appLaunchesJsonArray = appLaunchMapper.toJsonArray(session.getAppLaunches());
        }
        if (appLaunchesJsonArray != null ||
                (session.getSessionMetaData() != null &&
                        session.getSessionMetaData().getAppLaunchesTotalCount() > 0)) {
            JSONObject appLaunchesJsonObject = new JSONObject();
            if (appLaunchesJsonArray != null) {
                appLaunchesJsonObject.put(LAUNCHES_LIST, appLaunchesJsonArray);
            }
            fillDroppedAppLaunchLimits(session, appLaunchesJsonObject);
            jsonObject.put(SESSION_APP_LAUNCHES_LIST, appLaunchesJsonObject);
        }

    }

    private static void fillDroppedAppLaunchLimits(SessionCacheModel session, JSONObject appLaunchesJsonObject) throws JSONException {
        if (session.getSessionMetaData() != null) {
            int droppedRequestLimit = session.getSessionMetaData().getAppLaunchesDroppedCount();
            if (droppedRequestLimit != 0) {
                appLaunchesJsonObject.put(DROPPED_OCCURRENCES_REQUEST_LIMIT, droppedRequestLimit);
            }
        }
        if (session.getSessionMetaData() != null && session.getAppLaunches() != null) {
            int droppedStoreLimit = session.getSessionMetaData().getAppLaunchesTotalCount()
                    - session.getSessionMetaData().getAppLaunchesDroppedCount()
                    - session.getAppLaunches().size();
            // TODO revert that when fixing negative values for store limit issue with db query inside cycle
            if (droppedStoreLimit > 0) {
                appLaunchesJsonObject.put(DROPPED_OCCURRENCES_STORE_LIMIT, droppedStoreLimit);
            }
        }
    }

    private void fillNetworkLogs(SessionCacheModel session, JSONObject jsonObject) throws JSONException {
        JSONArray networkLogsJsonArray = null;
        if (session.getNetworkLogs() != null
                && !session.getNetworkLogs().isEmpty()) {
            networkLogsJsonArray = networkLogMapper.toJsonArray(session.getNetworkLogs());
        }
        if (networkLogsJsonArray != null ||
                (session.getSessionMetaData() != null &&
                        session.getSessionMetaData().getNetworkLogsTotalCount() > 0)) {
            JSONObject networkLogsJsonObject = new JSONObject();
            if (networkLogsJsonArray != null) {
                networkLogsJsonObject.put(NETWORK_LIST, networkLogsJsonArray);
            }

            if (session.getSessionMetaData() != null) {
                int droppedRequestLimit = session.getSessionMetaData().getNetworkLogsDroppedCount();
                if (droppedRequestLimit != 0) {
                    networkLogsJsonObject.put(DROPPED_OCCURRENCES_REQUEST_LIMIT, droppedRequestLimit);
                }
                if (session.getNetworkLogs() != null) {
                    int droppedStoreLimit = session.getSessionMetaData().getNetworkLogsTotalCount()
                            - session.getSessionMetaData().getNetworkLogsDroppedCount()
                            - session.getNetworkLogs().size();
                    if (droppedStoreLimit != 0) {
                        networkLogsJsonObject.put(DROPPED_OCCURRENCES_STORE_LIMIT, droppedStoreLimit);
                    }
                }
            }
            jsonObject.put(SESSION_NETWORK_LOGS_LIST, networkLogsJsonObject);
        }
    }

    private void fillFragmentSpans(
            @NonNull SessionCacheModel session,
            @NonNull JSONObject sessionJsonObject
    ) throws JSONException {
        if (fragmentSpanMapper != null && session != null && sessionJsonObject != null) {
            JSONObject fragmentSpansObject =
                    fragmentSpanMapper.toJsonObject(
                            session.getFragmentSpans(),
                            session.getSessionMetaData()
                    );
            if (fragmentSpansObject != null) {
                sessionJsonObject.put(FRAGMENT_SPANS_OBJECT, fragmentSpansObject);
            }
        }
    }
}
