package com.instabug.apm.configuration;

import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.model.LogLevel;

public interface APMConfigurationProvider {

    /**
     * Changes APM enabled state depending on BE feature flag
     *
     * @param enabled true to enable, false to disable
     */
    void setAPMFeatureAvailability(boolean enabled);

    /**
     * Gets APM state according to BE feature flag
     *
     * @return true if enabled. false if disabled
     */
    boolean isAPMFeatureAvailable();

    /**
     * Gets APM state depending on SDK integration
     *
     * @return true to enable, false to disable
     */
    boolean isAPMSdkEnabled();

    /**
     * Changes APM enabled state depending on SDK integration
     *
     * @param enabled true to enable, false to disable
     */
    void setAPMSdkEnabled(boolean enabled);

    /**
     * Gets APM enabled state depending on BE feature flag and SDK integration
     *
     * @return true if enabled. false if disabled
     */
    boolean isAPMEnabled();

    /**
     * Gets APM sync interval
     *
     * @return sync interval in seconds
     */
    long getSyncInterval();

    /**
     * Changes APM sync interval
     *
     * @param syncInterval interval in seconds
     */
    void setSyncInterval(long syncInterval);

    // EXECUTION TRACES

    /**
     * Gets execution traces metric state based on BE config
     *
     * @return boolean true if the metric was enabled, false otherwise
     */
    boolean isExecutionTraceFeatureEnabled();

    /**
     * Sets execution traces metric state based on BE config
     *
     * @param enabled boolean true to enable, false to disable
     */
    void setExecutionTraceFeatureEnabled(boolean enabled);

    /**
     * Gets execution traces metric maximum store limit based on BE config
     *
     * @return long maximum store limit
     */
    long getExecutionTraceStoreLimit();

    /**
     * Sets execution traces metric maximum store limit based on BE config
     *
     * @param limit long maximum store limit
     */
    void setExecutionTraceStoreLimit(long limit);

    /**
     * Gets execution traces metric maximum traces limit per single sync request based on BE config
     *
     * @return maximum traces limit per single sync request
     */
    long getExecutionTraceLimitPerRequest();

    /**
     * Gets execution traces metric maximum traces limit per single sync request based on BE config
     *
     * @param limit maximum traces limit per single sync request
     */
    void setExecutionTraceLimitPerRequest(long limit);

    /**
     * Gets custom attributes count per custom trace
     *
     * @return count of custom attributes per custom trace
     */
    int getExecutionTraceStoreAttributesLimit();

    /**
     * Sets custom attributes count per custom trace
     *
     * @param customAttributesCount count of custom attributes per custom trace
     */
    void setExecutionTraceStoreAttributesLimit(int customAttributesCount);

    /**
     * Gets cold app launch metric state based on BE config
     *
     * @return boolean true if the metric was enabled, false otherwise
     */
    boolean isColdAppLaunchesFeatureEnabled();

    /**
     * Gets hot app launch metric state based on BE config
     *
     * @return boolean true if the metric was enabled, false otherwise
     */
    boolean isHotAppLaunchesFeatureEnabled();

    /**
     * Sets hot app launch metric state based on BE config
     *
     * @param enabled boolean true to enable, false to disable
     */
    void setHotAppLaunchesFeatureEnabled(boolean enabled);

    /**
     * Sets cold app launch metric state based on BE config
     *
     * @param enabled boolean true to enable, false to disable
     */
    void setColdAppLaunchesFeatureEnabled(boolean enabled);

    /**
     * Sets cold app launch occurrences maximum store limit based on BE config
     *
     * @param limit long maximum store limit
     */
    void setColdAppLaunchesStoreLimit(long limit);

    /**
     * Sets cold app launch occurrences maximum store limit based on BE config
     *
     * @param limit long maximum store limit
     */
    void setHotAppLaunchesStoreLimit(long limit);

    /**
     * Sets number of cold app launch occurrences to be sent in sync request
     *
     * @param limit number of traces per request
     */
    void setColdAppLaunchesLimitPerRequest(long limit);

    /**
     * Sets number of hot app launch occurrences to be sent in sync request
     *
     * @param limit number of traces per request
     */
    void setHotAppLaunchesLimitPerRequest(long limit);

    /**
     * Gets timestamp of when was the last time APM has synced the data
     *
     * @return long timestamp
     */
    long getLastSyncTime();

    /**
     * Sets timestamp of when was the last time APM has synced the data
     *
     * @param lastSyncTime long time stamp
     */
    void setLastSyncTime(long lastSyncTime);

    /**
     * Gets cold app launch tracking state
     *
     * @return true if enabled, false if disabled
     */
    boolean isColdAppLaunchSDKEnabled();

    /**
     * Gets hot app launch tracking state
     *
     * @return true if enabled, false if disabled
     */
    boolean isHotAppLaunchSDKEnabled();

    /**
     * Sets cold app launch tracking state
     *
     * @param enabled true if enabled, false if disabled
     */
    void setColdAppLaunchSDKEnabled(boolean enabled);

    /**
     * Sets hot app launch tracking state
     *
     * @param enabled true if enabled, false if disabled
     */
    void setHotAppLaunchSDKEnabled(boolean enabled);

    /**
     * Gets cold app launches state, checks for SDK and BE configuration
     *
     * @return true if SDK API and (only and) BE flag are enabled, false if otherwise
     */
    boolean isColdAppLaunchesEnabled();

    /**
     * Gets hot app launches state, checks for SDK and BE configuration
     *
     * @return true if SDK API and (only and) BE flag are enabled, false if otherwise
     */
    boolean isHotAppLaunchesEnabled();

    /**
     * Gets network tracking state
     *
     * @return true if enabled, false if disabled
     */
    boolean isNetworkFeatureEnabled();

    /**
     * Gets network tracking state
     *
     * @return true if enabled and APM is enabled, false if otherwise
     */
    boolean isNetworkEnabled();

    /**
     * Sets state of Network tracking
     *
     * @param enabled true to enable.false to disable
     */
    void setNetworkEnabled(boolean enabled);

    /**
     * Gets limit of network logs to be sent in a single sync request
     *
     * @return network logs limit(count)
     */
    long getNetworkLogsRequestLimit();

    /**
     * Sets limit of network logs to be sent in a single sync request
     *
     * @param limit network logs limit(count)
     */
    void setNetworkLogsLimitPerRequest(long limit);

    /**
     * Gets limit of network logs to be stored locally
     *
     * @return network logs limit(count)
     */
    long getNetworkLogsCacheLimit();

    /**
     * Sets limit of network logs to be stored locally
     *
     * @param limit network logs limit(count)
     */
    void setNetworkLogsStoreLimit(long limit);

    /**
     * Gets timestamp of when was the last time APM has synced cached network logs
     *
     * @return long timestamp
     */
    long getNetworkLogsLastSyncTime();

    /**
     * Sets timestamp of when was the last time APM has synced cached network logs
     *
     * @param lastSyncTime long time stamp
     */
    void setNetworkLogsLastSyncTime(long lastSyncTime);

    /**
     * Gets network graphQL tracking state
     *
     * @return true if enabled and APM is enabled, false if otherwise
     */
    boolean isNetworkGraphQlFeatureEnabled();

    /**
     * Sets state of Network graphQL tracking
     *
     * @param enabled true to enable.false to disable
     */
    void setNetworkGraphQlFeatureEnabled(boolean enabled);

    /**
     * Gets network GRPC interception BE state
     *
     * @return true if enabled, false if otherwise
     */
    boolean isNetworkGrpcInterceptionFeatureEnabled();

    /**
     * Sets the state of network GRPC interception based of BE flag status
     *
     * @param enabled true to enable, false to disable
     */
    void setNetworkGrpcInterceptionFeatureEnabled(boolean enabled);

    /**
     * Gets network GRPC interception state
     *
     * @return true if enabled and APM is enabled and network interception is enabled, false if otherwise
     */
    boolean isNetworkGrpcInterceptionEnabled();

    /**
     * Gets duration threshold of what considered as large frames drop
     *
     * @return long duration
     */
    float getUiTraceLargeDropThreshold();

    /**
     * Sets duration threshold of when Ui trace is considered as large frames drop
     *
     * @param threshold long duration
     */
    void setUiTraceLargeDropThreshold(float threshold);

    /**
     * Gets the duration threshold of what considered as small frames drop
     *
     * @return long duration
     */
    float getUiTraceSmallDropThreshold();

    /**
     * Sets duration threshold of when Ui trace is considered as small frames drop
     *
     * @param threshold long duration
     */
    void setUiTraceSmallDropThreshold(float threshold);

    /**
     * Gets limit of UiTrace to be sent pre sync request
     *
     * @return long timestamp
     */
    long getUiTraceLimitPerRequest();

    /**
     * Sets limit of UiTrace to be sent pre sync request
     *
     * @param size long size
     */
    void setUiTraceLimitPerRequest(long size);

    /**
     * Gets limit of UiTrace to be persisted locally
     * If exceeded the limit clear the most old records
     *
     * @return long limit
     */
    long getUiTraceStoreLimit();

    /**
     * Sets limit of  UiTrace to be persisted locally
     *
     * @param limit long size
     */
    void setUiTraceStoreLimit(long limit);

    /**
     * Gets ui trace tracing SDK API state
     *
     * @return true if enabled, false if disabled
     */
    boolean isUiTraceSdkEnabled();

    /**
     * Gets ui trace tracing SDK API state
     *
     * @return true when it's enabled and at least one sub-feature is enabled
     */
    boolean isUiTraceSdkEnabledIncludingASubFeature();

    /**
     * Sets ui trace tracing SDK API state
     *
     * @param enabled define the state, true to enable, false to disable
     */
    void setUiTraceSdkEnabled(boolean enabled);

    /**
     * Gets ui hangs BE feature-flag status
     *
     * @return true if enabled, false if disabled, or BE flag is missing
     */
    boolean isUiHangsFeatureEnabled();

    /**
     * Sets ui hangs BE feature-flag status
     *
     * @param enabled define the state, true if enable, false if disable, default is disabled
     */
    void setUiHangsFeatureEnabled(boolean enabled);

    /**
     * Gets UI hangs SDK API status
     *
     * @return true if enabled, false if disabled
     */
    boolean isUiHangsSdkEnabled();

    /**
     * Sets UI hangs SDK API status
     *
     * @param enabled define the state, true to enable, false to disable
     */
    void setUiHangsSdkEnabled(boolean enabled);

    /**
     * Gets ui trace tracing state, checks for SDK and BE configuration
     *
     * @return true if SDK API and (only and) BE flag are enabled, false if otherwise
     */
    boolean isUiTraceEnabled();

    /**
     * Gets ui hangs state, checks for SDK and BE configuration
     *
     * @return true if SDK API and BE flag are enabled, false otherwise
     */
    boolean isAutoUiHangsEnabled();

    /**
     * Gets UI loading metric feature-BE flag status
     */
    boolean isUiLoadingMetricsFeatureEnabled();

    /**
     * Sets Ui loading metric BE feature-flag status
     *
     * @param enabled define the state, true if enable, false if disable, default is disabled
     */
    void setUiLoadingMetricsFeatureEnabled(boolean enabled);

    /**
     * Gets UI Traces feature-BE status
     * it's determined by isUiHangsFeatureEnabled and isUiLoadingMetricFeatureEnabled
     *
     * @return true if at least one of uiHangs feature flag or uiLoading metric feature flags is enabled
     */
    boolean isUiTraceFeatureEnabled();

    /**
     * Gets UI loading SDK API status
     *
     * @return true if enabled, false if disabled
     */
    boolean isUiLoadingMetricsSdkEnabled();

    /**
     * Sets UI Loading SDK API status
     *
     * @param enabled define the state, true to enable, false to disable
     */
    void setUiLoadingMetricsSdkEnabled(boolean enabled);

    /**
     * Gets ui loading metric state, checks for SDK and BE configuration
     *
     * @return true if SDK API and BE flag are enabled, false otherwise
     */
    boolean isAutoUiLoadingMetricsEnabled();

    /**
     * Gets ui loading metric state, along with APM and UiTRaces SDK value
     *
     * @return true if screenLoading SDK and BE, APM, and UiTRaces SDK are enabled, false otherwise
     */
    boolean isAutoUiLoadingMetricsFullyEnabled();

    /**
     * Sets EndScreenLoading BE feature-flag status
     *
     * @param enabled define the state, true if enable, false if disable, default is disabled
     */
    void setEndScreenLoadingFeatureEnabled(boolean enabled);

    /**
     * Gets EndScreenLoading BE feature-flag status
     *
     * @return true if enabled, false if disabled, or BE flag is missing
     */
    boolean isEndScreenLoadingFeatureEnabled();

    /**
     * Sets state of APM session crash detection
     *
     * @param isCrashDetectionEnabled true if enabled, false if disabled
     */
    void setCrashDetectionEnabled(boolean isCrashDetectionEnabled);

    /**
     * Gets state of APM session crash detection
     *
     * @return true if enabled, false if disabled
     */
    boolean isCrashDetectionEnabled();

    /**
     * Sets state of debug mode
     *
     * @param debugModeEnabled true if enabled, false if disabled
     */
    void setDebugModeEnabled(boolean debugModeEnabled);

    /**
     * Sets state of debug mode
     *
     * @return true if enabled, false if disabled
     */
    boolean isDebugModeEnabled();

    /**
     * Gets the printed logs priority filter.
     *
     * @return the level if it was set or {@link com.instabug.apm.constants.DefaultValues#DEFAULT_LOG_LEVEL} otherwise
     * @see LogLevel
     */
    @LogLevel
    int getLogLevel();

    /**
     * Sets attributes count per network trace
     *
     * @param attributesStoreLimit count of attributes per network trace
     */
    void setNetworkLogsStoreAttributesLimit(int attributesStoreLimit);

    /**
     * Gets attributes count per network trace
     *
     * @return count of attributes per network trace
     */
    int getNetworkLogsStoreAttributesLimit();

    /**
     * Sets the ColdAppLaunchEnd API state
     *
     * @param isEndColdLaunchEnabled true if enabled, false if disabled
     */
    void setEndColdAppLaunchFeatureEnabled(boolean isEndColdLaunchEnabled);

    boolean isEndColdAppLaunchFeatureEnabled();

    /**
     * Sets the HotAppLaunchEnd API state
     *
     * @param isEndHotLaunchEnabled true if enabled, false if disabled
     */
    void setEndHotAppLaunchFeatureEnabled(boolean isEndHotLaunchEnabled);

    boolean isEndHotAppLaunchFeatureEnabled();

    /**
     * Sets Warm app launch BE feature enable flag
     */
    void setWarmAppLaunchFeatureEnabled(boolean enabled);

    /**
     * @return Warm app launch BE feature enable flag
     */
    boolean isWarmAppLaunchFeatureEnabled();

    /**
     * Sets Warm app launch limit per request
     */
    void setWarmAppLaunchRequestLimit(long limit);

    /**
     * Sets Warm app launch store limit
     */
    void setWarmAppLaunchStoreLimit(long limit);

    /**
     * Sets end app warm launch BE feature enable flag
     */
    void setEndWarmAppLaunchFeatureEnabled(boolean enabled);

    /**
     * @return end app warm launch BE feature enable flag
     */
    boolean isEndWarmAppLaunchFeatureEnabled();

    /**
     * Sets Warm app launch SDK enable flag
     */
    void setWarmAppLaunchSdkEnabled(boolean enabled);

    /**
     * @return Warm app launch SDK enable flag
     */
    boolean isWarmAppLaunchSdkEnabled();

    /**
     * Checks if warm app launch is enabled from SDK & BE side including APM and Instabug status
     */
    boolean isWarmAppLaunchEnabled();

    /**
     * Reset Warm app launch BE enable flag
     */
    void resetWarmAppLaunchFeatureFlag();

    /**
     * Reset Warm app launch limit per request
     */
    void resetWarmAppLaunchRequestLimit();

    /**
     * Reset Warm app launch store limit
     */
    void resetWarmAppLaunchStoreLimit();

    /**
     * Reset End warm app launch BE enable flag
     */
    void resetEndWarmAppLaunchFeatureFlag();

    /**
     * Reset all warm app launch configurations
     */
    void resetWarmAppLaunchConfigurations();

    /**
     * Checks if any of app launch types is enabled
     *
     * @return true if at least one app launch type is enabled, false if all app launch types are not enabled
     */
    boolean isAppLaunchesEnabled();

    /**
     * get StoreLimit for an app launch by type
     */
    long getAppLaunchStoreLimit(@AppLaunchType String appLaunchType);

    /**
     * get RequestLimit for an app launch by type
     */
    long getAppLaunchRequestLimit(@AppLaunchType String appLaunchType);

    /**
     * Get app launch BE feature enable flag by type
     */
    boolean isAppLaunchFeatureEnabled(@AppLaunchType String appLaunchType);

    /**
     * Get app launch SDK feature enable flag by type
     */
    boolean isAppLaunchSdkEnabled(@AppLaunchType String appLaunchType);

    /**
     * Get end app launch BE feature enable flag by type
     */
    boolean isEndAppLaunchFeatureEnabled(@AppLaunchType String appLaunchType);

    /**
     * Checks an app launch type is enabled from SDK & BE side including APM and Instabug status
     */
    boolean isAppLaunchEnabled(@AppLaunchType String appLaunchType);

    /**
     * Sets the Experiments BE state
     *
     * @param enabled true if enabled, false if disabled
     */
    void setExperimentsFeatureEnabled(boolean enabled);

    /**
     * Reset the experiments BE state to it's default value
     */
    void resetExperimentsFeatureEnabledFlag();

    /**
     * Gets the Experiments BE state
     *
     * @return true for enabled, false for disabled
     */
    boolean isExperimentsFeatureEnabled();

    /**
     * Set the limit per request for experiments
     */
    void setExperimentsLimitPerRequest(int limit);

    /**
     * Reset the limit per request for experiments to it's default value
     */
    void resetExperimentsLimitPerRequest();

    /**
     * Get limit per request for experiments
     */
    int getExperimentsLimitPerRequest();

    /**
     * Sets last APM Sessions request time
     */
    void setLastApmSessionsRequestStartedAt(long time);

    /**
     * Sets last APM Sessions limited until period
     */
    void setApmSessionsLimitedUntil(int period);

    /**
     * returns true if the current time between request start time
     * and and the limitation period
     */
    boolean isApmSessionsRateLimited();

    void setSendLegacyAPMSessions(boolean status);

    boolean shouldSendLegacyAPMSessions();

    /**
     * Set Session store limit BE feature flag state
     */
    void setSessionStoreLimitEnabled(boolean enabled);

    /**
     * Get Session store limit BE feature flag state
     */
    boolean isSessionStoreLimitEnabled();

    /**
     * Remove the current stored state for Session store limit flag
     */
    void resetSessionStoreLimitEnabledFlag();

    /**
     * Set maximum number of sessions to be stored on disk
     */
    void setSessionStoreLimit(int limit);

    /**
     * Get maximum number of sessions to be stored on disk
     */
    int getSessionStoreLimit();

    /**
     * Remove the current stored session store limit
     */
    void resetSessionStoreLimit();

    /**
     * set v3 session enabled to be used to ignore v2 session starting event
     */
    void setShouldDependOnV3Session(boolean shouldDependOnV3Session);

    /**
     * get v3 session enabled to be used to ignore v2 session starting event
     */
    boolean shouldDependOnV3Session();

    /**
     * get store limit for fragments
     */
    int getFragmentSpansStoreLimit();

    /**
     * get limit per request for fragments
     */
    int getFragmentSpansLimitPerRequest();

    /**
     * Sets the Fragments BE state
     *
     * @param enabled true if enabled, false if disabled
     */
    void setFragmentSpansFeatureEnabledFlag(boolean enabled);

    /**
     * Set maximum number of fragments to be stored on disk
     */
    void setFragmentSpansStoreLimit(int limit);

    /**
     * Set the limit per request for fragments
     */
    void setFragmentSpansLimitPerRequest(int limit);

    /**
     * Get Fragments BE feature flag state
     */
    boolean isFragmentSpansFeatureEnabled();

    /**
     * Set the limit per request for experiments
     */
    void resetFragmentSpansFeatureEnabledFlag();

    /**
     * Set maximum number of fragments to be stored on disk to default value
     */
    void resetFragmentSpansStoreLimit();

    /**
     * Reset the limit per request for fragments to its default value
     */
    void resetFragmentSpansLimitPerRequest();

    /**
     * Sets fragment spans metric SDK API state
     *
     * @param enabled define the state, true to enable, false to disable
     */
    void setFragmentSpansSDKEnabled(boolean enabled);

    /**
     * Gets Fragment spans API status
     *
     * @return true if enabled, false if disabled
     */
    boolean isFragmentSpansSDKEnabled();


    /**
     * Gets fragments spans state, checks for SDK and BE configuration
     *
     * @return true if SDK API and BE flag are enabled, false otherwise
     */
    boolean isFragmentSpansEnabled();

    /**
     * Gets W3C external trace id feature enablement state.
     *
     * @return true if enabled false otherwise
     */
    boolean isW3CNetworkExternalTraceIdEnabled();

    /**
     * Gets generated W3C external trace id attaching feature BE enablement state.
     *
     * @return true if enabled false otherwise
     */
    boolean isAttachingGeneratedW3CExternalTraceIdFeatureAvailable();

    /**
     * Gets generated W3C external trace id attaching feature enablement state.
     *
     * @return true if enabled false otherwise
     */
    boolean isAttachingGeneratedW3CExternalTraceIdEnabled();

    /**
     * Gets captured W3C external trace id attaching feature BE enablement state.
     *
     * @return true if enabled false otherwise
     */
    boolean isAttachingCapturedW3CExternalTraceIdFeatureAvailable();

    /**
     * Gets captured W3C external trace id attaching feature enablement state.
     *
     * @return true if enabled false otherwise
     */
    boolean isAttachingCapturedW3CExternalTraceIdEnabled();
}
