package com.instabug.apm.appflow.usecases

import com.instabug.apm.appflow.configuration.AppFlowConfigurationProvider
import com.instabug.apm.appflow.handler.AppFlowHandler
import com.instabug.apm.appflow.log.logAttributeWasNotAddedNoMatchingActiveFlow
import com.instabug.apm.appflow.log.logAttributesExceededMaxAllowedCount
import com.instabug.apm.appflow.model.AppFlowAttribute
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.TimeCaptureBoundModel
import com.instabug.apm.sanitization.Sanitizer
import com.instabug.apm.sanitization.Validator

class SetFlowAttributeUseCase(
    private val handler: AppFlowHandler,
    private val logger: Logger,
    private val configurationsValidator: Validator<Unit>,
    private val attributeValidator: Validator<AppFlowAttribute>,
    private val attributeSanitizer: Sanitizer<AppFlowAttribute>,
    private val appFlowConfigurations: AppFlowConfigurationProvider,
    private val refreshBackgroundFlowUseCase: UseCase<Long, Boolean?>
) : UseCase<TimeCaptureBoundModel<AppFlowAttribute>, Unit> {

    override fun invoke(param: TimeCaptureBoundModel<AppFlowAttribute>) {
        param.takeIf { configurationsValidator.isValid(Unit) }
            ?.takeIf { attributeValidator.isValid(param.model) }
            ?.let { attributeSanitizer.sanitize(param.model) }
            ?.let { sanitized -> Pair(param.model, sanitized) }
            ?.takeUnless { passedAbandonmentThreshold(param) }
            ?.also { handleAttribute(it.first, it.second) }
    }

    private fun passedAbandonmentThreshold(param: TimeCaptureBoundModel<AppFlowAttribute>) =
        (refreshBackgroundFlowUseCase(param.timeCapture.getTimeStampMillis()) == true)
            .also { if (it) logNoMatchingActiveFlow(param.model.name) }

    private fun handleAttribute(
        originalAttribute: AppFlowAttribute,
        sanitizedAttribute: AppFlowAttribute
    ) {
        sanitizedAttribute.name?.let { appFlowName ->
            sanitizedAttribute.key?.let { key ->
                sanitizedAttribute.value?.also { value ->
                    addOrUpdateAttribute(appFlowName, key, value, originalAttribute)
                } ?: removeAttribute(appFlowName, key)
            }
        }
    }

    private fun addOrUpdateAttribute(
        flowName: String,
        key: String,
        value: String,
        originalAttribute: AppFlowAttribute
    ) = handler.updateAttributeValue(flowName, key, value)
        ?.takeUnless { it }
        ?.let { validateCountAndAddAttribute(flowName, key, value, originalAttribute) }

    private fun validateCountAndAddAttribute(
        flowName: String,
        key: String,
        value: String,
        originalFlow: AppFlowAttribute
    ) = handler.getAttributeCount(flowName)?.also { count ->
        val attributesStoreLimit = appFlowConfigurations.attributesStoreLimit
        takeIf { count < attributesStoreLimit }
            ?.also { addAttribute(originalFlow.name, flowName, key, value) }
            ?: logAttributeMaxCountReached(originalFlow, attributesStoreLimit)
    }

    private fun addAttribute(
        originalFlowName: String?,
        flowName: String,
        key: String,
        value: String
    ) {
        handler.addAttribute(flowName, key, value)
            ?.takeUnless { it }
            ?.let { logNoMatchingActiveFlow(originalFlowName) }
    }

    private fun logNoMatchingActiveFlow(flowName: String?) {
        flowName?.let(logger::logAttributeWasNotAddedNoMatchingActiveFlow)
    }

    private fun logAttributeMaxCountReached(
        attribute: AppFlowAttribute,
        attributesStoreLimit: Int
    ) {
        logger.logAttributesExceededMaxAllowedCount(
            attribute.name,
            attribute.key,
            attributesStoreLimit
        )
    }

    private fun removeAttribute(flowName: String, key: String) =
        handler.removeAttribute(flowName, key)
}