package com.instabug.apm.cache.handler.session;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.cache.model.SessionMetaData;

public interface SessionMetaDataCacheHandler {

    /**
     * Increments App launches total count by x
     *
     * @param sessionID  id of the session to increment app launches
     *                   total count for
     * @param addedCount count to be added to total app launches count
     */
    void addToAppLaunchesTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increments app launches dropped count by x
     *
     * @param sessionID  id of the session to increment launches
     *                   dropped count for
     * @param addedCount count to be added to dropped launches count
     */
    void addToAppLaunchesDroppedCount(String sessionID, int addedCount);

    /**
     * Increments Traces total count by x
     *
     * @param sessionID  id of the session to increment traces
     *                   total count for
     * @param addedCount count to be added to total traces count
     */
    void addToTracesTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increments Traces dropped count by x
     *
     * @param sessionID  id of the session to increment traces
     *                   dropped count for
     * @param addedCount count to be added to dropped traces count
     */
    void addToTracesDroppedCount(@NonNull String sessionID, int addedCount);

    /**
     * Increments Network logs total count by x
     *
     * @param sessionID  id of the session to increment network logs
     *                   total count for
     * @param addedCount count to be added to total network logs count
     */
    void addToNetworkLogsTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increments network logs dropped count by x
     *
     * @param sessionID  id of the session to increment network logs
     *                   dropped count for
     * @param addedCount count to be added to dropped network logs count
     */
    void addToNetworkLogsDroppedCount(String sessionID, int addedCount);

    /**
     * Increments UI traces total count by x
     *
     * @param sessionID  id of the session to increment ui traces
     *                   total count for
     * @param addedCount count to be added to total ui traces count
     * @return true if the count is successfully added and false if not
     */
    boolean addToUITracesTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increments ui traces dropped count by x
     *
     * @param sessionID  id of the session to increment ui traces
     *                   dropped count for
     * @param addedCount count to be added to dropped ui traces count
     */
    void addToUITracesDroppedCount(String sessionID, int addedCount);

    /**
     * Sets APM experiments total count
     *
     * @param sessionID id of the session to increment ui traces
     *                  total count for
     * @param count     experiments total count
     */
    void setExperimentsTotalCount(String sessionID, int count);


    /**
     * Increments fragments total count by x
     *
     * @param sessionID  id of the session to increment fragment spans
     *                   total count for
     * @param addedCount count to be added to total fragment spans count
     */
    void addToFragmentSpansTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increments fragments dropped count by x
     *
     * @param sessionID  id of the session to increment fragment spans
     *                   total count for
     * @param addedCount count to be added to total fragment spans count
     */
    void addToFragmentSpansDroppedCount(@NonNull String sessionID, int addedCount);

    /**
     * Increment compose spans total count for a given session
     */
    void addToComposeSpansTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increment compose spans dropped count for a given session
     */
    void addToComposeSpansDroppedCount(@NonNull String sessionID, int addedCount);

    /**
     * Increment webView traces total count for a given session
     */
    void addToWebViewTracesTotalCount(@NonNull String sessionID, int addedCount);

    /**
     * Increment webView traces dropped count for a given session
     */
    void addToWebViewTracesDroppedCount(@NonNull String sessionID, int addedCount);

    /**
     * Sets App flow total count for a given session
     */
    void setAppFlowTotalCount(@NonNull String sessionID, int count);

    /**
     * Sets App flow dropped count for a given session
     */
    void setAppFlowDroppedCount(@NonNull String sessionID, int count);

    /**
     * Gets meta data of the session for passed sessionID
     *
     * @param sessionID id of the session to get meta data for
     * @return SessionMetaData of the session. Null if no data found
     */
    @Nullable
    SessionMetaData getSessionMetaData(String sessionID);

    /**
     * Resets all app launches meta data
     */
    void resetAppLaunchesCounts();

    /**
     * Resets all custom traces meta data
     */
    void resetTracesCounts();

    /**
     * Resets all network logs meta data
     */
    void resetNetworkLogsCounts();

    /**
     * Resets all UI traces meta data
     */
    void resetUITracesCounts();

    /**
     * Resets all Experiments meta data
     */
    void resetExperimentsCount();

    /**
     * Resets all Fragment spans meta data
     */
    void resetFragmentSpansCounts();

    /**
     * Resets all Compose spans meta data
     */
    void resetComposeSpansCounts();

    /**
     * Resets all webView traces meta data
     */
    void resetWebViewTracesCounts();

    /**
     * Resets all App flow meta data
     */
    void resetAppFlowCounts();
}
