package com.instabug.apm.handler.session;

import static com.instabug.apm.constants.Constants.SessionEndStatusCode.UNDEFINED;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;

import com.instabug.apm.APMPlugin;
import com.instabug.apm.cache.handler.session.SessionCacheHandler;
import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.configuration.APMStateProvider;
import com.instabug.apm.constants.Constants.SessionSyncStatus;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.handler.experiment.ExperimentHandler;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.core.InstabugCore;
import com.instabug.library.internal.utils.stability.execution.Executable;
import com.instabug.library.internal.utils.stability.handler.exception.ExceptionHandler;
import com.instabug.library.model.common.Session;
import com.instabug.library.model.common.SessionVersion;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

public class SessionHandlerImpl implements SessionHandler {

    @NonNull
    private final APMConfigurationProvider configurationProvider;
    @NonNull
    private final APMStateProvider stateProvider;
    @NonNull
    private final SessionCacheHandler cacheHandler;
    @NonNull
    private final ExceptionHandler exceptionHandler;
    @NonNull
    private final Logger logger;
    @Nullable
    private Runnable sessionStartRunnable;

    @Nullable
    private volatile SessionCacheModel currentSession;
    @NonNull
    private final Executor sessionExecutor;

    public SessionHandlerImpl(@NonNull APMConfigurationProvider configurationProvider,
                              @NonNull APMStateProvider stateProvider,
                              @NonNull SessionCacheHandler cacheHandler,
                              @NonNull ExceptionHandler exceptionHandler,
                              @NonNull Logger logger) {
        this.configurationProvider = configurationProvider;
        this.stateProvider = stateProvider;
        this.cacheHandler = cacheHandler;
        this.exceptionHandler = exceptionHandler;
        this.logger = logger;
        sessionExecutor = ServiceLocator.getSyncThreadExecutor();
    }

    @Override
    public void startSession(@NonNull Session coreSession) {
        if (configurationProvider.isAPMEnabled() && getCurrentSession() == null && sessionStartRunnable == null) {
            sessionStartRunnable = getSessionStartRunnable(coreSession);
            if (configurationProvider.isAPMEnabled()) {
                sessionStartRunnable.run();
            }
        }
    }

    @Override
    public void endSession() {
        endSession(UNDEFINED);
    }

    @Override
    public void endSession(final int reason) {
        exceptionHandler.execute(new Executable() {
            @Override
            public void execute() {
                sessionStartRunnable = null;
                final SessionCacheModel finalCurrentSession = getCurrentSession();
                if (finalCurrentSession != null) {
                    long startTime = finalCurrentSession.getStartNanoTime();
                    long elapsedTime = TimeUnit.NANOSECONDS.toMicros(System.nanoTime() - startTime);
                    final SessionCacheModel cacheModel = new SessionCacheModel(finalCurrentSession.getId(),
                            finalCurrentSession.getCoreId(),
                            finalCurrentSession.getOs(),
                            finalCurrentSession.getAppVersion(),
                            finalCurrentSession.getUuid(),
                            elapsedTime, finalCurrentSession.getStartTimestampMicros(),
                            finalCurrentSession.getStartNanoTime(),
                            finalCurrentSession.getVersion(),
                            reason,
                            SessionSyncStatus.UNDEFINED);
                    setCurrentSession(null);
                    sessionExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            cacheHandler.update(cacheModel);
                        }
                    });
                    storeExperimentsAsync(cacheModel);
                    logger.logSDKDebug("Ending session #" + cacheModel.getId());
                } else {
                    logger.logSDKProtected("Attempted to end session without calling start");
                }
                configurationProvider.setShouldDependOnV3Session(InstabugCore.isV3SessionEnabled());
            }
        });
    }

    private void storeExperimentsAsync(SessionCacheModel cacheModel) {
        if (cacheModel.getVersion().equals(SessionVersion.V3)) return;
        ExperimentHandler experimentHandler = ServiceLocator.getExperimentHandler();
        if (experimentHandler != null) {
            experimentHandler.storeExperimentsAsync(cacheModel.getId());
        }
    }

    private synchronized void setCurrentSession(@Nullable SessionCacheModel sessionModel) {
        currentSession = sessionModel;
    }

    @Nullable
    @Override
    public synchronized SessionCacheModel getCurrentSession() {
        return currentSession;
    }

    @Nullable
    @Override
    public String getCurrentCoreSessionId() {
        SessionCacheModel localSession = currentSession;
        if (localSession != null) {
            return localSession.getCoreId();
        }
        return null;
    }

    @NonNull
    @Override
    public List<SessionCacheModel> getReadyToBeSentSessions() {
        return cacheHandler.getReadyToBeSentSessions();
    }

    @Nullable
    @Override
    public SessionCacheModel getNextSession(String currentSessionID) {
        return cacheHandler.getNextSession(currentSessionID);
    }

    @Override
    public void changeSessionSyncStatus(@NonNull List<String> sessionsIDs,
                                        @SessionSyncStatus int syncStatus) {
        cacheHandler.changeSessionSyncStatus(sessionsIDs, syncStatus);
    }

    @Override
    public void deleteSessionsBySyncStatus(int syncStatus) {
        cacheHandler.deleteSessionsBySyncStatus(syncStatus);
    }

    @Override
    public void updateSessionEndReason(@NonNull final String coreSessionId, final long sessionDuration, final int reason) {
        ServiceLocator.getIOExecutor().execute(new Runnable() {
            @Override
            public void run() {
                cacheHandler.updateEndReason(coreSessionId, sessionDuration, reason);
            }
        });
    }

    @NonNull
    @Override
    public List<SessionCacheModel> getSessionsByCoreIds(@NonNull List<String> coreIds) {
        return cacheHandler.queryByCoreIds(coreIds);
    }

    @Override
    public void deleteSessionsByCoreIds(@NonNull List<String> coreIds) {
        cacheHandler.deleteByCoreIds(coreIds);
    }

    @Nullable
    @Override
    public String getPreviousSessionId(@NonNull String currentSession) {
        if (currentSession != null) {
            SessionCacheModel previousSession = cacheHandler.getPreviousSession(currentSession);
            if (previousSession != null) {
                return previousSession.getId();
            }
        }
        return null;
    }

    @NonNull
    private Runnable getSessionStartRunnable(@NonNull final Session coreSession) {
        return new Runnable() {
            @Override
            public void run() {
                exceptionHandler.execute(new Executable() {
                    @Override
                    public void execute() {
                        sessionExecutor.execute(new Runnable() {
                            @Override
                            public void run() {
                                synchronized (APMPlugin.lock) {
                                    if (getCurrentSession() == null) {
                                        setCurrentSession(cacheHandler.insert(coreSession));
                                        final SessionCacheModel finalCurrentSession = getCurrentSession();
                                        if (finalCurrentSession != null) {
                                            resetDroppedSessionsCountIfPossible(finalCurrentSession);
                                            trimSessionsIfPossible();
                                            SessionObserverHandler.notifyNewSessionStarted(finalCurrentSession, cacheHandler.getPreviousSession(finalCurrentSession.getId()));
                                        }
                                    } else {
                                        logger.logSDKProtected("Attempted to start session while another session is already running. Skipping..");
                                    }
                                }
                            }
                        });
                    }
                });
            }
        };
    }

    @WorkerThread
    public void resetDroppedSessionsCountIfPossible(@NonNull SessionCacheModel insertedSession) {
        if ("1".equals(insertedSession.getId()) && stateProvider != null) {
            stateProvider.resetStoreLimitDroppedSessionCount();
        }
    }

    @WorkerThread
    public void trimSessionsIfPossible() {
        if (stateProvider != null &&
                configurationProvider != null &&
                configurationProvider.isSessionStoreLimitEnabled()) {
            int storeLimit = configurationProvider.getSessionStoreLimit();
            int droppedSessionsCount = cacheHandler.trimSessions(storeLimit);
            if (droppedSessionsCount > 0) {
                stateProvider.incrementStoreLimitDroppedSessionsCount(droppedSessionsCount);
            }
        }
    }
}