package com.instabug.apm.handler.networklog;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.model.APMNetworkLog;
import com.instabug.library.Instabug;

import java.util.List;
import java.util.Map;

public interface NetworkLogHandler {

    /**
     * Inserts network log in cache
     *
     * @param networkLog network log to be cached
     * @return id of cached network log
     */
    long insertNetworkLog(@NonNull APMNetworkLog networkLog);

    /**
     * Update cached network log values
     *
     * @param networkLog updated network log object
     */
    void updateNetworkLog(APMNetworkLog networkLog);

    /**
     * Removes invalid network logs
     * Invalid network log doesn't have (status code, error code, error message)
     */
    void cleanUp();

    /**
     * Removes all cached network logs
     */
    void removeAll();

    /**
     * Removes all graphQl related data
     */
    void removeGraphQlData();

    /**
     * Removes all GRPC related data
     */
    void removeGrpcData();

    /**
     * Gets list of cached ended network logs for passed session id
     *
     * @param sessionID id of the session to get cached network
     *                  logs for
     * @return List of {@link APMNetworkLog}
     */
    @Nullable
    List<APMNetworkLog> getEndedNetworkLogsForSession(String sessionID);

    /**
     * Force stop and ignore the currently running {@link APMNetworkLog} if the SDK was disabled
     *
     * @see Instabug#disable()
     */
    void forceStop();

    /**
     * Checks on new attribute if matches the SDK constraints
     *
     * @param traceName name of the associated trace
     * @param key       attribute key
     * @param value     attribute value
     */
    boolean isValidAttribute(String traceName, String key, String value);

    /**
     * Add new attribute to specific trace
     *
     * @param traceId   id of the associated trace
     * @param traceName name of the associated trace
     * @param key       attribute key
     * @param value     attribute value
     */
    void addAttribute(long traceId, String traceName, boolean executedInBackground, String key, String value);

    /**
     * Get attribute map of specific trace
     *
     * @param traceId   id of the associated trace
     * @return Map of attributes' keys and values
     */
    public Map<String, String> getTraceAttributes(long traceId);

    /**
     * Removes all W3C external trace id related data including generated and captured.
     */
    void clearW3CExternalTraceIdCache();

    /**
     * Removes generated W3C trace id.
     */
    void clearGeneratedW3CExternalTraceIdCache();

    /**
     * Removes captured W3C trace id.
     */
    void clearCapturedW3CExternalTraceIdCache();
}
