package com.instabug.apm.cache.handler.experiment

import android.content.ContentValues
import android.database.Cursor
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.internal.storage.cache.db.DatabaseManager
import com.instabug.library.internal.storage.cache.db.InstabugDbContract.APMExperimentEntry.*
import com.instabug.library.map.TwoWayMapper
import com.instabug.library.safetyUse

class ExperimentCacheHandlerImpl(
    private val databaseManager: DatabaseManager,
    private val logger: Logger,
    private val mapper: TwoWayMapper<List<String>, ByteArray>
) : ExperimentCacheHandler {

    override fun addExperiments(experiments: List<String>, sessionId: String): Long {
        try {
            val dbWrapper = databaseManager.openDatabase()
            val contentValue = getContentValue(experiments, sessionId)
            return dbWrapper.insertWithOnConflict(TABLE_NAME, null, contentValue)
        } catch (exception: Exception) {
            logger.logSDKError("DB execution a sql failed", exception)
            IBGDiagnostics.reportNonFatal(exception, "DB execution a sql failed")
        }
        return -1
    }

    override fun getExperiments(sessionId: String): List<String> {
        try {
            val dbWrapper = databaseManager.openDatabase()
            dbWrapper.query(
                TABLE_NAME,
                arrayOf(COLUMN_EXPERIMENT_ARRAY),
                "$COLUMN_SESSION_ID = ?",
                arrayOf(sessionId),
                null,
                null,
                null

            )?.let { cursor ->
                return getExperiments(cursor)
            }
        } catch (exception: Exception) {
            logger.logSDKError("DB execution a sql failed", exception)
            IBGDiagnostics.reportNonFatal(exception, "DB execution a sql failed")
        }
        return emptyList()
    }

    override fun clear() {
        try {
            val dbWrapper = databaseManager.openDatabase()
            dbWrapper.delete(
                TABLE_NAME,
                null,
                null
            )
        } catch (exception: Exception) {
            logger.logSDKError("DB execution a sql failed", exception)
            IBGDiagnostics.reportNonFatal(exception, "DB execution a sql failed")
        }
    }

    private fun getContentValue(experiments: List<String>, sessionId: String): ContentValues =
        ContentValues().apply {
            put(COLUMN_SESSION_ID, sessionId)
            put(COLUMN_EXPERIMENT_ARRAY, mapper.mapForwards(experiments))
        }

    private fun getExperiments(cursor: Cursor): List<String> =
        cursor.safetyUse {
            if (cursor.moveToFirst()) {
                val experimentsColumnIndex = cursor.getColumnIndex(COLUMN_EXPERIMENT_ARRAY)
                val experimentsByteArray = cursor.getBlob(experimentsColumnIndex)
                mapper.mapBackwards(experimentsByteArray)
            } else {
                emptyList()
            }
        }

}