package com.instabug.apm.uitrace.uihangs

import android.os.Build
import android.os.Handler
import android.view.Choreographer
import androidx.annotation.RequiresApi
import androidx.annotation.VisibleForTesting
import com.instabug.apm.logger.internal.Logger
import com.instabug.library.core.InstabugCore
import java.util.Collections
import java.util.WeakHashMap
import java.util.concurrent.Executor

interface APMChoreographer {
    fun addCallback(callback: Choreographer.FrameCallback)
    fun removeCallback(callback: Choreographer.FrameCallback)
}

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
class APMChoreographerImpl(
    private val mainThreadHandler: Handler,
    private val executor: Executor,
    private val logger: Logger
) : Choreographer.FrameCallback, APMChoreographer {

    @VisibleForTesting
    val callbacks: MutableSet<Choreographer.FrameCallback> =
        Collections.newSetFromMap(WeakHashMap())

    private var mainThreadChoreographer: Choreographer? = null

    override fun addCallback(callback: Choreographer.FrameCallback) {
        mainThreadHandler.post {
            callbacks.add(callback)
            startChoreographerIfRequired()
        }
    }

    private fun startChoreographerIfRequired() {
        if (mainThreadChoreographer == null) {
            mainThreadChoreographer = Choreographer.getInstance()
            mainThreadChoreographer?.postFrameCallback(this)
        }
    }

    override fun removeCallback(callback: Choreographer.FrameCallback) {
        mainThreadHandler.post {
            callbacks.apply { remove(callback) }
                .takeIf { it.isEmpty() }
                ?.let {
                    mainThreadChoreographer?.removeFrameCallback(this)
                    mainThreadChoreographer = null
                }
        }
    }

    override fun doFrame(frameTimeNanos: Long) = try {
        dispatchToFrameCallbacks(frameTimeNanos)
    } catch (throwable: Throwable) {
        logger.e(throwable.message)
    } finally {
        mainThreadChoreographer?.postFrameCallback(this)
    }

    private fun dispatchToFrameCallbacks(frameTimeNanos: Long) = callbacks.forEach {
        it.runCatching { doFrame(frameTimeNanos) }
            .onFailure(::reportAndLogError)
    }

    private fun reportAndLogError(throwable: Throwable) = executor.execute {
        logger.e(throwable.message)
        InstabugCore.reportError(throwable, "couldn't call callback.doFrame¬")
    }
}
