package com.instabug.apm.uitrace.repo

import com.instabug.apm.cache.model.UiTraceCacheModel
import com.instabug.apm.di.Provider
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.uitrace.UiTraceWrapper
import com.instabug.apm.uitrace.handler.UiTraceHandler
import com.instabug.apm.uitrace.handler.UiTraceWrapperHandler
import com.instabug.apm.uitrace.model.UiTraceEndParams
import com.instabug.apm.uitrace.model.UiTraceInitParams
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.util.collections.LimitedLinkedHashmap

interface UiTracesRepo {
    fun create(runtimeTraceId: String, traceId: Long): UiTraceWrapper?
    fun getOrCreateInitialUiTrace(runtimeTraceId: String, traceId: Long): UiTraceWrapper?
    fun start(runtimeTraceId: String, initModel: UiTraceInitParams)
    fun setSessionIdAndSaveIfPossible(sessionId: String)
    fun end(runtimeTraceId: String, params: UiTraceEndParams): UiTraceCacheModel?
    fun endAll(params: UiTraceEndParams)
    operator fun get(runtimeTraceId: String): UiTraceWrapper?
    fun clearAll()
    fun clearUiHangs()
}

class UiTracesRepoImpl(
    private val handlerProvider: Provider<UiTraceHandler?>,
    private val uiTraceWrapperFactoryProvider: Provider<ParameterizedFactory<UiTraceWrapper, Long>?>,
    private val logger: Logger,
    private val uiTraceWrapperHandler: UiTraceWrapperHandler
) : UiTracesRepo {

    private val uiTracesMap by lazy { LimitedLinkedHashmap<String, UiTraceWrapper>(5) }

    override fun create(runtimeTraceId: String, traceId: Long): UiTraceWrapper? =
        uiTraceWrapperFactoryProvider.takeIf { runtimeTraceId.isNotBlank() }
            ?.invoke()
            ?.create(traceId)
            ?.also { uiTracesMap[runtimeTraceId] = it }

    override fun start(runtimeTraceId: String, initModel: UiTraceInitParams) {
        getOrCreateInitialUiTrace(runtimeTraceId, initModel.traceId)?.apply {
            uiTraceWrapperHandler.start(this, initModel)
            handlerProvider()?.saveIfPossible(cacheModel)
        }
    }

    override fun getOrCreateInitialUiTrace(
        runtimeTraceId: String,
        traceId: Long
    ) = uiTracesMap[runtimeTraceId] ?: create(runtimeTraceId, traceId)

    override fun setSessionIdAndSaveIfPossible(sessionId: String) {
        runCatching {
            uiTracesMap.values.asSequence()
                .map { it.cacheModel }
                .filter { it.sessionId == null }.forEach {
                    it.sessionId = sessionId
                    handlerProvider()?.saveIfPossible(it)
                }
        }.onFailure(::reportAndLogError)
    }

    override fun end(runtimeTraceId: String, params: UiTraceEndParams): UiTraceCacheModel? =
        uiTracesMap[runtimeTraceId]
            .also { if (it == null) logger.logSDKProtected("uiTraceModel is null, can't update") }
            ?.runCatching {
                endSingleTrace(params)
                uiTracesMap.remove(runtimeTraceId)
                cacheModel
            }?.onFailure(::reportAndLogError)?.getOrNull()

    override fun endAll(params: UiTraceEndParams) {
        uiTracesMap.runCatching {
            onEach { (_, wrapper) -> wrapper.endSingleTrace(params) }
                .clear()
        }.onFailure (::reportAndLogError)
    }

    private fun UiTraceWrapper.endSingleTrace(params: UiTraceEndParams) {
        uiTraceWrapperHandler.end(this, params)
        handlerProvider()?.end(cacheModel)
    }


    override fun get(runtimeTraceId: String): UiTraceWrapper? = uiTracesMap[runtimeTraceId]

    override fun clearAll() {
        uiTracesMap.forEach { (_, wrapper) -> uiTraceWrapperHandler.clean(wrapper) }
        uiTracesMap.clear()
        handlerProvider()?.removeAll()
    }

    override fun clearUiHangs() =
        uiTracesMap.values.asSequence().map { it.uiHangsHandler }.forEach {
            it.stop()
            it.clearUiHangModel()
        }.also { handlerProvider()?.removeUiHangs() }

    private fun reportAndLogError(throwable: Throwable) {
        logger.logSDKError(throwable.message.orEmpty(), throwable)
        IBGDiagnostics.reportNonFatal(throwable, throwable.message)
    }
}
