package com.instabug.apm.sync;

import static com.instabug.apm.appflow.configuration.AppFlowConfigurationConstantsKt.APP_FLOW_REQUEST_LIMIT_DEFAULT;
import static com.instabug.apm.compose.compose_spans.configuration.ConfigurationConstantsKt.DEFAULT_COMPOSE_REQUEST_LIMIT;
import static com.instabug.apm.webview.webview_trace.configuration.ConfiguratonConstantsKt.DEFAULT_WEB_VIEW_REQUEST_LIMIT;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.apm.appflow.di.AppFlowServiceLocator;
import com.instabug.apm.appflow.configuration.AppFlowConfigurationProvider;
import com.instabug.apm.appflow.handler.AppFlowHandler;
import com.instabug.apm.appflow.model.AppFlowCacheModel;
import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.cache.handler.uitrace.UiTraceCacheHandler;
import com.instabug.apm.cache.model.AppLaunchCacheModel;
import com.instabug.apm.cache.model.ExecutionTraceCacheModel;
import com.instabug.apm.cache.model.FragmentSpansCacheModel;
import com.instabug.apm.cache.model.SessionCacheModel;
import com.instabug.apm.cache.model.UiTraceCacheModel;
import com.instabug.apm.compose.compose_spans.ComposeSpansServiceLocator;
import com.instabug.apm.compose.compose_spans.configuration.ComposeSpansConfigurationProvider;
import com.instabug.apm.compose.compose_spans.handler.ComposeSpansHandler;
import com.instabug.apm.compose.compose_spans.model.ComposeSpansCacheModel;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.configuration.APMStateProvider;
import com.instabug.apm.constants.AppLaunchType;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.handler.applaunch.AppLaunchesHandler;
import com.instabug.apm.handler.executiontraces.ExecutionTracesHandler;
import com.instabug.apm.handler.experiment.ExperimentHandler;
import com.instabug.apm.handler.fragment.FragmentSpansHandler;
import com.instabug.apm.handler.networklog.NetworkLogHandler;
import com.instabug.apm.handler.session.SessionHandler;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.APMNetworkLog;
import com.instabug.apm.networking.handler.SyncManagerNetworkHandler;
import com.instabug.apm.networking.mapping.sessions.SessionModelFiller;
import com.instabug.apm.webview.webview_trace.configuration.WebViewTraceConfigurationProvider;
import com.instabug.library.networkv2.RateLimitedException;
import com.instabug.library.networkv2.RequestResponse;
import com.instabug.library.networkv2.request.Request;

import java.util.ArrayList;
import java.util.List;

/**
 * Implementation of APM data syncing manager
 */
public class APMSyncManagerImpl implements APMSyncManager {

    private Logger apmLogger = ServiceLocator.getApmLogger();
    private APMConfigurationProvider apmConfigurationProvider = ServiceLocator.getApmConfigurationProvider();
    private SessionHandler sessionHandler = ServiceLocator.getSessionHandler();
    @Nullable
    private SessionMetaDataCacheHandler sessionMetaDataCacheHandler;
    @NonNull
    private AppLaunchesHandler appLaunchesHandler;
    @NonNull
    private NetworkLogHandler networkLogHandler;
    @NonNull
    private ExecutionTracesHandler executionTracesHandler;
    @NonNull
    private UiTraceCacheHandler uiTraceHandler;
    @NonNull
    SyncManagerNetworkHandler syncManagerNetworkHandler;
    @Nullable
    private ExperimentHandler experimentHandler;
    @Nullable
    private FragmentSpansHandler fragmentSpansHandler;

    /**
     * Boolean to avoid setting last sync interval without sending session.
     * So now the last sync will be only set after sending real session to the BE
     */
    @VisibleForTesting
    public boolean shouldSetLastSyncTime = false;

    public APMSyncManagerImpl() {
        sessionMetaDataCacheHandler = ServiceLocator.getSessionMetaDataCacheHandler();
        appLaunchesHandler = ServiceLocator.getAppLaunchesHandler();
        networkLogHandler = ServiceLocator.getNetworkLogHandler();
        executionTracesHandler = ServiceLocator.getExecutionTracesHandler();
        uiTraceHandler = ServiceLocator.getUiTraceCacheHandler();
        syncManagerNetworkHandler = ServiceLocator.getSyncManagerNetworkHandler();
        experimentHandler = ServiceLocator.getExperimentHandler();
        fragmentSpansHandler = ServiceLocator.getFragmentSpansHandler();
    }

    @Override
    public void start() {
        if (shouldSync()) {
            syncOldSessions();
        }
    }

    @Override
    public void start(boolean bypass) {
        if (bypass || shouldSync()) {
            syncOldSessions();
        }
    }

    /**
     * Checks if APM is enabled and sync interval is passed
     * Or if debug mode is enabled
     *
     * @return true if eligible for sync, false if not
     */
    @Override
    public boolean shouldSync() {
        if (!apmConfigurationProvider.shouldSendLegacyAPMSessions()) {
            //Don't sync session at all if the flag is false. Also, bypass the debug mode!
            return false;
        }

        return (apmConfigurationProvider.isAPMEnabled() && isSyncIntervalPassed()) || isDebugModeEnabled();
    }

    /**
     * Gets debug mode state
     *
     * @return true if enabled. false otherwise
     */
    private boolean isDebugModeEnabled() {
        return apmConfigurationProvider.isDebugModeEnabled() &&
                ServiceLocator.getDebugUtils().isInDebugMode();
    }

    /**
     * Checks for sync interval is passed
     *
     * @return true if sync interval is passed. false otherwise
     */
    private boolean isSyncIntervalPassed() {
        return System.currentTimeMillis() - apmConfigurationProvider.getLastSyncTime()
                >= apmConfigurationProvider.getSyncInterval() * 1000;
    }

    public void syncOldSessions() {
        List<SessionCacheModel> readyToBeSentSessionsList = sessionHandler.getReadyToBeSentSessions();
        if (!readyToBeSentSessionsList.isEmpty()) {
            for (SessionCacheModel model : readyToBeSentSessionsList) {
                fillSessionModel(model);
            }
            syncSessionsList(readyToBeSentSessionsList);
        } else {
            syncNextSessionsChunk();
        }
    }

    private void syncNextSessionsChunk() {
        int coldAppLaunchesCount = 0;
        int warmAppLaunchesCount = 0;
        int hotAppLaunchesCount = 0;
        int networkLogsOccurrenceCount = 0;
        int uiTracesOccurrenceCount = 0;
        int executionTracesOccurrenceCount = 0;
        int experimentsCount = 0;
        int fragmentSpansCount = 0;
        int composeSpansCount = 0;
        int webViewTraceCount = 0;
        int appFlowCount = 0;
        List<SessionCacheModel> sessionsList = new ArrayList<>();
        SessionCacheModel cacheModel;
        String nextSessionId = "-1";
        do {
            cacheModel = getNextSession(nextSessionId);
            if (cacheModel != null) {
                List<AppLaunchCacheModel> appLaunches = cacheModel.getAppLaunches();
                if (appLaunches != null) {
                    for (AppLaunchCacheModel appLaunch : appLaunches) {
                        if (AppLaunchType.WARM.equals(appLaunch.getType())) {
                            warmAppLaunchesCount++;
                        } else if (AppLaunchType.COLD.equals(appLaunch.getType())) {
                            coldAppLaunchesCount++;
                        } else {
                            hotAppLaunchesCount++;
                        }
                    }
                }

                List<APMNetworkLog> networkLogs = cacheModel.getNetworkLogs();
                networkLogsOccurrenceCount += networkLogs != null
                        ? networkLogs.size() : 0;

                List<UiTraceCacheModel> uiTraces = cacheModel.getUiTraces();
                uiTracesOccurrenceCount += uiTraces != null
                        ? uiTraces.size() : 0;

                webViewTraceCount += cacheModel.getWebViewTraceTotalCount();

                List<ExecutionTraceCacheModel> executionTraces = cacheModel.getExecutionTraces();
                executionTracesOccurrenceCount += executionTraces != null
                        ? executionTraces.size() : 0;

                List<String> experiments = cacheModel.getExperiments();
                experimentsCount += experiments != null
                        ? experiments.size() : 0;
                List<FragmentSpansCacheModel> fragmentSpans = cacheModel.getFragmentSpans();
                fragmentSpansCount += fragmentSpans != null
                        ? fragmentSpans.size() : 0;
                List<ComposeSpansCacheModel> composeSpans = cacheModel.getComposeSpans();
                composeSpansCount += composeSpans != null ? composeSpans.size() : 0;
                List<AppFlowCacheModel> appFlows = cacheModel.getAppFlows();
                appFlowCount += appFlows != null ? appFlows.size() : 0;
                if (exceededRequestLimit(
                        hotAppLaunchesCount,
                        coldAppLaunchesCount,
                        warmAppLaunchesCount,
                        networkLogsOccurrenceCount,
                        uiTracesOccurrenceCount,
                        executionTracesOccurrenceCount,
                        experimentsCount,
                        fragmentSpansCount,
                        composeSpansCount,
                        webViewTraceCount,
                        appFlowCount)) {
                    break;
                }
                sessionsList.add(cacheModel);
                nextSessionId = cacheModel.getId();
            }
        }
        while (cacheModel != null);
        syncSessionsList(sessionsList);
        apmLogger.logSDKDebug("syncNextSessionsChunk: " + sessionsList.size());
    }

    private boolean exceededRequestLimit(
            int hotAppLaunchesCount,
            int coldAppLaunchesCount,
            int warmAppLaunchesCount,
            int networkLogsOccurrenceCount,
            int uiTracesOccurrenceCount,
            int executionTracesOccurrenceCount,
            int experimentsCount,
            int fragmentSpansCount,
            int composeSpansCount,
            int webViewsTotalCount,
            int appFlowCount) {
        return (hotAppLaunchesCount > getLaunchLimitPerRequest(AppLaunchType.HOT) ||
                coldAppLaunchesCount > getLaunchLimitPerRequest(AppLaunchType.COLD) ||
                warmAppLaunchesCount > getLaunchLimitPerRequest(AppLaunchType.WARM) ||
                networkLogsOccurrenceCount > apmConfigurationProvider.getNetworkLogsRequestLimit() ||
                uiTracesOccurrenceCount > apmConfigurationProvider.getUiTraceLimitPerRequest() ||
                executionTracesOccurrenceCount > apmConfigurationProvider.getExecutionTraceLimitPerRequest() ||
                experimentsCount > apmConfigurationProvider.getExperimentsLimitPerRequest() ||
                fragmentSpansCount > apmConfigurationProvider.getFragmentSpansLimitPerRequest() ||
                composeSpansCount > getComposeSpansRequestLimit() ||
                webViewsTotalCount > getWebViewsRequestLimit() ||
                appFlowCount > getAppFlowRequestLimit()
        );
    }

    private int getComposeSpansRequestLimit() {
        ComposeSpansConfigurationProvider composeConfigurations = getComposeConfigurations();
        return composeConfigurations != null
                ? composeConfigurations.getRequestLimit() : DEFAULT_COMPOSE_REQUEST_LIMIT;
    }

    private int getWebViewsRequestLimit() {
        WebViewTraceConfigurationProvider configurations =
                ServiceLocator.getWebViewTraceConfigurationProvider();
        return configurations != null ? configurations.getRequestLimit() : DEFAULT_WEB_VIEW_REQUEST_LIMIT;
    }

    private int getAppFlowRequestLimit() {
        AppFlowConfigurationProvider configurationProvider = getAppFlowConfigurationProvider();
        return configurationProvider != null ?
                configurationProvider.getRequestLimit() : APP_FLOW_REQUEST_LIMIT_DEFAULT;
    }

    @Nullable
    private static AppFlowConfigurationProvider getAppFlowConfigurationProvider() {
        return AppFlowServiceLocator.INSTANCE.getConfigurationProvider();
    }

    private long getLaunchLimitPerRequest(@AppLaunchType String type) {
        return apmConfigurationProvider.getAppLaunchRequestLimit(type);
    }

    private void fillSessionModel(@Nullable SessionCacheModel cacheModel) {
        if (cacheModel != null) {
            String sessionID = cacheModel.getId();
            if (sessionMetaDataCacheHandler != null) {
                cacheModel.setSessionMetaData(sessionMetaDataCacheHandler.getSessionMetaData(sessionID));
            }
            cacheModel.setAppLaunches(appLaunchesHandler.getAppLaunchesForSession(sessionID));
            cacheModel.setNetworkLogs(networkLogHandler.getEndedNetworkLogsForSession(sessionID));
            cacheModel.setExecutionTraces(executionTracesHandler.getExecutionTracesForSession(sessionID));
            cacheModel.setUiTraces(uiTraceHandler.getUiTracesForSession(sessionID));
            ComposeSpansHandler composeSpansHandler = getComposeSpansHandler();
            if (composeSpansHandler != null)
                cacheModel.setComposeSpans(composeSpansHandler.getForSession(sessionID));
            if (experimentHandler != null) {
                cacheModel.setExperiments(experimentHandler.getSessionExperimentsAsync(sessionID));
            }
            if (fragmentSpansHandler != null) {
                cacheModel.setFragmentSpans(fragmentSpansHandler.getFragmentsForSession(sessionID));
            }
            fillSessionModelWithAppFlows(cacheModel, sessionID);
        }
    }

    private static void fillSessionModelWithAppFlows(@NonNull SessionCacheModel cacheModel, String sessionID) {
        SessionModelFiller appFlowSessionModelFiller = getAppFlowSessionModelFiller();
        if (appFlowSessionModelFiller != null) {
            appFlowSessionModelFiller.fill(sessionID, cacheModel);
        }
    }

    @Nullable
    private static SessionModelFiller getAppFlowSessionModelFiller() {
        return AppFlowServiceLocator.INSTANCE.getSessionModelFiller();
    }

    @Nullable
    private SessionCacheModel getNextSession(String currentSessionID) {
        SessionCacheModel cacheModel = sessionHandler.getNextSession(currentSessionID);
        if (cacheModel != null) {
            fillSessionModel(cacheModel);
        }
        return cacheModel;
    }

    @VisibleForTesting
    public Request.Callbacks<RequestResponse, Throwable> callback = new Request.Callbacks<RequestResponse, Throwable>() {
        @Override
        public void onSucceeded(RequestResponse response) {
            // Delete sessions marked as ready to be sent
            sessionHandler.deleteSessionsBySyncStatus(Constants.SessionSyncStatus.READY_TO_BE_SENT);
            apmConfigurationProvider.setLastApmSessionsRequestStartedAt(0);
            APMStateProvider stateProvider = ServiceLocator.getApmStateProvider();
            if (stateProvider != null) {
                stateProvider.resetStoreLimitDroppedSessionCount();
            }
            // Sync next chunk
            syncNextSessionsChunk();
        }

        @Override
        public void onFailed(Throwable error) {
            if (error instanceof RateLimitedException) {
                handleRateLimitException((RateLimitedException) error);
                return;
            }
            if (error != null && error.getMessage() != null) {
                apmLogger.e(error.getMessage());
            }
        }
    };

    private void handleRateLimitException(RateLimitedException exception) {
        apmConfigurationProvider.setApmSessionsLimitedUntil(exception.getPeriod());
        handleRateIsLimited();
    }

    /**
     * if rate is not limited
     * Marks sessions list as ready to be sent
     * Send list of messages to BE
     * When receives successful response, delete messages marked as ready to be sent
     * Continue with the next chunk
     * else the the list will be deleted
     *
     * @param sessions list of session to be sent to BE
     */
    private void syncSessionsList(@NonNull List<SessionCacheModel> sessions) {
        if (!sessions.isEmpty()) {
            shouldSetLastSyncTime = true;
            List<String> sessionsIDs = new ArrayList<>();
            for (SessionCacheModel session : sessions) {
                sessionsIDs.add(session.getId());
            }
            // Mark sessions as ready to be sent
            sessionHandler.changeSessionSyncStatus(sessionsIDs, Constants.SessionSyncStatus.READY_TO_BE_SENT);
            if (apmConfigurationProvider.shouldSendLegacyAPMSessions()) {
                apmLogger.logSDKDebug("SDK will send APM sessions on legacy APM sessions URL");
            }
            if (apmConfigurationProvider.isApmSessionsRateLimited()) {
                handleRateIsLimited();
            } else {
                apmConfigurationProvider.setLastApmSessionsRequestStartedAt(System.currentTimeMillis());
                syncManagerNetworkHandler.syncSessions(sessions, callback);
            }
        } else {
            if (shouldSetLastSyncTime)
                // No more sessions to be synced. Reset last sync time
                apmConfigurationProvider.setLastSyncTime(System.currentTimeMillis());
        }
    }

    private void handleRateIsLimited() {
        logRateIsLimited();
        sessionHandler.deleteSessionsBySyncStatus(Constants.SessionSyncStatus.READY_TO_BE_SENT);
        syncNextSessionsChunk();
    }

    private void logRateIsLimited() {
        apmLogger.d(String.format(RateLimitedException.RATE_LIMIT_REACHED, Constants.FEATURE_NAME));
    }

    @Nullable
    public static ComposeSpansHandler getComposeSpansHandler() {
        return ComposeSpansServiceLocator.INSTANCE.getHandler();
    }

    @Nullable
    public static AppFlowHandler getAppFlowHandler() {
        return AppFlowServiceLocator.INSTANCE.getHandler();
    }

    @Nullable
    public static ComposeSpansConfigurationProvider getComposeConfigurations() {
        return ComposeSpansServiceLocator.INSTANCE.getConfigurationProvider();
    }
}
