package com.instabug.apm.screenloading.manager

import android.app.Activity
import android.content.Context
import android.os.Build
import androidx.annotation.RequiresApi
import com.instabug.apm.cache.handler.uitrace.UiLoadingMetricCacheHandler
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.di.Provider
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.sanitization.Validator
import com.instabug.apm.screenloading.handler.CPScreenLoadingHandlerImpl
import com.instabug.apm.screenloading.handler.NativeScreenLoadingHandlerImpl
import com.instabug.apm.screenloading.repo.NativeScreenLoadingRepo
import com.instabug.apm.uitrace.activitycallbacks.CompositeApmUiTraceActivityCallbacks
import com.instabug.apm.util.runOrReportAPMError
import com.instabug.library.Platform
import com.instabug.library.factory.Factory
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.settings.SettingsManager
import java.util.concurrent.Executor

interface ScreenLoadingManager {
    fun start()
    fun startSynchronous()
    fun onStateChanged()
    fun endAll()
    fun <T : Activity> endScreenLoading(
        activityClass: Class<T>?,
        timeMetric: EventTimeMetricCapture?
    )

    fun endScreenLoadingCP(timeStampMicro: Long?, uiTraceId: Long?)
    fun reportScreenLoadingCP(startTimeStampMicro: Long?, durationMicro: Long?, uiTraceId: Long?)
    fun cacheCurrentScreenLoadingCP()
}

@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
class ScreenLoadingManagerImpl(
    private val configurationProvider: APMConfigurationProvider,
    private val executor: Executor,
    private val settingsManager: SettingsManager,
    private val nativeScreenLoadingHandlerFactoryProvider: Provider<ParameterizedFactory<NativeScreenLoadingHandlerImpl, NativeScreenLoadingRepo>>,
    private val cpScreenLoadingHandlerProvider: Provider<CPScreenLoadingHandlerImpl>,
    private val compositeApmUiTraceActivityCallbacks: CompositeApmUiTraceActivityCallbacks,
    private val nativeScreenLoadingRepoFactory: Factory<NativeScreenLoadingRepo>,
    private val cacheHandler: UiLoadingMetricCacheHandler,
    private val logger: Logger,
    private val contextProvider: Provider<Context?>,
    private val endScreenLoadingValidatorProvider: Provider<Validator<Unit>>
) : ScreenLoadingManager {

    private val isNativeApp
        get() = settingsManager.getEarlyCurrentPlatform(contextProvider()) == Platform.ANDROID
    private val isValidToUseEndAPI
        get() = endScreenLoadingValidatorProvider().isValid(Unit)

    private var nativeScreenLoadingHandler: NativeScreenLoadingHandlerImpl? = null
    private var nativeScreenLoadingRepo: NativeScreenLoadingRepo? = null
    private var cpScreenLoadingHandler: CPScreenLoadingHandlerImpl? = null

    override fun start() = executor.execute {
        if (configurationProvider.isAutoUiLoadingMetricsFullyEnabled) startSynchronous()
    }

    override fun startSynchronous() {
        runOrReportAPMError("Error while starting ScreenLoading feature", logger) {
            if (isNativeApp) startNativeScreenLoadingCapturing()
            else startCpScreenLoadingCapturing()
        }
    }

    private fun startNativeScreenLoadingCapturing() {
        if (nativeScreenLoadingRepo == null) {
            nativeScreenLoadingRepo = nativeScreenLoadingRepoFactory.create()
        }
        nativeScreenLoadingRepo?.let(::createAndRegisterNativeScreenLoadingHandlerIfPossible)
    }

    private fun createAndRegisterNativeScreenLoadingHandlerIfPossible(repo: NativeScreenLoadingRepo) {
        if (nativeScreenLoadingHandler == null) {
            nativeScreenLoadingHandler = nativeScreenLoadingHandlerFactoryProvider().create(repo)
                .also(compositeApmUiTraceActivityCallbacks::add)
        }
    }

    private fun startCpScreenLoadingCapturing() {
        if (cpScreenLoadingHandler == null) {
            cpScreenLoadingHandler = cpScreenLoadingHandlerProvider()
                .also(compositeApmUiTraceActivityCallbacks::add)
        }
    }

    override fun onStateChanged() = executor.execute {
        if (configurationProvider.isAutoUiLoadingMetricsFullyEnabled) startSynchronous()
        else stopSynchronous()
    }

    private fun stopSynchronous() =
        runOrReportAPMError("Error while stopping screenLoading feature", logger) {
            stopNativeScreenLoadingCapturing()
            stopCpScreenLoading()
            cacheHandler.removeAll()
        }

    private fun stopNativeScreenLoadingCapturing() {
        nativeScreenLoadingHandler?.let(compositeApmUiTraceActivityCallbacks::remove)
        nativeScreenLoadingRepo?.clearAll()
        nativeScreenLoadingHandler = null
        nativeScreenLoadingRepo = null
    }

    private fun stopCpScreenLoading() {
        cpScreenLoadingHandler?.let(compositeApmUiTraceActivityCallbacks::remove)
        cpScreenLoadingHandler = null
    }

    override fun endAll() = executor.execute {
        runOrReportAPMError("An error occurred while caching screenLoading traces", logger) {
            nativeScreenLoadingRepo?.endAll()
            cpScreenLoadingHandler?.cacheSynchronous()
        }
    }

    override fun <T : Activity> endScreenLoading(
        activityClass: Class<T>?,
        timeMetric: EventTimeMetricCapture?
    ) = executor.execute {
        if (isValidToUseEndAPI && activityClass != null && timeMetric != null) {
            nativeScreenLoadingHandler?.endScreenLoading(activityClass, timeMetric)
        }
    }

    override fun endScreenLoadingCP(timeStampMicro: Long?, uiTraceId: Long?) = executor.execute {
        if (isValidToUseEndAPI && timeStampMicro != null && uiTraceId != null) {
            cpScreenLoadingHandler?.endScreenLoading(timeStampMicro, uiTraceId)
        }
    }

    override fun reportScreenLoadingCP(
        startTimeStampMicro: Long?,
        durationMicro: Long?,
        uiTraceId: Long?
    ) {
        if (startTimeStampMicro == null || durationMicro == null || uiTraceId == null) return
        cpScreenLoadingHandler?.reportScreenLoading(startTimeStampMicro, durationMicro, uiTraceId)
    }

    override fun cacheCurrentScreenLoadingCP() {
        cpScreenLoadingHandler?.cacheCurrentTrace()
    }
}