package com.instabug.apm.logger.internal;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.constants.Constants;
import com.instabug.apm.model.LogLevel;
import com.instabug.library.util.InstabugSDKLogger;

import static com.instabug.apm.model.LogLevel.DEBUG;
import static com.instabug.apm.model.LogLevel.ERROR;
import static com.instabug.apm.model.LogLevel.INFO;
import static com.instabug.apm.model.LogLevel.NONE;
import static com.instabug.apm.model.LogLevel.VERBOSE;
import static com.instabug.apm.model.LogLevel.WARNING;

/**
 * Wrapper class to handle all APM related logs with unified tag.
 */
public class Logger {

    @NonNull
    private final APMConfigurationProvider configurationProvider;

    public Logger(@NonNull APMConfigurationProvider configurationProvider) {
        this.configurationProvider = configurationProvider;
    }

    /**
     * Wrap SDK verbose logs with the unified TAG
     *
     * @param message the message to be logged
     */
    public void logSDKVerbose(@NonNull String message) {
        v(message);
        logSDKProtected(message);
    }

    /**
     * Wrap SDK info logs with the unified TAG
     *
     * @param message the message to be logged
     */
    public void logSDKInfo(@NonNull String message) {
        i(message);
        logSDKProtected(message);
    }

    /**
     * Wrap SDK debug logs with the unified TAG
     *
     * @param message the message to be logged
     */
    public void logSDKDebug(@NonNull String message) {
        d(message);
        logSDKProtected(message);
    }

    /**
     * Wrap SDK warning logs with the unified TAG
     *
     * @param message the message to be logged
     */
    public void logSDKWarning(@NonNull String message) {
        w(message);
        logSDKProtected(message);
    }

    /**
     * Wrap SDK error logs with the unified TAG
     *
     * @param message   the message to be logged
     * @param throwable the cause of the exception
     */
    public void logSDKError(@NonNull String message, @NonNull Throwable throwable) {
        e(message + ". " + throwable.toString());
        logSDKProtected(message + ". " + throwable.toString());
    }

    /**
     * Wrap SDK error logs with the unified TAG, including stack trace
     *
     * @param message   the message to be logged
     * @param throwable the cause of the exception
     */
    public void logSDKErrorWithStackTrace(@NonNull String message, @NonNull Throwable throwable) {
        logSDKErrorProtected(message, throwable);
    }

    /**
     * Wrap SDK error logs with the unified TAG
     *
     * @param message the message to be logged
     */
    public void logSDKError(@NonNull String message) {
        e(message);
        logSDKProtected(message);
    }

    /**
     * Wrap SDK protected logs with the unified TAG
     *
     * @param message the message to be logged
     */
    public void logSDKProtected(@NonNull String message) {
        InstabugSDKLogger.p(Constants.LOG_TAG, message);
    }

    /**
     * Wrap SDK error logs with the unified TAG
     *
     * @param message the message to be logged
     * @param throwable the cause of the exception
     */
    public void logSDKErrorProtected(@NonNull String message, Throwable throwable) {
        InstabugSDKLogger.e(Constants.LOG_TAG, message, throwable);
    }

    /**
     * Print verbose messages independently of the SDK logger
     * Used to handle the logs that should be printed before initializing the SDK.
     *
     * @param message the message to be logged by the native android {@link Log}
     */
    public void v(String message) {
        if (shouldPrintLog(VERBOSE)) Log.v(Constants.LOG_TAG, message);
    }

    /**
     * Print info messages independently of the SDK logger
     * Used to handle the logs that should be printed before initializing the SDK.
     *
     * @param message the message to be logged by the native android {@link Log}
     */
    public void i(String message) {
        if (shouldPrintLog(INFO)) Log.i(Constants.LOG_TAG, message);
    }

    /**
     * Print debug messages independently of the SDK logger
     * Used to handle the logs that should be printed before initializing the SDK.
     *
     * @param message the message to be logged by the native android {@link Log}
     */
    public void d(String message) {
        if (shouldPrintLog(DEBUG)) Log.d(Constants.LOG_TAG, message);
    }

    /**
     * Print warning messages independently of the SDK logger
     * Used to handle the logs that should be printed before initializing the SDK.
     *
     * @param message the message to be logged by the native android {@link Log}
     */
    public void w(String message) {
        if (shouldPrintLog(WARNING)) Log.w(Constants.LOG_TAG, message);
    }

    /**
     * Print error messages independently of the SDK logger
     * Used to handle the logs that should be printed before initializing the SDK.
     *
     * @param message the message to be logged by the native android {@link Log}
     */
    public void e(String message) {
        if (shouldPrintLog(ERROR)) Log.e(Constants.LOG_TAG, message);
    }

    /**
     * Print error messages independently of the SDK logger
     * Used to handle the logs that should be printed before initializing the SDK.
     *
     * @param message the message to be logged by the native android {@link Log}
     * @param throwable the throwable to be printed by the native android
     */
    public void e(String message, Throwable throwable) {
        if (shouldPrintLog(ERROR)) Log.e(Constants.LOG_TAG, message, throwable);
    }

    @VisibleForTesting
    boolean shouldPrintLog(@LogLevel int level) {
        int allowedLogLevel = configurationProvider.getLogLevel();
        return allowedLogLevel != NONE && level <= allowedLogLevel;
    }
}
